/* FILE reg */
     /* Maintain register list (list,copy,remove)
	Do not consider individual channels ("CurrentChannel")	*/

     /* FUNCTIONS:
      *   ProofReg:		Test existence of register
      *   ListReg:		Print number & descibe arrays in list
      *   CopyReg:		Copy array in list to another position
      *   AvgReg:		Average two registers together
      *   NormalizeReg:		Normalize register to single trial
      *   SubtractReg:		Subtract 2 registers (Unit chan is special?)
      *   ShiftReg:		Shift reg n ms later (n<0: earlier)
      *	  TagReg:		Change tag of array in list
      *   MacroTagReg:		Ditto, don't prompt (for macros)
      *   RemoveReg:		Remove register from list
      *	  ContentReg:		Return trial_count
      *   StackNameReg:		Return stackname of register
      *   StackNumberReg:	Return stacknumber of register
      *   ClassNumberReg:	Return stackclass of register
      */

#include "array.h"				/* Array declarations	*/
#include <string.h>

/************************************************************************/

/* FUNCTION ProofReg */
	 /* Avoid gory segementation errors: test if register exists	*/
int ProofReg(int reg) {
	if (reg<MAX_REGISTERS && reg>=0)
	   return(OK);
	if (reg != INVALID_REG)
	   vtprint("Invalid number");
	return(FAIL);
}
/************************************************************************/
#define UsingRegister(i) (Register[i].tag[0]!=0 || Register[i].trial_count)

/* FUNCTION ListReg */
	 /* Print register numbers, contents & tag 	*/
void ListReg(void) {
    int Entries = 0;
    int i = -1;
    static char Bolden[] = "[1m";
    static char UnBolden[] = "[0m";
    int Columns = 3;					/* 2 or 3 col?	*/
    int TagSize = 16;

while (++i < MAX_REGISTERS) {				/* Count entries */
   if (UsingRegister(i))
      Entries++;
   if (strlen(Register[i].tag) > 16) {
      Columns = 2;
      TagSize = 29;				/* Really two short!	*/
      }
   }
i = -1;

fprintf(stderr,"     %d registers available; %d used",MAX_REGISTERS, Entries);

if (Entries < 7) {
  while (++i < MAX_REGISTERS)
    if (UsingRegister(i))
      fprintf(stderr,"\n%2d  (%2d)%s%-30s",
	    i,Register[i].trial_count,
	    (Register[i].trial_count <10) ? "  ":
	      ((Register[i].trial_count<100) ? " ":""),
	    Register[i].tag);
} else {
  Entries = 0;
  while (++i < MAX_REGISTERS)
     if (UsingRegister(i)) {
	 Entries++;
         fprintf(stderr," %s%s%-3d%s(%3d) %-*s",
     	    (Entries%Columns == 1) ? "\n " : "",
	    Bolden, i, UnBolden,
	    Register[i].trial_count, TagSize, Register[i].tag);
	 }
     }

putc('\n', stderr);
return;
}
/* END ListReg */
/************************************************************************/

/* FUNCTION RemoveReg */
	 /* Remove/zero reg from the list			*/ 
int RemoveReg(int reg) {
int *Spikes;
int i = MAX_CHANNELS;

if (ProofReg(reg) == FAIL)
   return(FAIL);
Register[reg].stacknumber = Register[reg].classnumber = 
Register[reg].frame_count = Register[reg].trial_count = 
Register[reg].shift = 0;
strcpy(Register[reg].tag, "");

while (i-- > 0) {			/* Laborious but safe!	*/
    FRAME *From = &(Register[reg].channel[i].frame[0]);
    int Count = MAX_FRAMES;

    for ( ; Count-- > 0; From++) {	/* Insurance		*/
        From->sum  = From->sum2 = 0;
        From->n    = 0;
        }
    }

for (Spikes=Register[reg].spikes, i=0; i<MAX_FRAMES; i++)
    *(Spikes++) = 0;

return(OK);
}
/* END RemoveReg */
/************************************************************************/

/* FUNCTION TagReg */
	 /* User is changing description of an existing (active) trial	*/
int TagReg(int changeReg) {
char newname[255];

if (ProofReg(changeReg) == FAIL)
   return(FAIL);
inputs("New tag:   ", newname);
if (newname[0] != 0 && (int)strlen(newname)<TAG_SIZE)
   strcpy(Register[changeReg].tag, newname);	/* Got nongarbage reply?*/
return(OK);
}
/* END TagReg */
/************************************************************************/

/* FUNCTION MacroTagReg */
	 /* Macro call to change register tag/title */
int MacroTagReg(int changeReg, char *newname) {

if (ProofReg(changeReg) == FAIL)
   return(FAIL);
if (*newname != 0) {
   if ((int)strlen(newname) >= TAG_SIZE)
      strncpy(Register[changeReg].tag, newname, TAG_SIZE);
   else
      strcpy(Register[changeReg].tag, newname);
   }
return(OK);
}
/* END TagReg */
/************************************************************************/

/* FUNCTION CopyReg */
	 /* Copy one register into another */
int CopyReg(int from, int to) {

if (ProofReg(to) == FAIL)
   return(FAIL);
if (ProofReg(from) == FAIL)
   return(FAIL);

Register[to] = Register[from];

if (!(Status & (COMMAND_MODE|MACRO_MODE|BATCH_MODE)))
   TagReg(to);					/* Request new name	*/

return(OK);
}
/* END CopyReg */
/* ******************************************************************** */

/* FUNCTION AvgReg */
	 /* Avg registers together	*/
int AvgReg(int from, int to) {				/* to += from	*/
int j = MAX_CHANNELS;

if (ProofReg(from) == FAIL)
   return(FAIL);
if (ProofReg(to) == FAIL)
   return(FAIL);
if (Register[from].trial_count == 0)
   return(OK);

if (Register[to].trial_count == 0) {			/* New one?	*/
   CopyReg(from, to);
   return(OK); 
   }

Register[to].trial_count += Register[from].trial_count;
Register[to].frame_count =
	MAX(Register[to].frame_count,Register[from].frame_count);

while (j-- > 0) {
    int Count = Register[from].frame_count;
    FRAME *From = &(Register[from].channel[j].frame[0]);
    FRAME *To   = &(Register[to  ].channel[j].frame[0]);

    for ( ; Count-- > 0; To++, From++)
       if (From->n) {
	if (To->n) {
           To->sum  +=  From->sum;			/* Do addition	*/
           To->sum2 +=  From->sum2;
           To->n    +=  From->n;
	 } else {			/* If there's nothing in 'To',	*/
           To->sum  =  From->sum;	/*  don't +=; may not be init'd */
           To->sum2 =  From->sum2;
           To->n    =  From->n;
	   }
        }
    }

if (RecordedChannel(UNIT)) {
   int Count 	     = Register[from].frame_count;
   int *spikes_to   = Register[to].spikes;
   int *spikes_from = Register[from].spikes;

   while (Count--)
      *(spikes_to++) += *(spikes_from++);
   }
return(OK); 
}
/* End AvgReg */
/* ******************************************************************** */

/* FUNCTION NormalizeReg */
	 /* Normalize all data to one trial	*/
void NormalizeReg(int reg) {
   int j = MAX_CHANNELS;

   if (ContentReg(reg) < 2)			/* Calls ProofReg	*/
     return;

   while (j-- > 0) {
       int Count = Register[reg].frame_count;
       FRAME *F = &(Register[reg].channel[j].frame[0]);

       F--;
       while (Count-- >= 0)
          if ((++F)->n) {
	    F->sum /= F->n;
	    F->sum2 = F->sum * F->sum;
            F->n = 1;
            }
       }
   Register[reg].trial_count = 1;		/* DIAG: Get spikes too!*/
}
/* ******************************************************************** */

#define VARiance(p) (p->sum2 - p->sum * p->sum / p->n) / (p->n -1)

/* FUNCTION SubtractReg */
	 /* Subtract registers.  Do nothing with spikes	*/
int SubtractReg(int to, int minus) {			/* to -= minus	*/
int    Count = MAX_FRAMES * MAX_CHANNELS;
FRAME *To, *Minus;
int OneTrial = 0;

if (ProofReg(minus) == FAIL)
   return(FAIL);
if (ProofReg(to) == FAIL)
   return(FAIL);

To    = &(Register[to   ].channel[0].frame[0]);
Minus = &(Register[minus].channel[0].frame[0]);

if (Register[minus].trial_count==0 || Register[to].trial_count==0)
   return(OK);
if (Register[minus].trial_count==1 || Register[to].trial_count==1)
   OneTrial = 1;

Register[to].frame_count =
   	MIN(Register[minus].frame_count,Register[to].frame_count);
Register[to].trial_count =
   	MIN(Register[minus].trial_count,Register[to].trial_count);


int LargerVariance = 1;

if (OneTrial == 0) {
   float a =0., b = 0.;
   while (Count--)
     if (Minus->n && To->n) {			/* Data to subtract? 	*/
	a = a + VARiance(To);
	b = b + VARiance(Minus);
	To++;
	Minus++;
        }
   LargerVariance = (a > b);
   Count = MAX_FRAMES * MAX_CHANNELS;
   To    = &(Register[to   ].channel[0].frame[0]);
   Minus = &(Register[minus].channel[0].frame[0]);
   }


for ( ; Count-- > 0; To++, Minus++)
   if (Minus->n && To->n) {			/* Data to subtract? 	*/
      if (OneTrial) {
	 To->sum  -= Minus->sum;		/* x1000's faster!	*/
	 To->sum2  = To->sum * To->sum;
      } else {					/* Slow and ponderous	*/
#ifdef OLD_VERSION
         /* FRAME *LargerVAR = (VARiance(To) > VARiance(Minus)) ? To : Minus;
	  */
         FRAME *LargerVAR = (LargerVariance) ? To : Minus;	/* ALTERNATE: Overall larger? */
         To->sum2 =			/* Incomplete calc of sum2 ...	*/
	    LargerVAR->sum2 - LargerVAR->sum*LargerVAR->sum/LargerVAR->n;
         To->sum  =			/* Subtraction MEANS, not sums	*/
             To->sum/To->n - Minus->sum/Minus->n;
         To->n = MIN(To->n,Minus->n);		/* Use the smaller N	*/
	
         To->sum  *= To->n;			/* Complete 'n' calc	*/
         To->sum2 += To->sum * To->sum / To->n;	/* Complete sum2 calc	*/

#else

             FRAME *LargerVAR = (LargerVariance) ? To : Minus;	/* ALTERNATE: Overall larger */
	     float SaveVar = VARiance(LargerVAR);

             To->sum  =			/* Subtraction MEANS, not sums	*/
               (To->sum/To->n - Minus->sum/Minus->n) * LargerVAR->n;
             To->n = LargerVAR->n;		/* Use the N of the larger variance	*/
	     To->sum2 = (To->n - 1) * SaveVar + To->sum * To->sum / To->n;
#endif



         }
   } else						/* Missing data?*/
      To->n = To->sum = To->sum2 = 0;			/* Undeff'd	*/
return(OK); 
}
/* End SubtractReg */
/* ******************************************************************** */

/* FUNCTION ShiftReg	*/
	 /* Shift data in reg __ forwards by the specified amount of ms	*/
int ShiftReg(int reg, int ms) {
	FRAME *From, *To;
	int   *spikes_to;
	int   *spikes_from;
	int    Count = FAIL;
	int i;

	if (ProofReg(reg) == FAIL)
	   return(FAIL);
	if (Register[reg].trial_count==0 || ms==0)
   	   return(OK);
	To = From = Register[reg].channel[0].frame;
	spikes_to = spikes_from = Register[reg].spikes;

	if (ms < 0) {				/* Shift BACKWARDS?	*/
	   if (Count - ms > MAX_FRAMES) {
	      vtprint("Tried to shift beyond bounds of data array!");
	      return(FAIL);         
	      }
	   for (i=0; i<MAX_CHANNELS; i++) {
	      To   = Register[reg].channel[i].frame;
	      From = To - ms;		/* Remember, ms is negative!	*/
	      Count = Register[reg].frame_count + ms;	/* ms is < 0!	*/

	      while (Count--)
	         *(To++) = *(From++);		/* Transfer whole struct*/
	      Count = -ms;
	      while (Count--)
	         (++To)->n = 0;			/* Zero out remainder	*/
	      }
	   Count = Register[reg].frame_count + ms;	/* Now do spikes*/
	   spikes_from -= ms;
	   while (Count--)
	      *(++spikes_to) = *(++spikes_from);
	   Count = -ms;
	   while (Count--)
	      *(++spikes_to) = 0;
	} else {				/* Shift FORWARDS	*/
	   for (i=0; i<MAX_CHANNELS; i++) {
	      Count = Register[reg].frame_count;
	      From  = Register[reg].channel[i].frame + Count; /* EOTrial*/
	      To  = From + ms;			/* New end of trial	*/

	      if (Count + ms > MAX_FRAMES) {
	         vtprint("Tried to shift beyond bounds of data array!");
		 return(FAIL);         
		 }

	      while (Count--)
	         *(To--) = *(From--);		/* Transfer whole struct*/

	      Count = ms;			/* Zero out what's left	*/
	      while (Count--)
	         (To--)->n = 0;
	      }

	   Count = Register[reg].frame_count;		/* Now do spikes*/
	   spikes_from += Register[reg].frame_count;	/* End of trial	*/
	   spikes_to   += Register[reg].frame_count + ms; /* Shifted end*/

	   while (Count--)		/* Move all data forward by ms	*/
	      *(spikes_to--) = *(spikes_from--);
	   Count = ms;
	   while (Count--)
	      *(--spikes_to) = 0;		/* Clear initial data	*/
	   }

	Register[reg].frame_count += ms;	/* Now longer or shorter*/
	Register[reg].shift += ms;		/* Keep track		*/
	return(OK);
	}
/* ******************************************************************** */

/* FUNCTION ContentReg */
int ContentReg(int reg) {
if (ProofReg(reg) == FAIL)
   return(FAIL);
return(Register[reg].trial_count);
}
/************************************************************************/

/* FUNCTION StackNameReg */
char *StackNameReg(int reg) {
if (ProofReg(reg) == FAIL)
   return(NULL);
return(Register[reg].stackname);
}
/************************************************************************/

/* FUNCTION StackNumberReg */
int StackNumberReg(int reg) {
if (ProofReg(reg) == FAIL)
   return(0);
return(Register[reg].stacknumber);
}
/************************************************************************/

/* FUNCTION ClassNumberReg */
int ClassNumberReg(int reg) {
if (ProofReg(reg) == FAIL)
   return(0);
return(Register[reg].classnumber);
}
/************************************************************************/

/* FUNCTION CountReg */
int CountReg(int reg) {
	if (ProofReg(reg) == FAIL)
	   return(-1);
	return(Register[reg].trial_count);
	}
/************************************************************************/
