/* FILE gelse.c */
     /* FUNCTIONS:
      * ScreenWidth, ScreenHeight
      * PixToTime:
      * TimeToPix:
      * GraphSetUp:		Initialize variables
      * GraphShutDown:		closepl();
      * GraphLine:		Draw or erase bracket or line
      * PrintGraphicsScaling:	Print scale values on upper left of plot
      * GraphEvents:		Indicate chair steps, ticks & zero line
      * GraphZeroLine:		Indicate zero line
      * Set_LineType:		Force a particular line type
      * GraphErase:		Call erase() (for files w/o graph.h)
      * GraphText:		Call move() and label() (for files w/o graph.h)
      * GraphFontSize:		Call font() for files w/o graph.h
      * GraphFontName:		Call font() for files w/o graph.h
      * GraphPrintf		Formatted print on graph
      * GraphFlush
      */
/* ******************************************************************** */
#include <stdarg.h>				/* For va(variable args)*/
#include "_imports/event.h"
#include "graph.h"

/* ******************************************************************** */
/* ******************************************************************** */
int	x_offset;				/* In milliseconds	*/
float	y_offset;				/* In pixels		*/
float	x_mult, y_mult;				/* ms,deg or deg/s->pix	*/

/* FUNCTION Get_x_offset */ /* FUNCTION Set_x_offset */
/* FUNCTION Get_y_offset */ /* FUNCTION Set_y_offset */
/* FUNCTION Get_x_mult */ /* FUNCTION Set_x_mult */
/* FUNCTION Get_y_mult */ /* FUNCTION Set_y_mult */
int  Get_x_offset(void) { return(x_offset); }
void Set_x_offset(int x) { x_offset = x; }
float Get_y_offset(void) { return(y_offset); }
void  Set_y_offset(float y) { y_offset = y; }
float Get_x_mult(void) { return(x_mult); }
void  Set_x_mult(float x) { x_mult = x; }
float Get_y_mult(void) { return(y_mult); }
void  Set_y_mult(float y) { y_mult = y; }
/* ******************************************************************** */

/* FUNCTION Screen_Height */
/* FUNCTION Screen_Width */
	 /* Avoid having to include everywhere */
int ScreenHeight(void) { return(SCREEN_HEIGHT); }
int ScreenWidth(void)  { return(SCREEN_WIDTH); }
/* ******************************************************************** */

/* FUNCTION PixToTime */
/* FUNCTION PixToVel */
	 /* Pixels (from mouse report) --> msec	or deg	*/
int PixToTime(int x) { return(irint(x / x_mult - x_offset)); }
float PixToVel(int y) { return((y - y_offset) / y_mult); }
/* ******************************************************************** */

/* FUNCTION TimeToPix */
/* FUNCTION VelToPix */
/* FUNCTION IndexToPix */
/* FUNCTION PixToIndex */
	 /* ms to screen pixels */
int TimeToPix(int ms) { return(irint(((ms) + x_offset) * x_mult)); }
float VelToPix(float y) { return((y) * y_mult + y_offset); }
int IndexToPix(int ms) { return(TimeToPix(ms)); }
int PixToIndex(int i) { return(PixToTime(i)); }
/* ******************************************************************** */

/* FUNCTION GraphSetUp */
	 /* Init variables of graph.h	*/
void GraphSetUp(void) {
   openpl();
   space(-BORDER,				/* BORDER on all sides	*/
	 -BORDER,
	  BORDER+SCREEN_WIDTH+RIGHT_MARGIN,	/* Width + axis space	*/
	  BORDER+SCREEN_HEIGHT+TOP_MARGIN);	/* Height + title space	*/
   /* ChangeParameter('t', "h");		 * Fails: no dflag yet	*/
   ChangeParameter('s', "0");
   ChangeParameter('w', "3000");
   ChangeParameter('h', "100");
   ChangeParameter('o', "0");
   ChangeParameter('l', "1");
   ChangeParameter('m', "1");
   ChangeParameter('i', "1");
   ChangeParameter('b', "0");
   ChangeParameter('e', "0");
   ChangeParameter('d', "0");
   ChangeParameter('c', "R");
   ChangeParameter('z', "0");			/* Time tics		*/
   ChangeParameter('Z', "0");			/* Time tics offset	*/
   }
/* ******************************************************************** */

/* FUNCTION GraphShutDown */
void GraphShutDown(void) { closepl(); }
/* ******************************************************************** */

/* FUNCTION Set_LineType */
	 /* Put out characters to set linetype; return OLD linetype	*/
	 /*   (return old type so can restore easily to that type)	*/
int Set_LineType(int type) {
   static int WasType = 0;
   int temp;

   if (type == NO_CHANGE)
      return(WasType);

   if (WasType<=0 && type>0)
      color(0,0,0);				/* Write black	*/
   else if (type <= 0)
      color(0xFF,0xFF,0xFF);				/* Erase mode	*/

   switch (type) {
      case 1: linemod("solid");		break;
      case 2: linemod("longdashed");	break;
      case 3: linemod("shortdashed");	break;
      case 4: linemod("dotdashed");	break;
      case 5: linemod("dotted");	break;
      case 6: linemod("disconnected");	break;
      default: linemod("solid");	break;	/* All the negatives	*/
      }

   temp = WasType;			/* Save old type for return	*/
   WasType = type;			/* Save current type for next	*/
   return(temp);			/* Return type it no longer is!	*/
   }
/* ******************************************************************** */

/* FUNCTION GraphLine */
	 /* Draw line of fixed size with lips	*/
	 /* ODD: if use 'line', thinks it's a ptr! */
int GraphLine(int when, int lip) {	/* Lip:at (ms),size & direction (pix) */
    int Line;				/* Location of line (pix) 	*/

    Line = TimeToPix(when);			/* Convert ms -> pixels	*/

    if (Line > SCREEN_WIDTH || Line < 1)	/* Off the screen?	*/
       return(FAIL);
    if (Line > SCREEN_WIDTH || Line < 1)	/* Off the screen?	*/
     while (Line + lip < 2)			/* Stop lip wraparound	*/
       lip++;
    while (Line + lip > SCREEN_WIDTH-2)
       lip--;

    if (lip) {
       move(Line+lip, 3*SCREEN_HEIGHT>>2);	/* From 3/4 up to	*/
       cont(Line, 3*SCREEN_HEIGHT>>2);
    } else
       move(Line, 3*SCREEN_HEIGHT>>2);
    cont(Line, SCREEN_HEIGHT>>2);	   	/* 1/4 down.		*/
    if (lip)
       cont(Line+lip, SCREEN_HEIGHT>>2);
    return(OK);
    }
/* ******************************************************************** */

/* FUNCTION PrintGraphicsScaling */
	 /* Print screen width X height, and tic spacing if flag is set	*/
void PrintGraphicsScaling(float HorizontalTicWidth) {
   int  CharIndent =  18;			/* Units of characters	*/
   char HeightUnits[6];
   extern int graph_events, time_tics;		/* chg.c		*/

   switch (CurrentChannel) {
      case OD_H: case OD_V: case ARM_H: case ARM_V:	/* Positions:	*/
      case ARM_H_WIDTH: case ARM_V_WIDTH:
      case HV_POS:					/* Summed ampl	*/
	  CharIndent *= 3;
	  strcpy(HeightUnits, "d");
	  break;
      case UNIT:					/* Unit firing	*/
      case SPIKE_2: case SPIKE_3: case SPIKE_4:
	  CharIndent *= 1;
	  strcpy(HeightUnits, "Hz");
	  break;
      case H_VEL: case V_VEL: case HV_VEL:
	  CharIndent *= 2;
	  strcpy(HeightUnits, "d/s");
	  break;
      default:						/* All else	*/
	  CharIndent *= 4;
	  strcpy(HeightUnits, "units");
      }

   GraphFontSize(15);
   move(0, SCREEN_HEIGHT+50);
   GraphPrintf("%4d ms X %*.*f %s",
      PixToTime(SCREEN_WIDTH)-PixToTime(0),		/* Width	*/
      CharIndent-10,					/* Indentation	*/
      (PixToTime(SCREEN_WIDTH)-PixToTime(0)<10)? 2:1,	/* Precision	*/
      PixToVel(SCREEN_HEIGHT)-PixToVel(0),		/* Screen height*/
      HeightUnits);					/* Units	*/

   if (!graph_events)				/* Skip tic spacing	*/
      return;
		
   move(0,SCREEN_HEIGHT + 25);
   GraphPrintf("[%d]",
      (time_tics) ?  time_tics :
        abs(PixToTime((SCREEN_WIDTH+5)/10)-PixToTime(0)));
   if (HorizontalTicWidth != 0)
     GraphPrintf("%*s     [%.1f]",	/* Duration scale reduced by 10	*/
        CharIndent-10+2, "",		/* Hacked to get good spacing	*/
        (float)fabs(2*HorizontalTicWidth));	/* Between wide tics	*/
}
/* ******************************************************************** */

/* FUNCTION GraphEvents */
	 /* Add scales, lines at chair steps, zero line			*/
void GraphEvents(int reg) {

float	Interval;				/* VELOC tween thin tics*/
int     SaveLineType = Set_LineType(NO_CHANGE);

			/* *** EVENTS  *** */	
if (reg==0) {
#include "_imports/deffs.h"
#include "_imports/event.h"
#ifdef AUTOMATIC_LINES
int	when;
int 	occurrence;

Set_LineType(2);
occurrence = 0;							/* CUES	*/
while ((when=StackExtract(ARM_CUE,++occurrence,TIME))!=FAIL)
   GraphLine(when, (StackExtract(ARM_CUE,occurrence,VALUE)? 2 : -2));

occurrence = 0;
while ((when=StackExtract(EYE_CUE,++occurrence,TIME))!=FAIL)
   GraphLine(when, (StackExtract(EYE_CUE,occurrence,VALUE)? 2 : -2));
#endif

Set_LineType(3);
}

			/* ***  HORIZONTAL SCALE *** */
{ int wide = 0;					/* 1st tic wide		*/
  int i = 0;
  extern int time_tics, time_tics_offset;	/* chg.c		*/
  Set_LineType(1);

if (time_tics) {
  int at = PixToTime(0) + i * time_tics + time_tics_offset;
  do {
    at = TimeToPix(at);
    move(at, 20);
    cont(at, 0);				/* Avoid rounding errs	*/
    at = PixToTime(0) + (++i * time_tics + time_tics_offset);
   } while (at < PixToTime(SCREEN_WIDTH));
  } else {
  for (i=0; i<=20; i++) {
    int at = (i*SCREEN_WIDTH+10) / 20;		/* Rounded 1/20 intvl's	*/
    move(at, ((wide=!wide) ? 20:7));		/* Draw as per 'wide'	*/
    cont(at,0);
    }
  }
}

		/* ***  VERTICAL SCALING AND TICS *** */
  { float   TestHeight = (float) fabs(SCREEN_HEIGHT/y_mult); 
        if (TestHeight < 1.0)		Interval = TestHeight/8;
   else if (TestHeight <  5.0)		Interval = 1.0;
   else if (TestHeight <  20.0)		Interval = 2.5;
   else if (TestHeight <= 50.0)		Interval = 5.0;
   else if (TestHeight <= 100.0)		Interval = 10.0;
   else if (TestHeight <= 200.0)		Interval = 25.0;
   else if (TestHeight <= 500.0)		Interval = 50.0;
   else					Interval = TestHeight/8;
  }

  {
  int  height = irint(VelToPix(0));		  /* Index screen height*/
  int  tic  = irint(fabs(VelToPix(Interval)-VelToPix(0))); /*Pixs 'tween tics*/
  int  wide = 0;				  /* Bool: wide tic?	*/

  while (height>0) {				/* Get underneath screen*/
     height -= tic;				/*  Move downward 1 tic	*/
     wide = 1- wide;				/*  wide->thin or th->wd*/
     }
  while (height<=1) {				/* Rise to just onscreen*/
     height += tic;				/*  Don't draw 0 tic	*/
     wide = 1- wide;				/*  keep track of width	*/
     }

  while (height <= SCREEN_HEIGHT) {		/* Here to screen top:	*/
     move(SCREEN_WIDTH+23, height);		/*  Aprropriate height	*/
     cont(SCREEN_WIDTH+((wide)?16:1), height);	/*  Appropriate width	*/
     wide = 1 - wide;				/* Toggle 'wide'	*/
     height += tic;				/* Move up 1 (thin) tic	*/
     }
  }

PrintGraphicsScaling(Interval);			/* Even in command mode!*/
GraphZeroLine();				/* ZERO LINE		*/
Set_LineType(SaveLineType);
return;
}
/* ******************************************************************** */

/* FUNCTION GraphZeroLine */
	 /* Add zero line to plot	*/
void GraphZeroLine(void) {
   float i = VelToPix(0);	

   if (i>SCREEN_HEIGHT || i<0.)			/* Zero off-screen?	*/
      return;
   move(0, i);
   cont(SCREEN_WIDTH, i);
   return;
}
/* ******************************************************************** */

/* FUNCTION GraphErase */
	 /* Calls for files w/o access to graph.h (plot library)	*/
void GraphFontSize(int size)	{ fontsize(size); }
void GraphFontName(char *name)	{ fontname(name); }
void GraphErase(void)		{ erase(); }
/* ******************************************************************** */

/* FUNCTION GraphText */
	 /* Call erase for files w/o access to graph.h (plot library) */
void GraphText(int x, int y, char *text) {
	move(x,y);
	label(text);
	}
/* ******************************************************************** */
/* ******************************************************************** */

/* FUNCTION GraphPrintf */
void GraphPrintf(char *format, ...) {
   char printstring[255];
   va_list args;

   va_start(args, format);
   vsprintf(printstring, format, args);
   va_end(args);
   label(printstring);
   }
/* ******************************************************************** */

/* FUNCTION GraphFlush */
void GraphFlush() { flushpl(); }
/* ******************************************************************** */
