/* FILE file.c */
      /* Transfer registers from program to disk & vice versa		*/
      /* FUNCTIONS:
       *    OpenFile:	Construct new unique mnemonic name & open file
       *    NameFile:	Return name of open file
       *    StatFile:	Write stats to file
       *    RawFile:	Write DataArray to file, raw
       *    AvgFile:	Calculate average and stddev and write to file
       *    TextFile:   Ascii output (another format)
       *    ReadFile:	Read Unix disk raw data (sum,sum2) into a file
       *    CloseAllFiles:  In desperation, close all FILE DESCRIPTORS
       */

/* NOTE: file writing: write only channels actually used.  For 'x' channels,
	use DataFlag() to _construct_ flag, and then use this constructed
	flag (NOT the global dataflag) to determine recorded channels (NOT
	RecordedChannel()).
       */

/* ********************************************************************	*/
#include "defs.h"
#include "array.h"
#include <fcntl.h>					/* open()	*/

#define NCOLN	5	/* time, eye posit & veloc, chair & world veloc	*/
#define CANNOT_OPEN	-1		/* Returned by failed open()	*/	
#define READ		0		/* Mode, passed to open		*/
#define WRITE		1		/* Mode, passed to open		*/
#define MASK		0644		/* Read by all, write by owner	*/
#define INT		sizeof(int)

char NewName[255];		/* Written file's name 		*/

static int DirtyWorkForReadFile(int reg, char *DiskName);
/* ********************************************************************	*/

/* FUNCTION OpenFile */
	 /* Open file with unique mnemonic name				*/

int OpenFile(char *filetype) {
 int   fd = CANNOT_OPEN;			/* File descriptor	*/
 char *name;
 char *directory;
 char  path[255];
 int   i;

getcwd(path, 250);
directory = strrchr(path, '/');		/* Ptr to current directory	*/
if (directory[strlen(directory)-2] == '.') {
   i = strlen(directory);
   directory[i-2] = directory[i-1];
   directory[i-1] = '\0';
   }
strcpy(NewName, filetype);			/* "a","r","vc", etc	*/
name = NewName + strlen(filetype);

if ((int)strlen(filetype) < 4) {		/* Only short names get-*/
   strncpy(name, directory+1, 6); 		/* 1st 6 chars of dir	*/
   name += ((int)strlen(directory+1) > 6) ? 6 : (int)strlen(directory+1);
   }

*name = '\0';					/* Needed for sprintf	*/
if ((int)strlen(filetype) > 2) {		/* Only long  names get-*/
   sprintf(NewName, "%s_%s%d",
      NewName, (Header_Unit()<10)?" ":"", Header_Unit());
   *name += 3;
   }
*name = '\0';					/* Name is now complete	*/

if (*filetype == 'r' ||				/* Always number r files*/
    (fd = open(NewName, READ)) != CANNOT_OPEN) {       	/* File exists!!*/	
   *(name++) = '.';					/* Evasive actio*/
   *(name++) = '1';					/* Append '1'	*/
   *(name--) = '\0';					/* Ptr at '1'	*/
   close(fd);						/* Housekeeping	*/
   }

if ((fd = open(NewName, READ)) != CANNOT_OPEN) {	/* Try,try again*/
   int tens = 0;
   do {						/* This exists, too!?!	*/
      close(fd);				/* Housekeeping again	*/
      if ((*name)++ == '9') {			/* Increment & test	*/
	  if (tens) {
	     (*(name-1))++;			/* Incr tens place	*/
	     *name = '0';
	  } else {
	     *(name++) = '1';
	     *(name++) = '0';
	     *(name) =  '\0';
	     name--;
	     tens = 1;
	     }
	  }
      } while ((fd = open(NewName, READ)) != CANNOT_OPEN);
   }

if ((fd = creat(NewName, MASK)) == CANNOT_OPEN)	/* Legit fd!! But won't write */
   fprintf(stderr,"\nCan't create file %s",NewName);

return(fd);			/* If unsuccessful, returns CANNOT_OPEN */
}
/* END OpenFile */
/* ********************************************************************	*/

/* FUNCTION NameFile */
	 /* Return name of open file	*/
char *NameFile(void) { return(NewName); }
/* END NameFile */
/* ********************************************************************	*/

/* FUNCTION StatFile */
	 /* In constant file write time,mean,stdev,stder,N of interval	*/
#define SPAN  2					/* Interval = +/- span	*/
int StatFile(int reg, int time) {
FILE	*fdopen(int fildes, const char *type);
		/* File descriptor->stream for buffered I/O; not ANSI	*/
static   FILE 	*ThisFile = NULL;
FRAME  *F, *G;
int		 Try,					/* Cnt test pts */
		 In,					/* Pts used	*/
		 acN;					/* Accumulate #	*/
double		 acM, acSD, acSE,			/* Ac m,stdev/er*/
		 stddev;				/* Temporary var*/

if (ProofReg(reg) == FAIL)
   return(FAIL);
if (ThisFile == NULL) {
   ThisFile = fdopen(OpenFile("s"), "w");
   if (ThisFile == NULL) {
      fprintf(stderr,"Can't open file!");
      return(FAIL);
      }
   fprintf(ThisFile,"%6s%8s%8s%8s%8s\n","Time","Mean","Stddev","Stderr","N");
   }

G = Register[reg].channel[CurrentChannel].frame;
F = G + time - SPAN;
acM = acSD = acSE = 0.0;
acN = In = 0;
Try = (2*SPAN + 1);

while (Try--) {
   if (F->n > 1) {				/* Skip 0s and 1s     */
      In++;					/* Points used in avgs*/
      acN  += F->n;				/* Trials used in avgs*/
      acM  += MEan(F);
      stddev = SD(F);
      acSD += stddev;
      acSE += (float)(stddev/sqrt((double)(F->n)));
      }
   F++;
   }

F -= (SPAN+1);

if (In == 0) { 						/* No points	*/ 
   vtprint("Too few data in neighborhood");
   return(FAIL);
   }

fprintf(ThisFile,"\n%6ld%8.2f%8.2f%8.2f%8.0f",
	 (long)(F-G), acM /In, acSD/In, acSE/In, .5 + (float)acN/In);
fflush(ThisFile);
return(OK);
}
/* END StatFile */
/* ********************************************************************	*/

/* FUNCTION AvgFile */
	 /* Writes mean and stdev to disk; pass it register to use	*/
int AvgFile(int reg) {
FILE	*fdopen(int fildes, const char *type),
	*ThisFile;
int     i;
int     Flag = DataFlag(reg);		/* Construct by looking at data	*/

if (ProofReg(reg) == FAIL)
   return(FAIL);
ThisFile = fdopen(OpenFile("t"), "w");
if (ThisFile == NULL) {
   vtprint("Can't open file!");
   return(FAIL);
   }

for (i=0; i<MAX_CHANNELS; i++)
   if (Flag & (1<<i)) {
      int FrameCount;
      FRAME *G = &(Register[reg].channel[i].frame[0]);
      FRAME *F = &(Register[reg].channel[i].frame[0]);

      fprintf(ThisFile, "Channel %d\n", i);

      for (FrameCount=Register[reg].frame_count; FrameCount--; F++)
	 if (F->n == 1)
            fprintf(ThisFile,"%5ld ms: %7.1f\n", (long)(F-G), F->sum);
	 else if (F->n > 1)
            fprintf(ThisFile,"%5ld ms: %7.1f %5.1f %5.1f %4d\n", 
			(long)(F-G), MEan(F), SD(F), SE(F), F->n);
      }

fclose(ThisFile);					/* Close & flush*/
return(OK);
}
/* END AvgFile */
/* ********************************************************************	*/

/* FUNCTION TextFile */
	 /* More compact version of AvgFile; give it register & file ptr */
int TextFile(int reg, FILE *ThisFile, int format) {
int     f,i;

if (ProofReg(reg) == FAIL)
   return(FAIL);
if (ThisFile == NULL) {
   vtprint("Can't open file!");
   return(FAIL);
   }

switch (format) {
 case 1:
   for (f=0; f<Register[reg].frame_count; f++) {
      FRAME *F = &(Register[reg].channel[0].frame[f]);	/* Init: insure*/
      for (i=0; i<MAX_CHANNELS; i++) {
         if (i<5) {	/* Units, eye [H and V], chair [pos & vel] */
            F = &(Register[reg].channel[i].frame[f]);
            fprintf(ThisFile,"%.1f %.1f   ", MEan(F), SD(F));
            }
         }
      fprintf(ThisFile, "   %d\n", F->n); 
      if ((F+1)->n == 0 && (F+2)->n == 0)
         break;
      }
   break;

 case 2:
   fprintf(ThisFile, 
     " ms      H   H.sd     V   V.sd   rate rate.sd  N     Register %d\n", reg);
   {
   FRAME *Fu = &(Register[reg].channel[UNIT].frame[0]),
	 *Fh = &(Register[reg].channel[OD_H].frame[0]),
	 *Fv = &(Register[reg].channel[OD_V].frame[0]);

   for (f=0; f<Register[reg].frame_count; f++) {	/* Go thru empties */
      if (Fu->n > 0)
         break;
      Fh++;
      Fv++;
      Fu++;
      }

   for (   ; f<Register[reg].frame_count; f++) {
      fprintf(ThisFile, "%5d  %5.1f %5.1f  %5.1f %5.1f  %5.1f %5.1f  %3d\n",
	      f, MEan(Fu), SD(Fu),
	         MEan(Fh), SD(Fh),
      	         MEan(Fv), SD(Fv),
              Fu->n); 
      (Fu)++;				/* Next frame (same channel)	*/
      (Fh)++;
      (Fv)++;
      if (Fu->n == 0 && (Fu+1)->n == 0)	/* If next two are empty, stop	*/
         break;
      }
   break;
   }
 }
return(OK);
}
/* END AvgFile */
/* ********************************************************************	*/

/* FUNCTION RawFile */
	 /* Writes array to a file					*/
int RawFile(int reg) {
int	ThisFile;				/* File descriptor	*/
int 	StackSize = 3 * MAX_EVENTS * sizeof(short);
int	FileVersion = 7;			/* 7th version of r_file*/
					/* 7: 7-7-99 LHS: includes tag	*/
int     Flag = DataFlag(reg);		/* Construct by looking at data	*/
int	i;

if (ProofReg(reg) == FAIL)
   return(FAIL);
ThisFile = OpenFile("r");
if (ThisFile == CANNOT_OPEN) {
   vtprint("Can't open file!");
   return(FAIL);
   }

write(ThisFile, &(FileVersion), INT);
i = strlen(Register[reg].tag);
write(ThisFile, &(i), INT);
write(ThisFile, Register[reg].tag, i);
write(ThisFile, &(Register[reg].frame_count), INT);
write(ThisFile, &(Register[reg].trial_count), INT);
write(ThisFile, &(StackSize), INT);
write(ThisFile, &(Flag), INT);
if (RecordedChannel(UNIT))
   write(ThisFile, Register[reg].spikes, Register[reg].frame_count * INT);
for (i=0; i<MAX_CHANNELS; i++)
   if (Flag & (1<<i))
      write(ThisFile, Register[reg].channel[i].frame,
      			Register[reg].frame_count*sizeof(FRAME));
write(ThisFile, Register[reg].stack,  StackSize);
write(ThisFile, Register[reg].stackname,  sizeof(Register[reg].stackname));
write(ThisFile, &(Register[reg].stacknumber),  INT);
write(ThisFile, &(Register[reg].classnumber),  INT);

close(ThisFile);				/* Close & flush	*/
return(OK);
}
/* END RawFile */
/* ********************************************************************	*/
#define SIZE 255

/* FUNCTION ReadFile */
	 /* Read a unix file from disk (sum,sum2) into a file		*/
	 /* Should compare INFO_new to INFO and appropriately modify	*/
int ReadFile(int reg) {
char 		DiskName[SIZE];			/* Array to be read	*/

if (ProofReg(reg) == FAIL)
   return(FAIL);
if (!(Status & COMMAND_MODE)) {
   fprintf(stderr," ls r*\n\n\t");
   system("sh -c '\\ls r* 1>&2'");		/* Redirect to stderr	*/
   }
inputs("Disk file to read: ",DiskName);		/* Get disk file name	*/
if (DirtyWorkForReadFile(reg,DiskName) == CANNOT_OPEN) {
   fprintf(stderr,"\nCannot read file %s\n", DiskName);
   return(FAIL);
   }
return(OK);
} /* END ReadFile */

/* ********************************************************************	*/
#include <ctype.h>
/* FUNCTION ReadFiles(reg) */
	 /* Read sequential unix files until failure	*/
int ReadFiles(int reg) {
char 	DiskName[SIZE];
int	Index = 0;

if (ProofReg(reg) == FAIL)
   return(FAIL);
if (!(Status & COMMAND_MODE))
   system("sh -c '\\ls r* 1>&2'");		/* Redirect to stderr	*/
inputs("First sequential disk file to read: ",DiskName);
if (DiskName[0] == NONE)
   return(FAIL);
while (DiskName[++Index]) ;			/* Find end of name	*/
Index--;

while (DirtyWorkForReadFile(reg,DiskName) == OK) {
   if (DiskName[Index] == '9') {
      if (isdigit(DiskName[Index-1])) {			/* eg, r_01.09	*/
         DiskName[Index] = '0';
         DiskName[Index-1] += (int) 1;
       } else {						/* eg, r_01.9	*/
	 DiskName[Index] = '1';
	 DiskName[Index+1] = '0';
	 DiskName[Index+2] = NONE;
	 Index++;
	 }
      }
   else
      DiskName[Index] += (int) 1; 
   if (ProofReg(++reg) == FAIL)			/* Is next reg there?	*/
      return(FAIL);
   }

if (!(Status & COMMAND_MODE))
   fprintf(stderr,"\nCannot read file %s\n", DiskName);
return(FAIL);
}
/* ******************************************************************** */

/* FUNCTION DirtyWorkForReadFile */
	 /* Does actual file reading for ReadFile & ReadFiles	*/
static int DirtyWorkForReadFile(int reg, char *DiskName) {
int   DiskFile;				/* File descriptor	*/
long  OverFlow = 0;
int   FileVersion;
int   StackSize, Flag = 0;
int i;

if ((DiskFile=open(DiskName,READ)) == CANNOT_OPEN)    /* Open disk file	*/
   return(CANNOT_OPEN);
if (!(Status & COMMAND_MODE))
   fputs(".", stderr);

read(DiskFile, (char *)&(FileVersion), INT);

if (FileVersion <= 5) {
   fprintf(stderr, "\nUnreadable r-file <%s>.  See r-convert documentation.",
   	DiskName);
   return(FAIL);
   }

if (FileVersion > 6) {
   read(DiskFile, &i, INT);
   read(DiskFile, (char *)&(Register[reg].tag), i);
} else
   strcpy(Register[reg].tag,DiskName);		/* Use unix file name	*/

read(DiskFile, (char *)&(Register[reg].frame_count), INT);
read(DiskFile, (char *)&(Register[reg].trial_count), INT);
read(DiskFile, (char *)&(StackSize), INT);
read(DiskFile, (char *)&(Flag), INT);

if (Register[reg].frame_count > MAX_FRAMES) {
   fprintf(stderr, "\n%d frames: truncating to %d",
   			Register[reg].frame_count, MAX_FRAMES);
   OverFlow = Register[reg].frame_count - MAX_FRAMES;
   Register[reg].frame_count = MAX_FRAMES;
   }

Write_Header_dflag(Flag);	/* Add check if CurrentChan still legal?*/

if (RecordedChannel(UNIT)) {		/* Read 'spikes'		*/
   read(DiskFile, (char *)Register[reg].spikes,INT * Register[reg].frame_count);
   lseek(DiskFile, OverFlow * INT, 1);
   }
if (Flag & (077<<MAX_CHANNELS))
   vtprint("Error: Too many channels to swallow! (incrase MAX_CHANNELS)");
for (i=0; i<MAX_CHANNELS; i++)
   if (Flag & (1<<i)) {		/* != RecordedChans(i) for i >= X1	*/
      read(DiskFile, (char *) Register[reg].channel[i].frame,
      				 Register[reg].frame_count*sizeof(FRAME));
      lseek(DiskFile, OverFlow * sizeof(FRAME), 1);
   } else {					/* Explicit zeroing	*/
      FRAME  *zero = &(Register[reg].channel[i].frame[0]);
      int    Count = MAX_FRAMES;
      while (Count-- > 0) {
         zero->n = zero->sum = zero->sum2 = 0;
         zero++;
         }
      }
read(DiskFile, (char *)Register[reg].stack,  StackSize);
read(DiskFile, Register[reg].stackname, sizeof(Register[reg].stackname));
read(DiskFile, &(Register[reg].stacknumber),  INT);
read(DiskFile, &(Register[reg].classnumber),  INT);
close(DiskFile);				/* Close & flush 	*/
return(OK);
}
/* ******************************************************************** */

/* FUNCTION CloseAllFiles */
	 /* Desperation	*/
void CloseAllFiles(void) {
int i;
for (i=4;i<20;i++)
    close(i);
} /* END CloseAllFiles */
/* ******************************************************************** */
