/* FILE data.c */
     /* Operate on CurrentChannel of specified register			*/
     /* NOTE: many operations are incorrect when applied to UNIT channel*/
     /* EVENTUALLY: ought to have global variable 'currentfunction'	*/
     /*	            -- very useful for debugging!			*/

     /* FUNCTIONS:
      *   GetVergenceData:	Get vergence and converge of eye position
      *	  SubtractData:		Subtract channel from current channel
      *	  DivideData:		Subtract current channel from two registers
      *	  ComputeAmplitudesData: Make euclidean position and veloc
      *	  ComputeProjectedAmplitudesData: Ditto, but re: a particular direction
      *   ChannelDataTransfer 	Move current channel data into other channel
      *   DivergeData:		Return diverge time of 2 registers
      *   DivergeFromZeroData:	Return diverge time from zero (1 reg)
      *   OffsetData:		Add offset to EV in all frames
      *   MultiplyData:		Multiply channel 'what' by constant
      *   HalveData:		Half-wave rectify below/above a certain point
      *   RectifyData:		Full-wave rectify
      *   DeleteData:		Remove current channel over given interval
      *   DeleteAllData:	Remove channels 1-MAX over given interval
      *   AvgSomeData:		Average values over given interval
      *   StatsSomeData:	Return Avg,SD,SE or N over interval
      *   AvgWrappedData:	Ditto, but wrap if bounds exceed limits
      *   FindPeakOfData:	Find peak value
      *   FindFirstDataOver:	Time of 1st pt < value; start at given time
      *   FindFirstDataUnder:	Time of 1st pt < value; start at given time
      *   FindDataZeroCrossings:When is data within < 0.1 ?
      *   DataFlag:		Any data recorded? Used by file.c
          (static)
      *   SetBounds:		convert ms->index for interval passed
      */

static int begin;
static int enddd;			/* Lint complains if just 'end'	*/

#include "defs.h"				/* Misc constants	*/
#include "array.h"				/* Array declarations	*/
#include <stdlib.h>				/* atof()		*/

static void DeleteFrame(int reg, int frame);
static int SetBounds(int reg,int start,int stop, char *functionname);
/* ******************************************************************** */

/* FUNCTION GetVergenceData */
void GetVergenceData(int reg) {
   if (reg!=INVALID_REG) vtprint("Cannot do this!");
   return;
   }
/* ******************************************************************** */

/* FUNCTION SubtractData */
	 /* In Register 'reg', channel Current -= minus			*/
int SubtractData(int reg, int minus) {
if (ProofReg(reg) == FAIL)
   return(FAIL);
if (!RecordedChannel(minus))			/* Test requested chan	*/
   vtprint("Channel not recorded");

if (Register[reg].trial_count > 1)		/* Sum2-Sum2 = nonsense	*/
   vtprint("Unnormalized data -- statistics may be wrong!");

fprintf(stderr, "DIAG: Might be broken! (LHS 7-19-00) Cmp results to SubtractReg()\n");


{
FRAME *F    = Register[reg].channel[CurrentChannel].frame;
FRAME *Minus= Register[reg].channel[minus].frame;
int Count = Register[reg].frame_count;

while (Count--) {
   switch (F->n) {
	case 0:			/* Easiest case; just transfer	*/
             F->sum   = -Minus->sum;
             F->sum2  =  Minus->sum2;
             (F++)->n = (Minus++)->n;
	     break;
	case 1:			/* Easy case; just subtract sum	*/
	  if (Minus->n == 1) {
      	     F->sum  -= (Minus++)->sum;
      	     F->sum2 = F->sum * F->sum;
      	     F++;
	     break;
	     }
	default:	/* Judgement call; use larger variance, smaller n*/
	     {
#    define VARiance(p) (p->sum2 - p->sum * p->sum / p->n) / (p->n -1)

             FRAME *LargerVAR = (VARiance(F) > VARiance(Minus)) ? F : Minus;

             F->sum2 =			/* Incomplete calc of sum2 ...	*/
	       LargerVAR->sum2 - LargerVAR->sum*LargerVAR->sum/LargerVAR->n;
             F->sum  =			/* Subtraction MEANS, not sums	*/
               F->sum/F->n - Minus->sum/Minus->n;
             F->n = MIN(F->n,Minus->n);		/* Use the smaller N	*/
	
             F->sum  *= F->n;			/* Complete 'n' calc	*/
             F->sum2 += F->sum * F->sum / F->n;	/* Complete sum2 calc	*/

             Minus++;
             F++;
	     }
        }
   }
}

return(OK);
}
/* END SubtractData */
/* ******************************************************************** */

/* FUNCTION DivideData */
	 /* In Register 'reg', channel Current /= divide	*/
int DivideData(int reg, int divide) {

if (ProofReg(reg) == FAIL)
   return(FAIL);
if (!RecordedChannel(divide))			/* Test requested chan	*/
   vtprint("Channel not recorded");

if (Register[reg].trial_count > 1 || Register[divide].trial_count > 1) {
   vtprint("Unnormalized data -- must write more code!");
   return(FAIL);
   }

{
FRAME *F     = Register[reg].channel[CurrentChannel].frame;
FRAME *Divide= Register[divide].channel[CurrentChannel].frame;
int Count = Register[reg].frame_count;

while (Count--) {
   if (F->n==0 || Divide->n==0) {		/* If either has no data*/
      F->sum = F->sum2 = 0;
      (F++)->n = 0;
      Divide++;
    } else {
      F->sum /= (Divide++)->sum;
      F->sum2 = F->sum * F->sum;
      F++;
      }
   }
}
return(OK);
}
/* END DivideData */
/* ******************************************************************** */

/* FUNCTION ComputeAmplitudesData */
	 /* Compute instantaneous position & velocity amplitudes	*/
	 /* One-dimensional and UNSIGNED (amplitude)			*/
int ComputeAmplitudesData(int reg) {
FRAME *F1 = Register[reg].channel[OD_H].frame;
FRAME *F2 = Register[reg].channel[OD_V].frame;
FRAME *F3 = Register[reg].channel[CurrentChannel].frame;
int Count = Register[reg].frame_count;

if (ProofReg(reg) == FAIL)
   return(FAIL);

while (Count--) {
   if (F1->n==0 || F2->n==0) {		/* If either has no data*/
      F3->sum = F3->sum2 = 0;
      F3->n = 0;
    } else {
      F3->sum = sqrt(MEan(F1)*MEan(F1) + MEan(F2)*MEan(F2));
      F3->sum2 = F3->sum * F3->sum;
      F3->n = 1;
      }
    F1++; F2++; F3++;
   }

F1 = Register[reg].channel[H_VEL].frame;  /* Making an assumption here! */
F2 = Register[reg].channel[V_VEL].frame;
F3 = Register[reg].channel[CurrentChannel+1].frame;
Count = Register[reg].frame_count;

while (Count--) {
   if (F1->n==0 || F2->n==0) {		/* If either has no data*/
      F3->sum = F3->sum2 = 0;
      F3->n = 0;
    } else {
      F3->sum = sqrt(MEan(F1)*MEan(F1) + MEan(F2)*MEan(F2));
      F3->sum2 = F3->sum * F3->sum;
      F3->n = 1;
      }
    F1++; F2++; F3++;
   }
return(OK);
}
/* ******************************************************************** */

/* FUNCTION ComputeProjectedAmplitudesData */
	 /* Compute pos & veloc projected along a particular vector	*/
	 /* Direction should be in degrees				*/
int ComputeProjectedAmplitudesData(int reg, int direction) {
FRAME *P1 = Register[reg].channel[OD_H].frame;
FRAME *P2 = Register[reg].channel[OD_V].frame;
FRAME *P3 = Register[reg].channel[CurrentChannel].frame;

FRAME *V1 = Register[reg].channel[H_VEL].frame;	/* Note the assumption!	*/
FRAME *V2 = Register[reg].channel[V_VEL].frame;
FRAME *V3 = Register[reg].channel[CurrentChannel+1].frame;
int Count = Register[reg].frame_count;

if (ProofReg(reg) == FAIL)
   return(FAIL);

direction = direction / (360 / (2*3.14159));	/* degrees to rads	*/

while (Count--) {
   if (P1->n==0 || P2->n==0) {			/* If either has no data*/
      P3->sum = P3->sum2 = V3->sum = V3->sum2 = 0;
      P3->n = V3->n = 0;
    } else {
      double Amplitude = sqrt(MEan(P1)*MEan(P1) + MEan(P2)*MEan(P2));
      double Angle = atan2(MEan(P2), MEan(P1));

      P3->sum = Amplitude * cos(Angle-direction);	/* Projection	*/

      P3->sum2 = P3->sum * P3->sum;
      P3->n = 1;

      Amplitude = sqrt(MEan(V1)*MEan(V1) + MEan(V2)*MEan(V2));
      Angle = atan2(MEan(V2), MEan(V1));

      V3->sum = Amplitude * cos(Angle-direction);	/* Projection	*/

      V3->sum2 = V3->sum * V3->sum;
      V3->n = 1;
      }
    P1++; P2++; P3++;
    V1++; V2++; V3++;
   }

return(OK);
}
/* ******************************************************************** */

/* FUNCTION ChannelDataTransfer */
	 /* In Register 'reg', transfer channel Current to new		*/
int ChannelDataTransfer(int reg, int new) {
if (ProofReg(reg) == FAIL)
   return(FAIL);

{
FRAME *F   = Register[reg].channel[CurrentChannel].frame;
FRAME *New = Register[reg].channel[new].frame;
int Count = Register[reg].frame_count;

while (Count--) {
   New->sum   =  F->sum;
   New->sum2  =  F->sum2;
   (New++)->n = (F++)->n;
   }
}

return(OK);
}
/* END ChannelDataTransfer */
/* ******************************************************************** */

int   Diverge_Points = 10;		/* This many contiguous points	*/
float Diverge_StdErrs =  3;		/* Each of which this far apart	*/
float Diverge_MaxStdErr =  0.6;		/* Set err flag if exceeds this	*/
int   Diverge_Backwards = 0;		/* Bool: look backwards in time	*/

int   MaxStdErr = 0;					/* Error flag	*/

/* FUNCTION SetupDivergeData */
	 /* Set paramters for diverge data	*/
void SetupDivergeData(void) {
   char answer[80];

   fprintf(stderr, "Enter number points [%d]", Diverge_Points);
   inputs(" > ", answer);
   if (answer[0] != NONE)
      Diverge_Points = atoi(answer);

   fprintf(stderr, "Enter stderrs [%.2f]", Diverge_StdErrs);
   inputs(" > ", answer);
   if (answer[0] != NONE)
      Diverge_StdErrs = (float) atof(answer);

   fprintf(stderr, "Enter big stderr [%.2f]", Diverge_MaxStdErr);
   inputs(" > ", answer);
   if (answer[0] != NONE)
      Diverge_MaxStdErr = (float) atof(answer);

   Diverge_Backwards =		/* Anything but 'no' sets it equal to 0	*/
   	query("Search direction [y==forward/n==backward] > ") == 0;
   fprintf(stderr, "\n");
}
/* ******************************************************************** */

/* FUNCTION DivergeData */
	 /* Return diverge time (1st of n points differing by y stderrs)*/


int DivergeData(int reg1, int reg2) {
FRAME *F1 = &(Register[reg1].channel[CurrentChannel].frame[0]);
FRAME *F2 = &(Register[reg2].channel[CurrentChannel].frame[0]);
FRAME  *FirstFrame = F1;
int Count;
int DivergedPoints = 0;
float SaveFirstStderr = 0.0;

if (ProofReg(reg1) == FAIL)
   return(FAIL);
if (ProofReg(reg2) == FAIL)
   return(FAIL);
Count = MIN(Register[reg1].frame_count,Register[reg2].frame_count);

if (Diverge_Backwards) {		/* Then start at the end	*/
   F1 += Count;
   F2 += Count;
   }

while (Count--) {
   if (Diverge_Backwards) {
       F1--; F2--;				/* Look backwards	*/
   } else {
       F1++; F2++;				/* Look forwards	*/
       }
   if (F1->n && F2->n) {
#    ifdef STDDEV
      float stderr1 = SD(F1);
      float stderr2 = SD(F2);
#    else
      float stderr1 = SE(F1);
      float stderr2 = SE(F2);
#    endif
      float criterionSE = MAX(stderr1,stderr2) * Diverge_StdErrs;

      if (fabs((double)(MEan(F1)-MEan(F2))) >= criterionSE) {
         if (DivergedPoints == 0)		/* 1st pt of diverge?	*/
	    SaveFirstStderr = criterionSE;	/* Store stderr		*/
	 if (++DivergedPoints >= Diverge_Points) {
	    if (SaveFirstStderr > Diverge_MaxStdErr)
	       fprintf(stderr, "Large stderr (regs %d,%d)\n", reg1, reg2);
		/* return(FAIL); */
	    if (Diverge_Backwards)
	       return(((int)(F1-FirstFrame))+Diverge_Points-1);
	    else
	       return(((int)(F1-FirstFrame))-Diverge_Points+1);
	    }
         }
         else
	    DivergedPoints = 0;
         }
       else
         DivergedPoints = 0;
      }
   return(FAIL);
}
/* END DivergeData */
/* ******************************************************************** */

/* FUNCTION DivergeFromZeroData */
	 /* Return diverge time from zero				*/

int DivergeFromZeroData(int reg) {
FRAME *F = &(Register[reg].channel[CurrentChannel].frame[0]);
FRAME  *FirstFrame = F;
int Count;
int DivergedPoints = 0;
float SaveFirstStderr = 0.0;		/* Init: Make -Wall happy	*/

if (ProofReg(reg) == FAIL)
   return(FAIL);
Count = Register[reg].frame_count;

if (Diverge_Backwards)			/* Then start from the end	*/
    F += Count;

while (Count--) {
   if (Diverge_Backwards)
      F--;
   else
      F++;
   if (F->n) {
      float stderr1 = SE(F);

      if (fabs((double)MEan(F)) >= stderr1 * Diverge_StdErrs) {
         if (DivergedPoints == 0)		/* 1st pt of diverge?	*/
	    SaveFirstStderr = stderr1;		/* Store stderr		*/
	 if (++DivergedPoints >= Diverge_Points) {
	    if (SaveFirstStderr > Diverge_MaxStdErr)
	       fprintf(stderr, "Large stderr (diverge of %d)\n",reg);
            if (Diverge_Backwards)
	       return(((int)(F-FirstFrame))+Diverge_Points-1);
	    else
	       return(((int)(F-FirstFrame))-Diverge_Points+1);
	    }
         }
      else
	 DivergedPoints = 0;
      }
    else
      DivergedPoints = 0;
    }
return(FAIL);
}
/* END DivergeFromZeroData */
/* ******************************************************************** */

/* FUNCTION OffsetData */
	 /* Accept and add offset to all CurrentChannel of Trial	*/
int OffsetData(int reg, float offset) {
if (offset == FAIL) {					/* Not enuf pts	*/
   vtprint("Not offset!");
   return(FAIL);
   }
if (ProofReg(reg) == FAIL)
   return(FAIL);
offset = -offset;					/* Change sign	*/

{
FRAME *F = &(Register[reg].channel[CurrentChannel].frame[0]);
int Count = Register[reg].frame_count;

while (Count--) {	/* Calc sum2 uses old sum, so calc sum2 FIRST!	*/
   F->sum2 += 2*offset*F->sum + offset*offset*F->n;
   F->sum  += offset * F->n;
   F++;
   }
}

return(OK);
}
/* END OffsetData */
/************************************************************************/
float HalveAt = 0.;			/* Point to start removing at	*/
int   HalveAbove = -1;			/* <=0: remove HalveAt and below*/
int   HalveAllData = -1;		/* <=0: no.			*/

/* FUNCTION SetupHalving */
/* FUNCTION HalveData */
void SetupHalving(void) {
char answer[80];

   if (! ((Status & BATCH_MODE) || (Status & COMMAND_MODE)))
      fprintf(stderr,
         "Halve (remove data) begining where? [%.2f]  > ", HalveAt);
   inputs(" > ", answer);
   if (answer[0] != NONE)
      HalveAt = (float) atof(answer);
   HalveAbove = query("Halve above (yes or no)?  > ");
   HalveAllData = query("Halve all data (yes or no)?  > ");
}

/* FUNCTION HalveData */
int HalveData(int reg) {
FRAME *F, *Begin;
int Count;

   if (ProofReg(reg) == FAIL)
      return(FAIL);
   F = &(Register[reg].channel[CurrentChannel].frame[0]) - 1;
   Begin = F + 1;

   Count = Register[reg].frame_count;

   if (HalveAllData <= 0) {			/* Don't halve all data	*/
      if (HalveAbove>0) {			/*   Halve above	*/
         while (Count--)
            if ((++F)->n)
	       if (MEan(F) >= HalveAt)
                  F->sum = F->sum2 = F->n = 0;
      } else					/*   Halve below	*/
         while (Count--)
            if ((++F)->n)
	       if (MEan(F) <= HalveAt)
                  F->sum = F->sum2 = F->n = 0;
   } else {					/* Do halve all data	*/
         while (Count--)
            if ((++F)->n)
               if ((HalveAbove>0) ? (MEan(F) >= HalveAt) : (MEan(F) <= HalveAt) )
		  DeleteFrame(reg, F - Begin);
   }
   return(OK);
} /* END HalveData */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * **/

/* FUNCTION RectifyData */
int RectifyData(int reg) {
FRAME *F;
int Count;

   if (ProofReg(reg) == FAIL)
      return(FAIL);
   F = &(Register[reg].channel[CurrentChannel].frame[0]) - 1;

   Count = Register[reg].frame_count;

   while (Count--)
      if ((++F)->n)
	 if (MEan(F) < 0)
             F->sum = - F->sum;
   return(OK);
} /* END RectifyData */

/************************************************************************/
/* FUNCTION DeleteFrame */
static void DeleteFrame(int reg, int frame) {
FRAME *F;				/* Pt at frames of data	*/
int i;
   for (i=0; i<MAX_CHANNELS; i++)
      if (RecordedChannel(i)) {
         F = &(Register[reg].channel[i].frame[frame]);
   	     F->sum = F->sum2 = F->n = 0;		/* Delete all	*/
         }
}
/************************************************************************/

float multiplier = -1.;

/* FUNCTION MultiplyData */
	 /* Multiply a single channel by a constant	*/
int MultiplyData(int reg) {
FRAME *F;
FRAME *End;
float factor = multiplier * multiplier - 1;
char answer[80];

if (ProofReg(reg) == FAIL)
   return(FAIL);
F   = Register[reg].channel[CurrentChannel].frame - 1;
End = Register[reg].channel[CurrentChannel].frame + Register[reg].frame_count;

if (!(Status & (BATCH_MODE|MACRO_MODE))) {
   if (!(Status & COMMAND_MODE)) 
      fprintf(stderr, "Enter new multiplier [%.2f]  > ", multiplier);
   inputs("", answer);
   if (answer[0] != NONE) {
      multiplier = (float) atof(answer);
      factor = multiplier * multiplier - 1;
      }
   }

if (multiplier == -1)			/* Special case ('factor' = 0)	*/
   while (++F < End)
       F->sum = -(F->sum);
else
   while (++F < End) {				/* Painful but accurate	*/
      F->sum2 += F->sum * F->sum / F->n * factor;
      F->sum  *= multiplier;
      }
return(OK);
}
/* END MultiplyData */
/************************************************************************/

/* FUNCTION DeleteData */
	 /* Delete all "CurrentChannel" data from a given interval	*/
int DeleteData(int reg, int Begin, int End) {
FRAME *F;

if (ProofReg(reg) == FAIL)
   return(FAIL);
if (SetBounds(reg,Begin,End,"DeleteData") == FAIL)
   return(FAIL);
F = &(Register[reg].channel[CurrentChannel].frame[begin]);
for ( ; begin++ < enddd; F++)
   F->n = F->sum = F->sum2 = 0;
return(OK);
}
/* END DeleteData */
/* ******************************************************************** */

/* FUNCTION DeleteUnitData */
	 /* Delete unit channels over a given interval	*/
int DeleteUnitData(int reg, int Begin, int End) {
FRAME *F;				/* Pt at frames of data	*/
int   *rm_spikes;

   if (!RecordedChannel(UNIT) || 
	SetBounds(reg,Begin,End,"DeleteUnitData") == FAIL)
      return(FAIL);

   rm_spikes =  Register[reg].spikes;
   F = &(Register[reg].channel[UNIT].frame[begin]);

   for ( ; begin++ < enddd; F++) {		/* Over interval:	*/
      F->n = F->sum = F->sum2 = 0;
      *(rm_spikes++) = 0;
      }
   return(OK);
}
/* END DeleteUnitData */
/* ******************************************************************** */

/* FUNCTION DeleteAllData */
	 /* Delete all channels over a given interval	*/
	 /* Could probably write more efficient version	*/
int DeleteAllData(int reg, int Begin, int End) {
int    j;				/* Count frames or chans*/
FRAME *F;				/* Pt at frames of data	*/
int i;

   if (SetBounds(reg,Begin,End,"DeleteAllData") == FAIL)
      return(FAIL);

   for (i=0; i<MAX_CHANNELS; i++)
      if (RecordedChannel(i)) {
         F = &(Register[reg].channel[i].frame[begin]);
	 for (j=begin; j<enddd; j++,F++)
   	     F->sum = F->sum2 = F->n = 0;		/* Delete all	*/
         }

   return(OK);
}
/* END DeleteAllData */
/* ******************************************************************** */

/* FUNCTION AvgSomeData */
	 /* Average values of single channel over particular interval	*/
float AvgSomeData(int reg, int Begin, int End) {
FRAME *F;
float  AvgSum = 0.;			/* Running sums		*/
int	AvgN   = 0 ;			/* Total points so far	*/
	 int    SaveBegin = Begin;		/* For err report	*/

if (SetBounds(reg,Begin,End,"AvgSomeData") == FAIL)
   return(FAIL);

F = &(Register[reg].channel[CurrentChannel].frame[begin]);
while (begin++ < enddd) {			/* Count down		*/
   AvgN    += F->n;				/* For current channel	*/
   AvgSum  += (F++)->sum;
   }
if (AvgN == 0) {
   char printstring[80];
   sprintf(printstring, "Cannot calc average (reg %d chan %d  %d to %d)",
	reg, TellCurrentChannel(), SaveBegin, End);
   vtprint(printstring);
   return(FAIL);
   }
if (! ((Status & BATCH_MODE) || (Status & COMMAND_MODE)))
   fprintf(stderr,"   Average = %5.2f\n", AvgSum/AvgN);
return(AvgSum/AvgN);
}
/* END AvgSomeData */
/* ******************************************************************** */

/* FUNCTION AvgWrappedData */
	 /* FOR PERIODIC DATA	*/
	 /* As above, but if Begin<0, wrap round end; similar for End	*/
float AvgWrappedData(int reg, int Begin, int End) {
FRAME *F;
float  AvgSum = 0.;			/* Running sums		*/
int	AvgN   = 0 ;			/* Total points so far	*/
int SaveBegin = Begin;				/* For err report	*/

if (ProofReg(reg) == FAIL)
   return((float)FAIL);
if (Register[reg].frame_count == 0 || Register[reg].trial_count == 0)
   return((float)FAIL);

End++;
if (Begin < 0) {
   F = &(Register[reg].channel[CurrentChannel].
			frame[Register[reg].frame_count + Begin - 1]);
   while (++Begin < 0) {			/* Get wrapped portion	*/
      AvgN    += F->n;				/* Notes:see AvgSomeData*/
      AvgSum  += F->sum;
      F++;
      }
   }

if (End > Register[reg].frame_count) {
   F = &(Register[reg].channel[CurrentChannel].frame[0]);
   while (End-- >= Register[reg].frame_count) {
      AvgN    += F->n;
      AvgSum  += F->sum;
      F++;
      }
   }

F = &(Register[reg].channel[CurrentChannel].frame[Begin]);
while (Begin++ <= End) {			/* Count down		*/
   AvgN    += F->n;				/* For current channel	*/
   AvgSum  += F->sum;
   F++;						/* Increment pointer	*/
   }

if (AvgN == 0) {
   char printstring[255];
   sprintf(printstring,
	"Cannot calc wrapped average (reg %d chan %d  %d to %d)",
	reg, TellCurrentChannel(), SaveBegin, End);
   vtprint(printstring);
   return((float)FAIL);
   }
fprintf(stderr,"\n   Average = %5.2f\n", AvgSum/AvgN);
return(AvgSum/AvgN);
}
/* END AvgWrappedData */
/* ******************************************************************** */

/* FUNCTION StatsSomeData */
	 /* Return Mean,SD,SE or N of data over interval		*/
	 /* Stat:   Mean or Avg,SD,SE,N					*/
	 /*  NOTE: SD and SE returned are average single point values!	*/
float StatsSomeData(int reg, int Begin, int End, char *Stat) {
FRAME *F;
float  SumMean = 0.;			/* Running sum of mean	*/
float  SumSD   = 0.;			/* Running sum of SD	*/
float  SumSE   = 0.;			/* Running sum of SE	*/
int	SumN    = 0 ;			/* Running sum of N	*/
	 int    Count   = 0 ;			/* Number of points	*/

if (SetBounds(reg,Begin,End,"StatsSomeData") == FAIL)
   return((float) FAIL);

F = &(Register[reg].channel[CurrentChannel].frame[begin]);

while (begin++ < enddd) {			/* Count down		*/
   if (F->n > 1) {				/* If n == 0, no mean	*/
      float sd = SD(F);				/* If n == 1, no SD/SE	*/
      SumMean += MEan(F);
      SumN    += F->n;
      SumSD   += sd;
      SumSE   += sd / ((float) sqrt((double) (F->n)));
      Count++;
      } 
   F++;						/* Increment pointer	*/
   }

if (SumN == 0.0)
   return((float)FAIL);

switch (*Stat) {
   case 'M': case 'm': case 'A': case 'a':
	return(SumMean/Count);

   case 'n': case 'N': case '#':
	return((float)SumN/Count);

   case 'S': case 's':
       switch (*(Stat+1)) {
	  case 'D': case 'd':
		return(SumSD/Count);
	  case 'E': case 'e':
		return(SumSE/Count);
	  default: ;
	  }
   default:  fprintf(stderr, "\nStatsSomedata(): Stat %c?\n", *Stat);
	     return((float)FAIL);
   }
}
/* END StatsSomeData */
/* ******************************************************************** */

/* FUNCTION FindPeakOfData */
	 /* FindPositivePeak:	* Test (>0) vs (<=0)	*/
int FindPeakOfData(int reg, int Begin, int End, int FindPositivePeak) {
FRAME *F;
FRAME *PeakFrame = 0;				/* When Max occured	*/
float Peak;					/* Absolute peak	*/

   if (ProofReg(reg) == FAIL)
      return(FAIL);
   if (SetBounds(reg,Begin,End, "FindPeakOfData") == FAIL)
      return(FAIL);
   F = &(Register[reg].channel[CurrentChannel].frame[enddd]);

   Peak = (FindPositivePeak>0) ? -10000.:10000.;

   while (enddd-- > begin)
      if ((--F)->n)
	 if ((FindPositivePeak>0) ? MEan(F)>Peak : MEan(F)<Peak) {
	    Peak = MEan(F);
	    PeakFrame = F;
	    }
   if (fabs(Peak)==10000.)
      return(FAIL);
   return(PeakFrame-Register[reg].channel[CurrentChannel].frame);
}
/* END FindPeakOfData */
/* ******************************************************************** */

/* FUNCTION FindFirstDataOver */
int FindFirstDataOver(int reg, float Value, int BeginAt) {
FRAME *F;

   if (ProofReg(reg) == FAIL)
      return(FAIL);
   if (SetBounds(reg,BeginAt,BeginAt+2,"FindFirstDataOver") == FAIL)	/* Fake the end	*/
      return(FAIL);
   F = &(Register[reg].channel[CurrentChannel].frame[begin]);

   while (begin++ < Register[reg].frame_count)
      if ((++F)->n)
	 if (MEan(F) > Value)
	    return(begin);
   return(FAIL);
}
/* END FindFirstDataOver */
/* ******************************************************************** */

/* FUNCTION FindFirstDataUnder */
int FindFirstDataUnder(int reg, float Value, int BeginAt) {
FRAME *F;

   if (ProofReg(reg) == FAIL)
      return(FAIL);
   if (SetBounds(reg,BeginAt,BeginAt+2,"FindFirstDataUnder") == FAIL)	/* Fake the end	*/
      return(FAIL);
   F = &(Register[reg].channel[CurrentChannel].frame[begin]);

   while (begin++ < Register[reg].frame_count)
      if ((++F)->n)
	 if (MEan(F) < Value)
	    return(begin);
   return(FAIL);
}
/* END FindFirstDataUnder */
/* ******************************************************************** */

/* FUNCTION FindDataZeroCrossings */
int FindDataZeroCrossings(int reg, int BeginAt, int EndAt) {
   FRAME *F;
   int count = 0;
   #define MAX_CROSSINGS	200
   static int ZeroCrossing[MAX_CROSSINGS + 2];
   float CRITERION = 1.0;
   int BeginSave;

   if (ProofReg(reg) == FAIL)
      return(FAIL);
   if (SetBounds(reg,BeginAt,EndAt,"FindFirstDataOver") == FAIL)
      return(FAIL);
   F = &(Register[reg].channel[CurrentChannel].frame[begin]);
   BeginSave = begin;

   TRY_AGAIN:				/* Make CRITERION smaller	*/

   while (begin++ < enddd)
      if ((++F)->n)
	 if (fabs(MEan(F)) < CRITERION) {
	    ZeroCrossing[count++] = begin;
	    if (count > MAX_CROSSINGS) {
	       if (CRITERION >= 0.001) {
		   CRITERION /= 5;
		   begin = BeginSave;
   		   F = &(Register[reg].channel[CurrentChannel].frame[begin]);
		   count = 0;
	           goto TRY_AGAIN;
	           }
	       fprintf(stderr, "Too many crossings! -- Fix & try again!\n");
	       return(FAIL);
	       }
	    }

   if (count == 0) {
      fprintf(stderr, "No crossings! -- Fix & try again!\n");
      return(FAIL);
      }
   
   if (count % 2 == 1)					/* Odd number?	*/
      return(ZeroCrossing[(count-1)/2]);	/* Median cross time	*/
   else						/* Even -mean of medians*/
      return((ZeroCrossing[count/2] + ZeroCrossing[count/2 - 1])/2);
}
/* END FindDataZeroCrossings */
/* ******************************************************************** */

/* FUNCTION DataFlag */
	 /* Bit map of where data is; used by file.c to write files	*/
int DataFlag(int reg) {
int flag = Header_dflag();				/* Bits 0-8	*/
int i, j;

for (i=MAX_RECORDED_CHANNELS; i<MAX_CHANNELS; i++) {
   FRAME *F = &(Register[reg].channel[i].frame[0]);
   for (j=0; j<MAX_FRAMES-20; j+=12, F+=12)
        if (F->n) {
	   flag |= (1<<i);
	   break;
	   }
   }
return(flag);
}
/* ******************************************************************** */

/* FUNCTION SetBounds */
	 /* Set and test file-global variables begin and enddd	*/
static int SetBounds(int reg,int start,int stop, char *functionname) {
if (ProofReg(reg) == FAIL)
   return(FAIL);
if (Register[reg].frame_count == 0 || Register[reg].trial_count == 0)
   return(FAIL);

begin = start-1;		/* Make it inclusive	*/
enddd = stop;	

if (begin < 0)
   begin = 0;

#ifdef NOT_NECESSARY
if (functionname[0]=='D' && functionname[1]=='e') {
   if (enddd > MAX_FRAMES)	/* Deletion function: go to very end!	*/
       enddd = MAX_FRAMES;
   if (begin > MAX_FRAMES)
      return(FAIL);
 } else 			/* Other than Delete():find end of data	*/
#endif
   {
   if (enddd >= Register[reg].frame_count)
      enddd = Register[reg].frame_count + 1;		/* +1:Insurance	*/
   if (enddd > MAX_FRAMES)				/* Whoops!	*/
       enddd = MAX_FRAMES;
   if (begin >= Register[reg].frame_count) {
      fprintf(stderr,"%s: Begin (%d) after trial ends (%d)!  (reg %d)\n",
  		 functionname, begin, Register[reg].frame_count, reg);
      return(FAIL);
      }
   }

if (begin > enddd || enddd < 0) {
   fprintf(stderr,"%s: Bad bounds: %d to %d (reg %d)\n",
		 functionname, begin, enddd, reg);
   return(FAIL);
   }
return(OK);
}
/* ******************************************************************** */
