/* FILE calib.c */
     /* Reverse calibrations */

#include "defs.h"

#include <string.h>
#include <ctype.h>
#include <unistd.h>
/* ******************************************************************** */

/* PUBLIC: 
 *
 *  Init_CalibFile
 *  Read_CalibFile
 *  Do_Calibration
 *
 * PRIVATE:
 *
 */
/* ******************************************************************** */

# define DEGtoAD(d)	( (d) * 20 )	

 static int CalibFileNumber = -1;		/* Last file read	*/
 static char *CalibFilePath;
 static FILE *CalibFile;

 static int CalibType = -1;			/* Unset	*/

#define CHAIR_EYE_CROSSTALK	Calibs[0]
#define EYE_LEFT_BOOST		Calibs[1]
#define EYE_DOWN_BOOST		Calibs[2]
#define HORIZONTAL_OFFSET	Calibs[3]
#define VERTICAL_OFFSET		Calibs[4]

/* Calib type 1: some are set.  2+3: none are used */
#define CHAIR_RIGHT_CROSSTALK	  ((CalibType==1)? Calibs[5] : 0)
#define EYE_CROSSTALK		  ((CalibType==1)? Calibs[6] : 0)
#define DOWNWARD_CROSSTALK	  ((CalibType==1)? Calibs[7] : 0)
#define DOWNWARD_CROSSTALK_OFFSET ((CalibType==1)? Calibs[8] : 0) /*in deg*/

/* Calib type 2: used.  1: not used */
#define H_GAIN	((CalibType>1)? Calibs[5] : 0)
#define V_GAIN	((CalibType>1)? Calibs[6] : 0)

/* Calib type 3: used.  1+2: not used */
#define H_ON_V_CROSSTALK	((CalibType==3)? Calibs[7] : 0)
#define V_ON_H_CROSSTALK	((CalibType==3)? Calibs[8] : 0)

#define MAX_CALIBS		  9
 static int Calibs[MAX_CALIBS];

/* ******************************************************************** */

/* FUNCTION Init_CalibFile */
	 /* Init calib file number */
void Init_CalibFile(char *inputfilename) {
	CalibFileNumber = -1;
	CalibFilePath = inputfilename;
	}
/* ******************************************************************** */

#define Get_Next_Line						\
	   while (fgets(Line, 80, CalibFile) != NULL)		\
	      /* if ((Line[0] != '#') && (Line[1] != '\n')) */	\
	      if (isdigit(Line[0]) || Line[0]=='-')		\
	         break;
/* ******************************************************************** */
	
/* FUNCTION Read_CalibFile */
	 /* Read in the specified calib file */
void Read_CalibFile(int number) {
	if (CalibFileNumber != number) {
	   char FileName[105];
	   char Line[81];
	   int i;

	   if (number == -999) {		/* Already calibrated	*/
	      CalibFileNumber = number;
	      return;
	      }

	   if (number <= 0) {			/* No calib file ?	*/
	      fprintf(stderr, "  uncalibrated\n");
	      CalibFileNumber = number;
	      return;
	      }

	   CalibFileNumber = number;
	   sprintf(FileName, "calib.%d", number);
	   CalibFile = fopen(FileName, "r");

	   if (CalibFile == NULL) {	/* Try same directory as file	*/
	      int i;
	      for (i=strlen(CalibFilePath); i>0; i--)	/* Find slash	*/
	         if (*(CalibFilePath+i) == '/') {
		    strncpy(FileName, CalibFilePath, (size_t) i);
		    FileName[i] = 0;
	   	    sprintf(FileName, "%s/calib.%d", FileName, number);
	   	    CalibFile = fopen(FileName, "r");
		    break;		/* Tried your best, now go away	*/
		    }
	      }

	   if (CalibFile == NULL) {
	      fprintf(stderr, "Cannot find calibration file '%s'\n", FileName);
	      return;
	      }


	   Get_Next_Line;		/* Calibration type	*/
	   CalibType = atoi(Line);	/* Ignore it for now	*/
	   
	   for (i=0; i<MAX_CALIBS; i++)
	       Calibs[i] = 0;

	   for (i=0; i<MAX_CALIBS; i++) {
	       Get_Next_Line;
	       Calibs[i] = atoi(Line);
	       if (feof(CalibFile))
	          break;
	       }

	   fclose(CalibFile);
	   }
	}
/* ******************************************************************** */

/* FUNCTION Do_Calibration */
	 /* Duplicate the calibration done by reach */
void Do_Calibration(float *h_eye, float *v_eye) {
		/* Has Chuck's box run out of offset?		*/
	*h_eye += HORIZONTAL_OFFSET/20.;	/* ADtoDEG	*/
	*v_eye += VERTICAL_OFFSET/20.;		/* ADtoDEG	*/

	/* Leftward eye calibration	*/
	if (*h_eye < 0.)
	   *h_eye += *h_eye * EYE_LEFT_BOOST / 100.;

	/* Downward eye calibration	*/
	if (*v_eye < 0)
	   *v_eye += *v_eye * EYE_DOWN_BOOST / 100.;

	/* Crosstalk (required where there's a large vertical offset)	*/
	*v_eye += *v_eye * EYE_CROSSTALK/100.;

   /* Adjust for horiz shift proportional to vertical AND horizontal, centered
    * at vert = 0 and horiz = OFFSET */
	if (*v_eye < 0)
	   *h_eye +=  
		   (*h_eye - DEGtoAD(DOWNWARD_CROSSTALK_OFFSET)) *
		    -*v_eye * DOWNWARD_CROSSTALK/10000.; 
	/* if at (0,-20) and values are -10 [deg] & 10 [%], then add to H_EYE:
	 *    10 deg * -20/20 * 10/100 = 10 * -1 * .1 = -1 deg	*/

	if (H_GAIN)
	   *h_eye *= H_GAIN/100.;
	if (V_GAIN)
	   *v_eye *= V_GAIN/100.;

	if (H_ON_V_CROSSTALK)
	   *v_eye += *h_eye * H_ON_V_CROSSTALK/100.;
	if (V_ON_H_CROSSTALK)
	   *h_eye += *v_eye * V_ON_H_CROSSTALK/100.;
	}
/* ******************************************************************** */
