#include "plot.h"
#include <stdlib.h>

/* init.c
 * 
 * This is where we open and select windows in response to calls to
 * device().  For simplicity we keep a bunch of other gnarly X code
 * here (e.g. event handling), leaving only the business of actually
 * drawing to most of the usual routines.
 */

#define X_WINDOW_AT 589			/* LHS  8-17-96		*/
#define Y_WINDOW_AT  -5			/* LHS  8-17-96		*/


/* A global linked list of all of our windows */
XplWindow 	       *xpl_windows = NULL;

/* A pointer to the current window */
XplWindow	       *xpl_window = NULL;


/* Which events are we interested in?  This is defined so as to match the 
 * XSelectEvents and XCheckEvent calls. */

unsigned long		xpl_event_mask = ExposureMask | ButtonPressMask 
				         | StructureNotifyMask;


/* The size of the window to open if it's under our control */
static int	        default_width  = 560;
static int	      	default_height = 420;

/* The display to use if it's under our control */
static Display	       *default_display = NULL;


#define min(a, b) ((a)<(b)?(a):(b))



/* xpl_new_window:
 * 
 * Create a new window.  If display and/or window are provided we will use
 * them.  Otherwise we open new ones.  We leave xpl_window pointing to the 
 * new structure. */

void xpl_new_window (Display *d, Window w, char *name)
{
  /* Make a new structure */
  xpl_window = malloc (sizeof (XplWindow));
  
  /* Add it to the linked list. */
  xpl_window->next = xpl_windows;
  xpl_windows = xpl_window;

  /* Set some stuff */
  xpl_window->closed = 0;

  /* Set the display.  If d is NULL then use the default. */
  if (d) {
    xpl_window->dpy = d;
  } else {
    if (default_display == NULL) {
      default_display = XOpenDisplay (NULL);
    }
    xpl_window->dpy = default_display;
  } 


  /* Set the window.  If w is NULL then open one. */
  if (w) {
    xpl_window->win = w;
    xpl_window->external_win = 1;
  } else {
    XSetWindowAttributes  attributes;
    long		  attribute_mask = 0;

    

    xpl_window->win = XCreateSimpleWindow
      (xpl_window->dpy,
       RootWindow (xpl_window->dpy, DefaultScreen(xpl_window->dpy)),
       X_WINDOW_AT, Y_WINDOW_AT, default_width, default_height,
       1,
       BlackPixel (xpl_window->dpy, DefaultScreen(xpl_window->dpy)),
       WhitePixel (xpl_window->dpy, DefaultScreen(xpl_window->dpy))
       );

    xpl_window->foreground = 
      BlackPixel (xpl_window->dpy, DefaultScreen(xpl_window->dpy));

    xpl_window->background = 
      WhitePixel (xpl_window->dpy, DefaultScreen(xpl_window->dpy));

    attributes.backing_store = Always;
    attribute_mask |= CWBackingStore;
    XChangeWindowAttributes 
      (xpl_window->dpy, xpl_window->win, 
       attribute_mask, &attributes);
    xpl_window->external_win = 0;

    {		/* LHS 8-18-96: demand a specific location	*/
    XSizeHints            *size_hints = XAllocSizeHints();

    size_hints->flags = PPosition | PSize;
    XSetWMNormalHints(
	xpl_window->dpy, xpl_window->win, size_hints);
    }

  } 

  /* Set the name of the window (for display in the title bar) */
  if (name) {
    XStoreName (xpl_window->dpy, xpl_window->win, name);
  }

  /* Get the window size (in case we didn't get what we wanted, or
   * in case the window was given to us) */
  {
    XWindowAttributes attr;
    XGetWindowAttributes (xpl_window->dpy, xpl_window->win, &attr);
    xpl_window->width  = attr.width;
    xpl_window->height = attr.height;
  }

  /* Create a GC */
  {
    XGCValues           	values;
    long                	value_mask = 0;
    xpl_window->gc = XCreateGC (xpl_window->dpy, xpl_window->win, 
				value_mask, &values);
  }

  /* Create a pixmap to mirror the window (and handle expose events) */
  {
    XWindowAttributes attr;
    XGetWindowAttributes  (xpl_window->dpy, xpl_window->win, &attr);
    xpl_window->pix = XCreatePixmap(xpl_window->dpy, xpl_window->win, 
				    xpl_window->width, xpl_window->height,
				    attr.depth);
  }

  /* Mask input events on the window */
  XSelectInput (xpl_window->dpy, xpl_window->win, xpl_event_mask);

  
  /* Map the window */
  XMapWindow   (xpl_window->dpy, xpl_window->win);
  xpl_update();

}


/* xpl_select_window:
 * 
 * Send subsequent output to the selected window.  This is simply achieved 
 * by setting xpl_window to equal the incoming pointer.  But we need to
 * make sure that the window is valid.  We could also try and raise the window
 * but let's leave that for later.
 */

int xpl_select_window (XplWindow *win)
{
  XplWindow *ptr;

  /* Traverse the list of xpl_windows to see if win is still active.
     If so, select it. */

  for (ptr = xpl_windows; ptr; ptr = ptr->next) {
    if (ptr == win) {
      xpl_window = win;
      XRaiseWindow (xpl_window->dpy, xpl_window->win);
      break;
    }
  }

  /* If we selected the window xpl_window will equal win */
  if (xpl_window != win)
    return -1;
  else
    return 0;
}


/*
 * xpl_unlink_window:
 *
 * remove the specified window from the linked list.
 */

void xpl_unlink_window (XplWindow *win)
{
  XplWindow *ptr;

  if (win == xpl_windows) 
    xpl_windows = win->next;
  else {
    for (ptr = xpl_windows; ptr->next && ptr->next != win; ptr = ptr->next)
      ;
    if (ptr->next)
      ptr->next = win->next;
  }

  win->next = NULL;
}


/* xpl_destroy_window:
 * 
 * Destroy the specified window.  (If the XWindow was passed to us
 * from outside we don't actually call XDestroyWindow on it). 
 */

void xpl_destroy_window (XplWindow *win)
{
  XFreePixmap (win->dpy, win->pix);
  XFreeGC     (win->dpy, win->gc);
  if (!win->external_win) {
    XDestroyWindow (win->dpy, win->win);
    fflush (NULL);
  }
  free (win);
}


/* xpl_update:
 * 
 * Handle incoming events on all the events in the xpl_windows list.
 */

void xpl_update()
{
  XEvent ev;
  XplWindow *win;

  for (win = xpl_windows; win; win = win->next) {
    while (XCheckWindowEvent (win->dpy, win->win, xpl_event_mask, &ev)) {
      switch (ev.type) {

      case Expose:
	if ( ev.xexpose.count == 0 )
	  XCopyArea (win->dpy, win->pix, win->win, win->gc, 
		     0, 0, win->width, win->height, 
		     0, 0);
	break;

      case ConfigureNotify: {
	Pixmap newpix;
	XWindowAttributes attr;
	XGetWindowAttributes  (win->dpy, win->win, &attr);

	newpix = XCreatePixmap(win->dpy, win->win, 
			       ev.xconfigure.width, ev.xconfigure.height, 
			       attr.depth);
	XCopyArea (win->dpy, win->pix, newpix, win->gc,
		   0, 0, 
		   min(win->width, ev.xconfigure.width), 
		   min(win->height,ev.xconfigure.height),
		   0, 0);
	XFreePixmap (win->dpy, win->pix);
	win->pix = newpix;

	win->width  = ev.xconfigure.width;
	win->height = ev.xconfigure.height;

	/* rescale */
	spaceX (0, 0, 0, 0);
	break;
      }

      case ButtonPress:
	if ( win->closed ) {
	  xpl_destroy_window (win);
	  exit (0);
	}
      }
    }
  }
}
