/* FILE ps.c */
     /* Postscript driver */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "plot.h"

#define DEBUG		0

#define Y_PAGE	11.0
#define X_PAGE	 8.5
#define EDGE	 0.5				/* Margin on each side	*/

#define	CENTER		1

static FILE *out;

static int stroked;				/* Stroked path?	*/
static int orientation;				/* Takes 1 of 3 values	*/
#define NEW_PLOT	2
#define LANDSCAPE	1
#define PORTRAIT	0

#define MULTIPLE_PAGES	1		/* Implement a hack!		*/
static int page = 1;			/* Used with multi-page hack	*/

/* PRIVATE FUNCTIONS */
static void Set_font();
static void Stroke();
/* ********************************************************************	*/
/* ********************************************************************	*/

/* FUNCTION openplID */
	 /* Write postscript header, set defaults, save graphics state	*/
int openplID(void) {
   orientation = NEW_PLOT;			/* Flag to space()	*/
   out = stdout;
   stroked = 1;					/* Initialize		*/

#  if MULTIPLE_PAGES
   if (page == 1) {
      fputs("%!PS-Adobe-2.0\n", out);
      fputs("%%Pages: 1000\n", out);		/* Upper bound!		*/
      }
   fprintf(out, "%%%%Page: %d %d\n", page, page);
   page++;
#  else
   fputs("%!\n", out);
   if (page > 1)
      fprintf(stderr, "Must turn on MULTIPLE_PAGES in ps.c\n");
#  endif
   fputs("gsave\n\n", out);
   fputs("/M { moveto } bind def\n", out);
   fputs("/L { lineto } bind def\n", out);
   fputs("/R { rlineto } bind def\n", out);
   fputs("/H { 0 rlineto } bind def\n", out);
   fputs("/V { 0 exch rlineto } bind def\n", out);
   fputs("/S { stroke } bind def\n", out);
   fputs("/lw { setlinewidth } bind def\n", out);
   fputs("/tr { translate } bind def\n", out);
   fputs("/sh { show } bind def\n", out);
   fputs("/sf { scalefont setfont } bind def\n", out);
   fputs("% Improve greys\n", out);
   fputs("106 45 {dup mul exch dup mul add 1.0 exch sub } setscreen\n", out);
   fputs("/DeviceRGB setcolorspace\n", out);	/* Versus DeviceGray	*/
   fputs("true setstrokeadjust\n", out);	/* Fine point (manual!)	*/
   Set_font();
   linewidthID(1);
   return(0);
   }
/* ********************************************************************	*/

/* FUNCTION closeplID */
	 /* Tell postscript to show page, then restore graphics state	*/
int closeplID(void) {
   fputs("S\n", out);				/* Stroke current path	*/
   fputs("showpage\n", out);
   fputs("grestore\n", out);
   return(0);
   }
/* ********************************************************************	*/

/* FUNCTION flushplID */
	 /* Extension */
int flushplID(void) {
   fflush(out);
   return(0);
   }
/* ********************************************************************	*/

/* FUNCTION bkgndID */
	 /* Extension */
int bkgndID(int a, int b, int c) {
   Stroke();
   return(0);
   }
/* ********************************************************************	*/

/* FUNCTION dotID */
	 /* Extension */
int dotID(int a, int b, int c, int d, char *e) {
   Stroke();
   return(0);
   }
/* ********************************************************************	*/

/* FUNCTION lineID */
	 /* Extension */
int lineID(int x1, int y1, int x2, int y2) {
   moveID(x1, y1);
   contID(x2, y2);
   Stroke();
   return(0);
   }
/* ********************************************************************	*/
/* ********************************************************************	*/

/* FUNCTION spaceID */
	 /* Set up rotation, scaling and offset per request */
int spaceID(int x0, int y0, int x1, int y1) {
   fputs("90 rotate   50  -550 tr .67 .60 scale\n", out);
   return(0);
   }
/* ********************************************************************	*/
/* ********************************************************************	*/
static int x_was;
static float y_was;

/* FUNCTION moveID */
	 /* Move without drawing line; first stroke previous line */
int moveID(int x, float y) {
	Stroke();				/* Sets stroked to 1	*/
	fprintf(out, "%d %.1f M\n", x_was=x, y_was=y);
        return(0);
	}
/* ********************************************************************	*/

/* FUNCTION contID */
	 /* Move and draw line */
int contID(int x, float y) {
	if (x == x_was) {
	   if (y != y_was)
	      fprintf(out, "%.1f V\n", y-y_was);
	   y_was = y;
	} else if (y == y_was) {
	   fprintf(out, "%d H\n", x-x_was);
	   x_was = x;
	} else {
	   fprintf(out, "%d %.1f R\n", x-x_was, y-y_was);
	   x_was = x;
	   y_was = y;
	   }
	/* fprintf(out, "%d %.1f L\n", x,y); */
	stroked = 0;
        return(0);
	}
/* ********************************************************************	*/

/* FUNCTION boxID */
	 /* Draw a box */
int boxID(int x0, int y0, int x1, int y1) {
	Stroke();
        fprintf(out, "%d %d %d %d rectstroke\n", x0, y0, x1-x0, y1-10);
        /* fprintf(out, "%d %d %d %d %d %d %d %d %d %d M L L L L S\n", */
	/* x0,y0, x0,y1, x1,y1, x1,y0, x0,y0); */
        return(0);
	}
/* ********************************************************************	*/

/* FUNCTION eraseID */
	 /* Clear page */
int eraseID(void) {
	fputs("clippath 1 setgray fill\n", out);
        fputs("/DeviceRGB setcolorspace\n", out);	/* Need:setgray	*/
        return(0);
	}

/* ********************************************************************	*/

/* FUNCTION arcID */
	 /* Draw a arc */
int arcID(int xc, int yc, int x0, int y0, int x1, int y1) {
	double radius = sqrt( (double)(xc-x0)*(xc-x0) + (yc-y0)*(yc-y0));
	double theta1 = asin( ((y0-yc) / radius) * (2 * 3.14159265 / 360));
	double theta2 = asin( ((y1-yc) / radius) * (2 * 3.14159265 / 360));
		
	Stroke();
	fprintf(out, "%d %d M     %d %d %f %f %f arc\n",
		0, y0,     xc,yc, radius, theta1, theta2);
	stroked = 0;
        return(0);
	}
/* ********************************************************************	*/

/* FUNCTION circIDle */
	 /* Draw a circle */
int circleID(int x, int y, int r) {
	Stroke();
	fprintf(out, "%d %d M   %d %d %d %d %d arc\n", x+r, y, x, y, r, 0, 360);
	stroked = 0;
        return(0);
	}
/* ********************************************************************	*/

/* FUNCTION pointID */
	 /* Draw a point (filled circle of unspecified radius) */
int pointID(int x, int y) {
	circleID(x,y,1);
	fillID(1);
        return(0);
	}
/* ********************************************************************	*/
/* ********************************************************************	*/
static int colors[3] = {0,0,0};

/* FUNCTION colorID */
	 /* color(0,0,0)==black,   color(0xffff,0xffff,0xffff)==white	*/
int colorID(int red, int green, int blue) {
	Stroke();
	if (colors[0]!=red || colors[1]!=green || colors[2]!=blue) {
	   if ( (red   == 0 || red   > 0xFF00) &&	/* 0 or ~1?	*/
		(green == 0 || green > 0xFF00) &&
		(blue  == 0 || blue  > 0xFF00)) {
	      if (red)   red   = 0xFFFF;		/* Round up	*/
	      if (blue)  blue  = 0xFFFF;
	      if (green) green = 0xFFFF;
	      fprintf(out, "%d %d %d setcolor\n",	/* Use int	*/
		(colors[0]=red) ?   1:0,  /* int: less chars than float	*/
		(colors[1]=green) ? 1:0,
		(colors[2]=blue) ?  1:0);
	   } else
	      fprintf(out, "%.2f %.2f %.2f setcolor\n",
		(colors[0]=red)  /(float)0xFFFF,
		(colors[1]=green)/(float)0xFFFF,
		(colors[2]=blue) /(float)0xFFFF);
	   }
        return(0);
	}
/* ********************************************************************	*/

/* FUNCTION linewidthID */
	 /* Extension.  Set to HALF requested. */
	 	/* Why are we doing that?	*/
int linewidthID(int width) {
	static int prev_width = 0;
	Stroke();
	if (prev_width != width)
	   fprintf(out, "%.1f lw\n", (prev_width=width)/2.);
        return(0);
	}
/* ********************************************************************	*/

int linemodID(char *s) {
	static char prev_line;		/* Use 4th char (is unique)	*/

	Stroke();
	if (*(s+3) == prev_line)
	   return(0);

	switch (*(s+3)) {		/* Check the 4th letter:	*/
	 case 'i':				/* soli*d	*/
		fputs("[] 0 setdash\n", out);
		break;
		/* OK */
	 case 'g':				/* long*dashed	*/
		fputs("[25 10] 0 setdash\n", out);
		break;
		/* 20 10 */
	 case 'c':				/* disc*onected	*/
		fputs("[10 10] 0 setdash\n", out);
		/* 20 20 */
		break;
	 case 'd':				/* dotd*ashed	*/
		fputs("[16 9 2 9] 17 setdash\n", out);
		break;
	 case 't':				/* dott*ed	*/
		fputs("[2 10] 0 setdash\n", out);
		/* 3 9 */
		break;
	 case 'r':				/* shor*tdashed	*/
		fputs("[20 20] 0 setdash\n", out);
		/* 10 10 */
		break;
	 default:
	   fprintf(stderr, "Unrecognizable line pattern <%s>\n", s);
	 }
	 /* Last #: phase of dots; brackets: white, black */

	prev_line = *(s+3);			/* Set to 4th char	*/
	return(0);
	}
/* ********************************************************************	*/

/* FUNCTION fillID */
	 /* 0:no fill  1: black fill   0xffff: white fill	*/
	 /* Instead of using black level, use color saturation:	*/
	 /*   1:use current color   0x000f: VERY light current color	*/
	 /*  FILL LEVELS OTHER THAN 0 and 1 ARE NOT IMPLEMENTED!	*/
	 /* color(0,0,0)==black,   color(0xffff,0xffff,0xffff)==white	*/
int fillID(int level) {
	if (level == 0)				/* NO fill		*/
	   fputs("S\n", out);
	else if (level == 1)			/* Current color fill	*/
	   fprintf(out, "fill\n");
	else			/* Diluted fill (eg, red --> pink)	*/
	   fprintf(out, "fill\n");		/* NOT IMPLEMENTED	*/
	stroked = 1;
	return(0);
	}
/* ********************************************************************	*/
/* ********************************************************************	*/

/* FUNCTION Stroke */
static void Stroke() {
     if (! stroked) {
	fputs("S\n", out);
	stroked = 1;
	}
     }
/* ********************************************************************	*/
/* ********************************************************************	*/

/* SUB-FILE font */
     /* Deal with fonts, strings */
/* ********************************************************************	*/

/* FUNCTION labelID */
	 /* Print a text string on the screen */
int labelID(char *s) {
	fprintf(out, "(%s) sh\n", s);
	stroked = 1;
	return(0);
	}
/* ********************************************************************	*/

static int text_rotate = 0;			/* Degrees of text rotation	*/
static int font_size    = 35.;
static char font_name[80] = "Helvetica";

/* FUNCTION Set_font */
	 /* Tell postscript to install new font(size)	*/
	 /* Correct so font size is independent of page scaling	*/
static void Set_font() {
  	fprintf(out, "/%s findfont %d sf\n", font_name, font_size);
   	}
/* ********************************************************************	*/

/* FUNCTION fontsizeID */
	 /* Set font size to size*/
int fontsizeID(int size) {
  if (font_size != 2 * size) {
     font_size = 2 * size;
     Set_font();
     }
  return(0);
  }
/* ********************************************************************	*/

/* FUNCTION fontnameID */
	 /* Postscript wants cap'd names; plot lib doesn't. Take either	*/
int fontnameID (char *s) {
  char *dash;
  if (*s >= 'a' && *s <= 'z')			/* Capitalize 1st letter*/
     *s = *s - 'a' + 'A';
  dash = strchr(s, '-');				/* Find a dash	*/
  if (dash++ != NULL)
     if (*dash >= 'a' && *dash <= 'z') 
         *dash = *dash - 'a' + 'A';			/*  Cap letter!	*/
  if (strcmp(font_name, s) != 0) {			/* Change?	*/
     strcpy(font_name, s);
     Set_font();
     }
  return(0);
  }
/* ********************************************************************	*/

/* FUNCTION rotateID */
	 /* Set text rotation angle.  Note: w and h are ignored! */
int rotateID (int w, int h, int angle) {
  text_rotate = angle;
  return(0);
  }
/* ********************************************************************	*/

/* FUNCTION alabelID */
	 /* label with x_justify l(left)/r/c; y_justify b(bottom)/c/t */
int alabelID (int x_justify, int y_justify, char *s) {

  if (text_rotate)
     fprintf(out, "%d rotate\n", text_rotate);

  if (x_justify != 'l' || y_justify != 'b') {
     switch (y_justify) {
	/* NEED TO USE charpath (with bool == FALSE, I think) then pathbbox */
    	case 'b': /* bottom */
	   /* fputs("pop ", out);			/* Pop y size off stack	*/
      	   break;
    	case 'c': /* centered */
	   /* fputs("2 div 0 exch rmoveto ", out); /* Move up by 1/2	*/
      	   break;
    	case 't': /* top */
	   /* fputs("0 exch rmoveto ", out); 	/* Move up full height	*/
      	   break;
    	}
     switch (x_justify) {
    	case 'c': 				    /* centered		*/
	   fprintf(out, "(%s) stringwidth\n", s);   /* x,y size on stack*/
	   fputs("pop 2 div neg 0 rmoveto ", out);  /* Move to left	*/
      	   break;
    	case 'r': 				   /* right justified	*/
	   fprintf(out, "(%s) stringwidth\n", s);  /* x,y size on stack	*/
	   fputs("exch neg exch rmoveto ", out);	   /* Left full width	*/
      	   break;
    	case 'l': /* left justified */
      	   break;
    	}
     }

  labelID(s);
  if (text_rotate)
     fprintf(out, "%d rotate\n", -text_rotate);		/* Restore	*/
  return(0);
  }
/* ********************************************************************	*/
/* ********************************************************************	*/
