/* plot2ps, a utility for converting Unix plot files into postscript.
   Copyright (C) 1989 Free Software Foundation, Inc.

Plot2ps is distributed in the hope that it will be useful, but WITHOUT ANY
WARRANTY.  No author or distributor accepts responsibility to anyone for the
consequences of using it or for whether it serves any particular purpose or
works at all, unless he says so in writing.  Refer to the GNU General Public
License for full details.

Everyone is granted permission to copy, modify and redistribute plot2ps, but
only under the conditions described in the GNU General Public License.  A copy
of this license is supposed to have been given to you along with plot2ps so
you can know your rights and responsibilities.  It should be in a file named
COPYING.  Among other things, the copyright notice and this notice must be
preserved on all copies.  */

#include "plot.h"

/* CURRENT_FONT is a string containing the name of the current font.  Since
   postscript symbols must generally be less that 256 characters long, we can
   reasonably assume that the name will fit in a string this long. */

char* IDcurrent_font = 0;

/* DEFAULT_FONT is a string containing the name of the default font.
   This string should be a valid match for one of the font names.
   It is allocated and set in openpl(). */

char* default_font = 0;

/* FIND_FONT takes a string argument S containing a font name and returns the
   index of the font structure FONT_INFO corresponding to that name. If no match
   is found for the font name, an error message is printed and the index of the
   default font is returned. */

int find_font(s)
     char *s;
{
  int font_index;
  for (font_index=0; font_index<NO_OF_FONTS; font_index++)
    {
      if (strcmp (font_info[font_index].name, s) == 0)
	return font_index;
    }
  fprintf (stderr, "Unable to find font `%s'. Using default font `%s'.\n",
	   IDcurrent_font, default_font);
  font_index = find_font (default_font);

  return font_index;
};

/* STRING_WIDTH is takes a string argument and returns the width of the
   string in font coordinates (1/1000 points). */

int
string_width (s)
     char *s;
{
  int indx = 0;
  int width = 0;
  int font_index;
  
  font_index = find_font (IDcurrent_font);

  for (indx=0; s[indx]!='\0'; indx++)
    width += font_info[font_index].width[ s[indx]];

  return width;
}

/* TEXT_ROTATION is the angle in degress counterclockwise from the
   horizontal for rotation of labels. */

int IDtext_rotation=0;

/* FONT_SIZE is the font size in printer's points (-f option). */

double IDfont_size = 14.;

/* FONT_SCALING takes into account the font scaling by idraw.  Version 2.4
   of idraw scales the whole drawing down by a factor of SCALING, we must
   scale the fonts back up by 1.25 so that a 14 point fonts appears as
   a 14 point font on the hardcopy. Version 2.5 of idraw does not seem
   to need this adjustment, but it is left here for compatibility. */

double font_scaling = 1.;

/* ALABEL takes three arguments X_JUSTIFY, Y_JUSTIFY, and S and places
   the label S according to the x and y axis adjustments specified in
   X_JUSTIFY and Y_JUSTIFY respectively.  X_JUSTIFY is a character
   containing either l, c, or r for left, center or right justified with
   respect to the current x coordinate.  Y_JUSTIFY is a character
   containing either b, c, or t for placing the bottom center or top of
   the label even with the current y coordinate. S is a string containing
   the label. The current point is moved to follow the end of the text. */

int
alabelID (x_justify, y_justify, s)
     int x_justify, y_justify;
     char *s;
{
  int i;
  int font_index;
  double width;
  double x_char_offset = 0., y_char_offset = 0.;

  font_index = find_font (IDcurrent_font);
  draw_line ();
  fputs ("Begin %I Text\n", PSoutfile);
  fprintf (PSoutfile, "%%I cfg Black\n%g %g %g SetCFg\n",
	  fgcolor_red, fgcolor_green, fgcolor_blue);
  fprintf (PSoutfile, "%%I f *%s*-%d-*\n", font_info[font_index].X_name, (int) (IDfont_size));
  fprintf (PSoutfile, "/%s %g SetF\n", font_info[font_index].ps_name,
	  IDfont_size * font_scaling);
  fputs ("%I t\n[ ", PSoutfile);
  for (i=0; i<4; i++ )
    fprintf (PSoutfile, "%g ", text_transformation_matrix[i]);

  /* first we figure out the offsets, then apply them depending on rotation. */

  /* width in printer's points */
  width = string_width (s) * IDfont_size / 1000.;
  switch (x_justify)
    {
    case 'l': /* left justified */
      x_char_offset = 0.0;
      break;

    case 'c': /* centered */
      x_char_offset = -0.5;
      break;

    case 'r': /* right justified */
      x_char_offset = -1.0;
      break;
    }

  switch (y_justify)
    {
    case 'b': /* bottom */
      y_char_offset = 1.0;
      break;

    case 'c': /* centered */
      y_char_offset = 0.5;
      break;

    case 't': /* top */
      y_char_offset = 0.0;
      break;
    }

  if (IDtext_rotation == 90 )
    {
      fprintf (PSoutfile, "%g ", IDx_output_min +
	      IDlast_x/IDscaleup - IDfont_size * font_scaling * y_char_offset);
      fprintf (PSoutfile, "%g ", IDy_output_min +
	      IDLast_y/IDscaleup + width * font_scaling * x_char_offset);
      set_range (IDlast_x/IDscaleup - IDfont_size * font_scaling * y_char_offset,
		 IDLast_y/IDscaleup + width * font_scaling * x_char_offset);
      set_range (IDlast_x/IDscaleup - IDfont_size * font_scaling *(y_char_offset-1.2),
		 IDLast_y/IDscaleup + width * font_scaling * (x_char_offset+1.0));
      IDLast_y += width * IDscaleup;
    }
  else
    {
      fprintf (PSoutfile, "%g ", IDx_output_min +
	      IDlast_x/IDscaleup + width * font_scaling * x_char_offset);
      fprintf (PSoutfile, "%g ", IDy_output_min +
	      IDLast_y/IDscaleup + IDfont_size * font_scaling * y_char_offset);
      set_range (IDlast_x/IDscaleup + width * font_scaling * x_char_offset,
		 IDLast_y/IDscaleup + IDfont_size * font_scaling*(y_char_offset-1.2));
      set_range (IDlast_x/IDscaleup + width * font_scaling * (x_char_offset+1.0),
		 IDLast_y/IDscaleup + IDfont_size * font_scaling*(y_char_offset+.0));
      IDlast_x += width * IDscaleup;
    }

  fputs (" ] concat\n\
%I\n\
[\n\
(", PSoutfile);
  for (i=0; s[i]!='\0'; i++)
    {
      switch (s[i])
	{
	case '(':
	case ')':
	case '\\':
	  fputc ((char) '\\', PSoutfile);
	default:
	  fputc (s[i], PSoutfile);
	}
    }
  fprintf (PSoutfile, ")\n\
] Text\n\
End\n\
\n");
  return 0;
}

struct font_info_struct font_info[NO_OF_FONTS] = {
{
  "courier-bold",
  "courier-bold-r",
  "Courier-Bold",
 {0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 0, 600, 600, 600, 600, 600, 600, 600, 600,
 600, 600, 600, 600, 600, 600, 600, 600, 600, 600,
 600, 600, 600, 600, 600, 600, 600, 600, 600, 600,
 600, 600, 600, 600, 600, 600, 600, 600, 600, 600,
 600, 600, 600, 600, 600, 600, 600, 600, 600, 600,
 600, 600, 600, 600, 600, 600, 600, 600, 600, 600,
 600, 600, 600, 600, 600, 600, 600, 600, 600, 600,
 600, 600, 600, 600, 600, 600, 600, 600, 600, 600,
 600, 600, 600, 600, 600, 600, 600, 600, 600, 600,
 600, 600, 600, 600, 600, 600, 600, 0, 0, 0,
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 600, 600, 600, 600, 600, 600, 600, 600, 600,
 600, 600, 600, 600, 0, 0, 0, 600, 600, 600,
 600, 0, 600, 600, 600, 600, 600, 600, 600, 0,
 0, 600, 0, 600, 600, 600, 600, 600, 600, 600,
 600, 0, 600, 600, 0, 600, 600, 600, 600, 0,
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 0, 0, 0, 0, 0, 0, 600, 0, 0,
 0, 0, 600, 600, 0, 600, 0, 0, 0, 0,
 0, 0, 0, 0, 0, 600, 0, 0, 600, 600,
 0, 600, 0, 0, 0, 0}},
{
  "courier-boldoblique",
  "courier-bold-o",
  "Courier-BoldOblique",
 {0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 0, 600, 600, 600, 600, 600, 600, 600, 600,
 600, 600, 600, 600, 600, 600, 600, 600, 600, 600,
 600, 600, 600, 600, 600, 600, 600, 600, 600, 600,
 600, 600, 600, 600, 600, 600, 600, 600, 600, 600,
 600, 600, 600, 600, 600, 600, 600, 600, 600, 600,
 600, 600, 600, 600, 600, 600, 600, 600, 600, 600,
 600, 600, 600, 600, 600, 600, 600, 600, 600, 600,
 600, 600, 600, 600, 600, 600, 600, 600, 600, 600,
 600, 600, 600, 600, 600, 600, 600, 600, 600, 600,
 600, 600, 600, 600, 600, 600, 600, 0, 0, 0,
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 600, 600, 600, 600, 600, 600, 600, 600, 600,
 600, 600, 600, 600, 0, 0, 0, 600, 600, 600,
 600, 0, 600, 600, 600, 600, 600, 600, 600, 0,
 0, 600, 0, 600, 600, 600, 600, 600, 600, 600,
 600, 0, 600, 600, 0, 600, 600, 600, 600, 0,
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 0, 0, 0, 0, 0, 0, 600, 0, 0,
 0, 0, 600, 600, 0, 600, 0, 0, 0, 0,
 0, 0, 0, 0, 0, 600, 0, 0, 600, 600,
 0, 600, 0, 0, 0, 0}},
{
  "courier-oblique",
  "courier-medium-o",
  "Courier-Oblique",
 {0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 0, 600, 600, 600, 600, 600, 600, 600, 600,
 600, 600, 600, 600, 600, 600, 600, 600, 600, 600,
 600, 600, 600, 600, 600, 600, 600, 600, 600, 600,
 600, 600, 600, 600, 600, 600, 600, 600, 600, 600,
 600, 600, 600, 600, 600, 600, 600, 600, 600, 600,
 600, 600, 600, 600, 600, 600, 600, 600, 600, 600,
 600, 600, 600, 600, 600, 600, 600, 600, 600, 600,
 600, 600, 600, 600, 600, 600, 600, 600, 600, 600,
 600, 600, 600, 600, 600, 600, 600, 600, 600, 600,
 600, 600, 600, 600, 600, 600, 600, 0, 0, 0,
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 600, 600, 600, 600, 600, 600, 600, 600, 600,
 600, 600, 600, 600, 0, 0, 0, 600, 600, 600,
 600, 0, 600, 600, 600, 600, 600, 600, 600, 0,
 0, 600, 0, 600, 600, 600, 600, 600, 600, 600,
 600, 0, 600, 600, 0, 600, 600, 600, 600, 0,
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 0, 0, 0, 0, 0, 0, 600, 0, 0,
 0, 0, 600, 600, 0, 600, 0, 0, 0, 0,
 0, 0, 0, 0, 0, 600, 0, 0, 600, 600,
 0, 600, 0, 0, 0, 0}},
{
  "courier",
  "courier-medium-r",
  "Courier",
 {0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 0, 600, 600, 600, 600, 600, 600, 600, 600,
 600, 600, 600, 600, 600, 600, 600, 600, 600, 600,
 600, 600, 600, 600, 600, 600, 600, 600, 600, 600,
 600, 600, 600, 600, 600, 600, 600, 600, 600, 600,
 600, 600, 600, 600, 600, 600, 600, 600, 600, 600,
 600, 600, 600, 600, 600, 600, 600, 600, 600, 600,
 600, 600, 600, 600, 600, 600, 600, 600, 600, 600,
 600, 600, 600, 600, 600, 600, 600, 600, 600, 600,
 600, 600, 600, 600, 600, 600, 600, 600, 600, 600,
 600, 600, 600, 600, 600, 600, 600, 0, 0, 0,
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 600, 600, 600, 600, 600, 600, 600, 600, 600,
 600, 600, 600, 600, 0, 0, 0, 600, 600, 600,
 600, 0, 600, 600, 600, 600, 600, 600, 600, 0,
 0, 600, 0, 600, 600, 600, 600, 600, 600, 600,
 600, 0, 600, 600, 0, 600, 600, 600, 600, 0,
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 0, 0, 0, 0, 0, 0, 600, 0, 0,
 0, 0, 600, 600, 0, 600, 0, 0, 0, 0,
 0, 0, 0, 0, 0, 600, 0, 0, 600, 600,
 0, 600, 0, 0, 0, 0}},
{
  "helvetica-bold",
  "helvetica-bold-r",
  "Helvetica-Bold",
 {0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 0, 556, 333, 474, 556, 556, 889, 722, 278,
 333, 333, 389, 584, 278, 333, 278, 278, 556, 556,
 556, 556, 556, 556, 556, 556, 556, 556, 333, 333,
 584, 584, 584, 611, 975, 722, 722, 722, 722, 667,
 611, 778, 722, 278, 556, 722, 611, 833, 722, 778,
 667, 778, 722, 667, 611, 722, 667, 944, 667, 667,
 611, 333, 278, 333, 584, 556, 278, 556, 611, 556,
 611, 556, 333, 611, 611, 278, 278, 556, 278, 889,
 611, 611, 611, 611, 389, 556, 333, 611, 556, 778,
 556, 556, 500, 389, 280, 389, 584, 0, 0, 0,
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 333, 556, 556, 167, 556, 556, 556, 556, 238,
 500, 556, 333, 333, 611, 611, 0, 556, 556, 556,
 278, 0, 556, 350, 278, 500, 500, 556, 1000, 1000,
 0, 611, 0, 333, 333, 333, 333, 333, 333, 333,
 333, 0, 333, 333, 0, 333, 333, 333, 1000, 0,
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 0, 0, 0, 0, 1000, 0, 370, 0, 0,
 0, 0, 611, 778, 1000, 365, 0, 0, 0, 0,
 0, 889, 0, 0, 0, 278, 0, 0, 278, 611,
 944, 611, 0, 0, 0, 0}},
{
  "helvetica-boldoblique",
  "helvetica-bold-o",
  "Helvetica-BoldOblique",
 {0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 0, 556, 333, 474, 556, 556, 889, 722, 278,
 333, 333, 389, 584, 278, 333, 278, 278, 556, 556,
 556, 556, 556, 556, 556, 556, 556, 556, 333, 333,
 584, 584, 584, 611, 975, 722, 722, 722, 722, 667,
 611, 778, 722, 278, 556, 722, 611, 833, 722, 778,
 667, 778, 722, 667, 611, 722, 667, 944, 667, 667,
 611, 333, 278, 333, 584, 556, 278, 556, 611, 556,
 611, 556, 333, 611, 611, 278, 278, 556, 278, 889,
 611, 611, 611, 611, 389, 556, 333, 611, 556, 778,
 556, 556, 500, 389, 280, 389, 584, 0, 0, 0,
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 333, 556, 556, 167, 556, 556, 556, 556, 238,
 500, 556, 333, 333, 611, 611, 0, 556, 556, 556,
 278, 0, 556, 350, 278, 500, 500, 556, 1000, 1000,
 0, 611, 0, 333, 333, 333, 333, 333, 333, 333,
 333, 0, 333, 333, 0, 333, 333, 333, 1000, 0,
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 0, 0, 0, 0, 1000, 0, 370, 0, 0,
 0, 0, 611, 778, 1000, 365, 0, 0, 0, 0,
 0, 889, 0, 0, 0, 278, 0, 0, 278, 611,
 944, 611, 0, 0, 0, 0}},
{
  "helvetica-oblique",
  "helvetica-medium-o",
  "Helvetica-Oblique",
 {0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 0, 556, 278, 355, 556, 556, 889, 667, 222,
 333, 333, 389, 584, 278, 333, 278, 278, 556, 556,
 556, 556, 556, 556, 556, 556, 556, 556, 278, 278,
 584, 584, 584, 556, 1015, 667, 667, 722, 722, 667,
 611, 778, 722, 278, 500, 667, 556, 833, 722, 778,
 667, 778, 722, 667, 611, 722, 667, 944, 667, 667,
 611, 278, 278, 278, 469, 556, 222, 556, 556, 500,
 556, 556, 278, 556, 556, 222, 222, 500, 222, 833,
 556, 556, 556, 556, 333, 500, 278, 556, 500, 722,
 500, 500, 500, 334, 260, 334, 584, 0, 0, 0,
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 333, 556, 556, 167, 556, 556, 556, 556, 191,
 333, 556, 333, 333, 500, 500, 0, 556, 556, 556,
 278, 0, 537, 350, 222, 333, 333, 556, 1000, 1000,
 0, 611, 0, 333, 333, 333, 333, 333, 333, 333,
 333, 0, 333, 333, 0, 333, 333, 333, 1000, 0,
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 0, 0, 0, 0, 1000, 0, 370, 0, 0,
 0, 0, 556, 778, 1000, 365, 0, 0, 0, 0,
 0, 889, 0, 0, 0, 278, 0, 0, 222, 611,
 944, 611, 0, 0, 0, 0}},
{
  "helvetica",
  "helvetica-medium-r",
  "Helvetica",
 {0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 0, 556, 278, 355, 556, 556, 889, 667, 222,
 333, 333, 389, 584, 278, 333, 278, 278, 556, 556,
 556, 556, 556, 556, 556, 556, 556, 556, 278, 278,
 584, 584, 584, 556, 1015, 667, 667, 722, 722, 667,
 611, 778, 722, 278, 500, 667, 556, 833, 722, 778,
 667, 778, 722, 667, 611, 722, 667, 944, 667, 667,
 611, 278, 278, 278, 469, 556, 222, 556, 556, 500,
 556, 556, 278, 556, 556, 222, 222, 500, 222, 833,
 556, 556, 556, 556, 333, 500, 278, 556, 500, 722,
 500, 500, 500, 334, 260, 334, 584, 0, 0, 0,
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 333, 556, 556, 167, 556, 556, 556, 556, 191,
 333, 556, 333, 333, 500, 500, 0, 556, 556, 556,
 278, 0, 537, 350, 222, 333, 333, 556, 1000, 1000,
 0, 611, 0, 333, 333, 333, 333, 333, 333, 333,
 333, 0, 333, 333, 0, 333, 333, 333, 1000, 0,
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 0, 0, 0, 0, 1000, 0, 370, 0, 0,
 0, 0, 556, 778, 1000, 365, 0, 0, 0, 0,
 0, 889, 0, 0, 0, 278, 0, 0, 222, 611,
 944, 611, 0, 0, 0, 0}},
{
  "symbol",
  "symbol-medium-r",
  "Symbol",
 {0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 0, 500, 333, 713, 500, 549, 833, 778, 439,
 333, 333, 500, 549, 250, 549, 250, 278, 500, 500,
 500, 500, 500, 500, 500, 500, 500, 500, 278, 278,
 549, 549, 549, 444, 549, 696, 660, 710, 612, 652,
 763, 603, 765, 351, 631, 724, 686, 918, 739, 750,
 768, 741, 580, 592, 632, 690, 439, 768, 645, 795,
 650, 333, 863, 333, 658, 500, 500, 631, 549, 549,
 494, 439, 521, 411, 603, 329, 603, 549, 549, 576,
 521, 549, 549, 521, 549, 603, 439, 576, 713, 686,
 493, 686, 494, 480, 200, 480, 549, 0, 0, 0,
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 620, 247, 549, 167, 713, 500, 753, 753, 753,
 753, 1042, 987, 603, 987, 603, 400, 549, 411, 549,
 549, 713, 494, 460, 549, 549, 549, 549, 1000, 603,
 1000, 658, 823, 686, 795, 987, 768, 768, 823, 768,
 768, 713, 713, 713, 713, 713, 713, 713, 768, 713,
 790, 790, 890, 823, 549, 250, 713, 603, 603, 1042,
 987, 603, 987, 603, 494, 329, 790, 790, 786, 713,
 384, 384, 384, 384, 384, 384, 494, 494, 494, 494,
 790, 329, 274, 686, 686, 686, 384, 384, 384, 384,
 384, 384, 494, 494, 494, 0}},
{
  "times-bold",
  "times-bold-r",
  "Times-Bold",
 {0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 0, 500, 333, 555, 500, 500, 1000, 833, 333,
 333, 333, 500, 570, 250, 333, 250, 278, 500, 500,
 500, 500, 500, 500, 500, 500, 500, 500, 333, 333,
 570, 570, 570, 500, 930, 722, 667, 722, 722, 667,
 611, 778, 778, 389, 500, 778, 667, 944, 722, 778,
 611, 778, 722, 556, 667, 722, 722, 1000, 722, 722,
 667, 333, 278, 333, 581, 500, 333, 500, 556, 444,
 556, 444, 333, 500, 556, 278, 333, 556, 278, 833,
 556, 500, 556, 556, 444, 389, 333, 556, 500, 722,
 500, 500, 444, 394, 220, 394, 520, 0, 0, 0,
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 333, 500, 500, 167, 500, 500, 500, 500, 278,
 500, 500, 333, 333, 556, 556, 0, 500, 500, 500,
 250, 0, 540, 350, 333, 500, 500, 500, 1000, 1000,
 0, 500, 0, 333, 333, 333, 333, 333, 333, 333,
 333, 0, 333, 333, 0, 333, 333, 333, 1000, 0,
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 0, 0, 0, 0, 1000, 0, 300, 0, 0,
 0, 0, 667, 778, 1000, 330, 0, 0, 0, 0,
 0, 722, 0, 0, 0, 278, 0, 0, 278, 500,
 722, 556, 0, 0, 0, 0}},
{
  "times-bolditalic",
  "times-bold-i",
  "Times-BoldItalic",
 {0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 0, 500, 389, 555, 500, 500, 833, 778, 333,
 333, 333, 500, 570, 250, 333, 250, 278, 500, 500,
 500, 500, 500, 500, 500, 500, 500, 500, 333, 333,
 570, 570, 570, 500, 832, 667, 667, 667, 722, 667,
 667, 722, 778, 389, 500, 667, 611, 889, 722, 722,
 611, 722, 667, 556, 611, 722, 667, 889, 667, 611,
 611, 333, 278, 333, 570, 500, 333, 500, 500, 444,
 500, 444, 333, 500, 556, 278, 278, 500, 278, 778,
 556, 500, 500, 500, 389, 389, 278, 556, 444, 667,
 500, 444, 389, 348, 220, 348, 570, 0, 0, 0,
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 389, 500, 500, 167, 500, 500, 500, 500, 278,
 500, 500, 333, 333, 556, 556, 0, 500, 500, 500,
 250, 0, 500, 350, 333, 500, 500, 500, 1000, 1000,
 0, 500, 0, 333, 333, 333, 333, 333, 333, 333,
 333, 0, 333, 333, 0, 333, 333, 333, 1000, 0,
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 0, 0, 0, 0, 944, 0, 266, 0, 0,
 0, 0, 611, 722, 944, 300, 0, 0, 0, 0,
 0, 722, 0, 0, 0, 278, 0, 0, 278, 500,
 722, 500, 0, 0, 0, 0}},
{
  "times-italic",
  "times-medium-i",
  "Times-Italic",
 {0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 0, 500, 333, 420, 500, 500, 833, 778, 333,
 333, 333, 500, 675, 250, 333, 250, 278, 500, 500,
 500, 500, 500, 500, 500, 500, 500, 500, 333, 333,
 675, 675, 675, 500, 920, 611, 611, 667, 722, 611,
 611, 722, 722, 333, 444, 667, 556, 833, 667, 722,
 611, 722, 611, 500, 556, 722, 611, 833, 611, 556,
 556, 389, 278, 389, 422, 500, 333, 500, 500, 444,
 500, 444, 278, 500, 500, 278, 278, 444, 278, 722,
 500, 500, 500, 500, 389, 389, 278, 500, 444, 667,
 444, 444, 389, 400, 275, 400, 541, 0, 0, 0,
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 389, 500, 500, 167, 500, 500, 500, 500, 214,
 556, 500, 333, 333, 500, 500, 0, 500, 500, 500,
 250, 0, 523, 350, 333, 556, 556, 500, 889, 1000,
 0, 500, 0, 333, 333, 333, 333, 333, 333, 333,
 333, 0, 333, 333, 0, 333, 333, 333, 889, 0,
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 0, 0, 0, 0, 889, 0, 276, 0, 0,
 0, 0, 556, 722, 944, 310, 0, 0, 0, 0,
 0, 667, 0, 0, 0, 278, 0, 0, 278, 500,
 667, 500, 0, 0, 0, 0}},
{
  "times-roman",
  "times-medium-r",
  "Times-Roman",
 {0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 0, 500, 333, 408, 500, 500, 833, 778, 333,
 333, 333, 500, 564, 250, 333, 250, 278, 500, 500,
 500, 500, 500, 500, 500, 500, 500, 500, 278, 278,
 564, 564, 564, 444, 921, 722, 667, 667, 722, 611,
 556, 722, 722, 333, 389, 722, 611, 889, 722, 722,
 556, 722, 667, 556, 611, 722, 722, 944, 722, 722,
 611, 333, 278, 333, 469, 500, 333, 444, 500, 444,
 500, 444, 333, 500, 500, 278, 278, 500, 278, 778,
 500, 500, 500, 500, 333, 389, 278, 500, 500, 722,
 500, 500, 444, 480, 200, 480, 541, 0, 0, 0,
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 333, 500, 500, 167, 500, 500, 500, 500, 180,
 444, 500, 333, 333, 556, 556, 0, 500, 500, 500,
 250, 0, 453, 350, 333, 444, 444, 500, 1000, 1000,
 0, 444, 0, 333, 333, 333, 333, 333, 333, 333,
 333, 0, 333, 333, 0, 333, 333, 333, 1000, 0,
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 0, 0, 0, 0, 889, 0, 276, 0, 0,
 0, 0, 611, 722, 889, 310, 0, 0, 0, 0,
 0, 667, 0, 0, 0, 278, 0, 0, 278, 500,
 722, 500, 0, 0, 0, 0}}
};
