/* FILE ttest */
     /* From david */
     /*  Assumes different variances for each sample */
#include <math.h>
#include <stdio.h>

#define	DEBUG		0

/* FUNCTIONS:
 * Public
 *     Ttest		Do a Student's T test: compare 2 groups
 *     Ttest_short	Do a Ttest given 2 means, 2 standard errors
 *     Ttest_vs_zero	Do a Student's T test: test 1 group against zero
 */

/* STATIC FUNCTION DECLARATIONS: */
static float tprob(double t, int dof, int tails);
static double Mean(int *data, int count);
static double Variance(int *data, int count);
static double betai (double a, double b, double x);
static double gammln(double xx);
static double betacf (double a, double b, double x);

extern void Warning(char *string);
extern void Exit(char *string, char *name);
/* ********************************************************************	*/

/* FUNCTION Ttest */
	 /* Unpaired, 1 or 2 sided ttest (is data1 != data2) ?	*/
float Ttest(int *data1, int count1, int *data2, int count2, int sides) {
   double t_value;
   double sp = sqrt( ((count1-1)*Variance(data1,count1) +
		      (count2-1)*Variance(data2,count2)) / (count1+count2-2));

   if (DEBUG==3) {
      int i;
      for (i=0; i<count1; i++)
         fprintf(stderr, "%d ", *(data1+i));
      fprintf(stderr, "\n");
      for (i=0; i<count2; i++)
         fprintf(stderr, "%d ", *(data2+i));
      fprintf(stderr, "\n");
      }
   if (DEBUG==1)
      fprintf(stderr, "DEBUGt: sp=%.1f = sqrt ( (%d*%.1f + %d*%.1f) / %d)\n",
		sp,count1-1,Variance(data1,count1),count2-1,
		Variance(data2,count2),count1+count2-2);

   if (fabs(sp) < 0.0001) {	/* Probably NO SPIKES in the interval!	*/
      if (DEBUG) {
         if (fabs(Mean(data1,count1))<0.001 || fabs(Mean(data2,count2))<0.001) {
	    Warning("sp is zero, but the means aren't!");
	    fprintf(stderr, "DEBUG: %.1f %.1f (%.1f*%d + %.1f*%d)\n",	
		Mean(data1,count1), Mean(data2,count2),
		Variance(data1,count1), count1-1,
		Variance(data2,count2), count2-1);
	    }
	 }
      return(1.);
      }
   
   t_value = (Mean(data1,count1) - Mean(data2,count2)) /
   			(sp * sqrt((double)1/count1 + (double)1/count2));

   if (DEBUG==1)
       fprintf(stderr, "DEBUG: %4.1f - %4.1f / (%4.1f) --> %2.1f\n",
   		Mean(data1,count1), Mean(data2,count2),
   		sp * sqrt((double)1/count1 + (double)1/count2),
   	        tprob(t_value, count1 + count2 - 2, sides));
   return(tprob(t_value, count1 + count2 - 2, sides));
   }
/* ********************************************************************	*/

/* FUNCTION Ttest_short */
	 /* Unpaired, 1 or 2 sided ttest (is data1 != data2) ?	*/
	 /* NOTE: variance = count * sem * sem			*/
float Ttest_short(float mean1, int count1, float sem1,
                  float mean2, int count2, float sem2,
	          int sides) {
   double t_value;
   double sp = sqrt( ((double)(count1-1)*count1*sem1*sem1 +
		      (double)(count2-1)*count2*sem2*sem2) / (count1+count2-2));
   if (fabs(sp) < 0.00001)
      return(1.);
   if (DEBUG==1)
      fprintf(stderr, "DEBUGs: sp=%.3f = sqrt ( (%d*%.3f + %d*%.3f) / %d)\n",
		sp,count1-1,count1*sem1*sem1,
		   count2-1,count2*sem2*sem2,count1+count2-2);
   t_value = (mean1 - mean2) / (sp * sqrt((double)1/count1 + (double)1/count2));
   return(tprob(t_value, count1 + count2 - 2, sides));
   }
/* ********************************************************************	*/

/* FUNCTION Ttest_vs_zero */
	 /* 1 or 2 sided ttest (is data1 != 0) ?	*/
float Ttest_vs_zero(int *data, int count, int sides) {
   double t_value;
   double sp = sqrt(Variance(data,count));
   if (DEBUG==1)
       fprintf(stderr, "DEBUG: sp=%.1f\n", sp);

   if (fabs(sp) < 0.00001) {	/* Probably NO SPIKES in the interval!	*/
      if (DEBUG) {
         if (fabs(Mean(data,count)) < 0.00001) {
	    Warning("sp is zero, but the mean isn't!");
	    fprintf(stderr, "DEBUG: %.1f %.1f\n", Mean(data,count), sp);
	    }
	 }
      return(1.);
      }
   
   t_value = Mean(data,count) / (sp/sqrt((double)count));
   if (DEBUG==1)
       fprintf(stderr, "DEBUG: %4.1f - %4.1f / (%4.1f) --> %5.1f\n",
   		Mean(data,count), sp, sqrt((double)count),
   	        tprob(t_value, count-1, sides));
   return(tprob(t_value, count - 1, sides));
   }
/* ********************************************************************	*/

/* FUNCTION Mean */
	 /* Mean of a vector */
static double Mean(int *data, int count) {
   int i;
   int sum = 0;

   for (i=0; i<count ; i++)
      sum += *(data+i);
   return(sum/(double)count);
   }
/* ********************************************************************	*/

/* FUNCTION Variance */
	 /* Variance of a vector */
static double Variance(int *data, int count) {
   int i;
   int sum = 0;
   double avg;
   double var = 0;

   for (i=0; i<count; i++)
      sum += *(data+i);
   avg = sum / (double) count;

   for (i=0; i<count; i++)
      var += pow(data[i] - avg, 2.0);
   return( var/(count-1));
   }
/* ********************************************************************	*/

/* FUNCTION tprob */
	 /* Calc probability of 'accidentally' getting this t-value */
static float tprob(double t, int dof, int tails) {
   double atv;

   if (dof < 2)
      Exit((dof==1)?"One deg freedom":"No deg freedom", "ttest.c:tprob()");
   if (tails != 1 && tails != 2)
      Exit("tails of ttest must be 1 or 2", "ttest.c: tprob()");
   atv = betai( (double)dof/2.0,  0.5,  (double)dof/((double)dof+t*t) );
   return((float) atv*(double)tails/2.0);
   }
/* ********************************************************************	*/

/* FUNCTION betai */
	 /* Incomplete beta dstribution, to calc t distribute prob	*/
static double betai (double a, double b, double x) { 
   double bt;

   if (x < 0.0 || x > 1.0)
      Exit("cryptic err", "ttest.c:betai()");
   if (fabs(x) < 0.00001 || fabs(x-1) < 0.00001)
      bt = 0.0;
   else
      bt = exp(gammln(a+b)-gammln(a)-gammln(b)+a*log(x)+b*log(1.0-x));

   if (x < (a+1.0)/(a+b+2.0))
      return(bt*betacf(a,b,x)/a);
   else
      return(1.0-bt*betacf(b,a,1.0-x)/b);
   }
/* ********************************************************************	*/

/* FUNCTION gammln */
	 /* Used by betai(); from Numerical Recipes 	*/
static double gammln(double xx) {
   double x,tmp,ser;
   static double cof[6] = {76.18009173,-86.50532033,24.01409822,
			-1.231739516,0.120858003e-2,-0.536382e-5};
   int j;
   x = xx-1.0;
   tmp = x+5.5;
   tmp -= (x+0.5)*log(tmp);
   ser = 1.0;
   for(j=0; j<=5; j++) {
     x += 1.0;
     ser += cof[j]/x;
     }
   return(-tmp+log(2.50662827465*ser));
   }
/* ********************************************************************	*/

/* FUNCTION betacf */
	 /* Used by betacf */
static double betacf (double a, double b, double x) {
   int m, m2;
   double aa,c,d,del,h,qab,qam,qap;
#  define MAXIT 100
#  define EPS 3.0e-7
#  define FPMIN 1.0e-30

   qab = a+b;
   qap = a+1.0;
   qam = a-1.0;
   c   = 1.0;
   d   = 1.0-qab*x/qap;
   if (fabs(d) < FPMIN)
      d = FPMIN;
   d = 1.0/d;
   h = d;
   for (m=1; m<=MAXIT; m++) {
     m2 = 2 * m;
     aa = m*(b-m)*x/((qam+m2)*(a+m2));
     d = 1.0 + aa*d;
     if (fabs(d) < FPMIN) d = FPMIN;
     c = 1.0 + aa/c;
     if (fabs(c) < FPMIN) c = FPMIN;
     d = 1.0/d;
     h *=  d*c;
     aa = -(a+m)*(qab+m)*x/((a+m2)*(qap+m2));
     d = 1.0 + aa*d;
     if (fabs(d) < FPMIN) d = FPMIN;
     c = 1.0 + aa/c;
     if (fabs(c) < FPMIN) c = FPMIN;
     d = 1.0/d;
     del = d*c;
     h *= del;
     if (fabs(del-1.0) < EPS) break;
     }
   if (m > MAXIT)
      Exit("a or b too big, or MAXIT too small", "ttest.c:betacf()");
   return(h);
   }
/* ********************************************************************	*/
