/* FILE time */
     /* Do trial timing stuff */

/* Alignment is a major issue.  Here is the sequence:
   main.c	main()			calls Set_TrialTypes()
   trial.c	Set_TrialTypes()	calls Set_TrialTiming()
   time.c	Set_TrialTiming()	calls Set_AlignTime()

   There is a separate module (align.c) that does actual alignment

   Within time.c:
    Set_TrialTiming()	calls Set_AlignTime() (in align.c)
    Get_AlignTime()	STATIC: returns TrialAlignmentTime to this file only.
	Shift data by this much (ms) to get the user's requested alignment.
    Get_ZeroTime()	PUBLIC: returns -StartTime to rest of world.
	This is when (ms) time zero occurs in the aligned data.
	This is a function of where the alignment is in the stack, and
	 also exactly which data ended up being retained.
	 Interval time zero corresponds to Get_ZeroTime() in the analog data
	    that was retained.
    Get_TapeOnTime()	Might be a mistake to export this; use carefully ..
 */

#include "config.h"
#include "deffs.h"
#include "event.h"

#define  DEBUG 		0		/* -1:align  0:none  1,2:timing	*/

/* ********************************************************************	*/
/* ********************************************************************	*/
/*
PUBLIC:
  Init_Times		Call ONCE by Set_TrialTypes: set up start/finish calc
  Set_TrialTiming	Call ONCE by Set_TrialTimes: set start and finish
			times COMMON TO ALL TRIALS and calls Set_AlignTime.
  DurationTime		Return longest duration (earliest ending - latest start)
  SkipAnalogTime	Ms to skip (from start) to align with all other trials
  SkipSpikeTime		Ms to subtract from spike times so aligned with analog
  Get_ZeroTime		Alignment point falls here in (analog & spike) data
  Get_TargetTime	Time of i-th target on - time at pixel 0
  Get_TargetOffTime	Time of i-th target off - time at pixel 0
  Get_TargetBlankTime	Time of i-th target blank on/off - time at pixel 0
  Get_TapeOnTime	Help convert from times to location in data stream
  Set_Times_From_CmdLine      Cmd-line requests for start/stop times
  Unset_Times_From_CmdLine
  Get_Times_From_CmdLine
  Get_RedrawTimes	Time of i-th redraw - time at pix 0

  Interval_To_Stack_Time
  Stack_To_Interval_Time

  Force_Zero_SkipTime	Make it return 0 (for sac.c code)

PRIVATE:
 */
/* ********************************************************************	*/
/* ********************************************************************	*/

static int StartTime;			/* Earliest on entire page (ms)	*/
static int EndTime;			/* Latest   on entire page (ms)	*/
static int Duration;
/* ********************************************************************	*/

static int TimesFromCmdLine = 0;	/* Boolean			*/
static int Zero_SkipTime = 0;
	/* 'Skip_{Analog,Spike}Time()' returns 0: Set_AlignTime(),sac.c	*/
/* ********************************************************************	*/
/* *************  PUBLIC FUNCTIONS ************************************	*/
/* ********************************************************************	*/
/* ********************************************************************	*/

/* FUNCTION Init_Times */
	 /* Set up for a mess of Set_TrialTiming() calls */
void Init_Times() {
	if (TimesFromCmdLine == 0) {
	   StartTime = -32000;
	   EndTime   =  32000;
	   }
	InitAlignTimes();				/* align.c	*/
	}
/* ********************************************************************	*/

/* FUNCTION Set_TrialTiming */
	 /* Set trial INDEPENDENT start & finish times			*/
	 /* (Use latest TAPE_ON, earliest TAPE_OFF of entire file)	*/
	 /* Could Rewrite to set timing for each CLASS separately?	*/
	 /* Called only once per trial by trial.c			*/
void Set_TrialTiming() {
   int align = Set_AlignTime();		/* CALL ONLY ONCE PER TRIAL	*/
   int start = EventExtract(TAPE_ON, TIME, 1) - align;	/* Relative start*/
   int end   = EventExtract(TAPE_OFF, TIME, 1) - align;	/* Relative end	 */

   if (align == SKIP_TRIAL)	/* -ax: skip trials with bad alignment	*/
      return;

   if ((align==FAIL) || (start==FAIL-align) || (end==FAIL-align)) {
      fprintf(stderr, "\n      start=%d, end=%d, align=%d", start, end, align);
      fprintf(stderr, "      tape on=%d, tape off=%d\n\n",
          EventExtract(TAPE_ON,TIME,1), EventExtract(TAPE_OFF,TIME,1));
      PrintHeader();
      Exit("Bad timing", "Set_TrialTiming()");
      }
   if (end < EndTime)		/* Use shortest trial to set endtime	*/
      EndTime = end;
   if (start > StartTime)	/* Use latest trial to set start time	*/
      StartTime = start;

   Duration = EndTime - StartTime;

   if (RunStatus_Header() != SUCCESS) {
      if (FrameCount_Header() <= 0) {
          Set_FrameCount_Header(0);
	  Duration = 0;
	  EndTime = StartTime;
          }
      }

   if (DEBUG == 2)
      fprintf(stderr,"DEBUG-Times: %d to %d = %d\n",StartTime,EndTime,Duration);
   }
/* ********************************************************************	*/

/* FUNCTION DurationTime */
	 /* Return longest duration (earliest ending - latest start)	*/
int DurationTime() { return(Duration); }
/* ********************************************************************	*/

/* FUNCTION Force_Zero_SkipTime */
  	 /* Make Skip{Analog,Spike}Time() return 0	*/
	 /* Called by sac.c and Set_AlignTime 		*/
void Force_Zero_SkipTime(int value) {
	Zero_SkipTime = value; 		/* ((value==ON) ? 1 : 0);	*/
	Duration = FrameCount_Header() * MsPerFrame_Header();
     	}  /* duration may not be set yet; set so SkipTime() is ok	*/
/* ********************************************************************	*/

/* FUNCTION SkipAnalogTime */
	 /* Ms to skip in analog sample so that each is aligned	*/
	 /* StartTime is first frame common to ALL TRIALS	*/
	 /* NOTE: StartTime + Get_AlignTime() == time at pixel 0	*/
	 /* NOTE: Use for switches, since switch times are stack times	*/
	/* If use only a single trial, is zero.  Else is how much data
	 * (in ms) is thrown away for this particular trial.  (If you're
	 * analyzing only one trial, then for the trial that starts the
	 * latest, SkipAnalogTime is still 0)
	 */
int SkipAnalogTime() {
	if (DEBUG>0)
           fprintf(stderr, "skip analog is %d + %d - %d = %d (trial %d)\n",
	       StartTime, Get_AlignTime(), EventExtract(TAPE_ON,TIME,1),
	       StartTime + Get_AlignTime() - EventExtract(TAPE_ON,TIME,1),
	       TrialNumber_Header());
	if (Zero_SkipTime)		/* So don't skip in analog read	*/
	    return(0);
	return(StartTime + Get_AlignTime() - EventExtract(TAPE_ON,TIME,1));
	}	/* Time at pix 0 + time of this start: usu >= 0 */
/* ********************************************************************	*/

/* FUNCTION SkipSpikeTime */
	 /* Ms to subtract from spike times so that aligned with analog	*/
int SkipSpikeTime() {
	if (Zero_SkipTime==2)		/* For reading arm data		*/
	    return(Get_TapeOnTime());
	if (Zero_SkipTime)		/* So don't skip in analog read	*/
	    return(0);
	return(StartTime + Get_AlignTime());
	}		/* Time at pixel zero 				*/
/* ********************************************************************	*/

/* FUNCTION Get_ZeroTime */
	 /* Alignment event falls here in (analog + spike) data streams	*/
/* Input data:    align analog/spike/switch data using this time
 * graph, histo:  time returned is 'zero time' requested by user	
 */
int Get_ZeroTime() {
	if (StartTime == -32000)			/* Not yet set	*/
	   return(0);
	if (StartTime <= FAIL)
	   Exit("StartTime failed!", "Get_ZeroTime()");
	return(-StartTime);		/* Return a positive number!	*/
	} /* StartTime: ms from align event back to 1st (retained) frame*/
/* ********************************************************************	*/

/* FUNCTION Interval_To_Stack_Time */
int Interval_To_Stack_Time(int interval_time) {
	return((interval_time==FAIL) ? FAIL : interval_time + Get_AlignTime());
	}
/* FUNCTION Stack_To_Interval_Time */
int Stack_To_Interval_Time(int stack_time) {
	return((stack_time==FAIL) ? FAIL : stack_time - Get_AlignTime());
	}
/* ********************************************************************	*/

/* FUNCTION Get_TargetTime */
	/* Time of target on (re: time at pixel 0)	*/
int Get_TargetTime(int i) {
	int time = AnyTargetEventExtract(TIME, i);

	if (time == FAIL)
	   return(FAIL);
	return(time - (StartTime + Get_AlignTime()));
	}
/* ********************************************************************	*/

/* FUNCTION Get_RedrawTime */
	/* Time of Redraw (re: time at pixel 0)	*/
int Get_RedrawTime(int i) {
	int time = EventExtract(TARGET_REDRAW, TIME, i);

	if (time == FAIL)
	   return(FAIL);
	return(time - (StartTime + Get_AlignTime()));
	}
/* ********************************************************************	*/

/* FUNCTION Get_TargetOffTime */
	/* Time of target off (re: time at pixel 0)	*/
int Get_TargetOffTime(int i) {
	int time = EventExtract(TARGET_OFF, TIME, i);

	if (time == FAIL)
	   return(FAIL);
	return(time - (StartTime + Get_AlignTime()));
	}
/* ********************************************************************	*/

/* FUNCTION Get_TargetBlankTime */
	 /* Time of target blank/unblank (re: time at pix 0) */
int Get_TargetBlankTime(int i) {
	int time = EventExtract(TARGET_BLANK, TIME, i);

	if (time == FAIL)
	   return(FAIL);
	return(time - (StartTime + Get_AlignTime()));
	}
/* ********************************************************************	*/

/* FUNCTION Get_TapeOnTime */
	 /* When does tape turn on ? */
int Get_TapeOnTime() { return(EventExtract(TAPE_ON, TIME, 1)); }
/* ********************************************************************	*/

/* FUNCTION Set_Times_From_CmdLine */
	 /* Allow for cmd-line requests for start/stop re: aligntime	*/
void Set_Times_From_CmdLine(int start, int finish) {
	TimesFromCmdLine = 1;
	StartTime =  start;
	EndTime = finish;	/* So can see if needs to finish earlier */
	Duration = finish - start;
	if (DEBUG>0) fprintf(stderr,"From cmd line: %d to %d\n",start,finish);
	}
/* ********************************************************************	*/


/* FUNCTION Get_Times_From_CmdLine */
/* FUNCTION Unset_Times_From_CmdLine */
void Unset_Times_From_CmdLine() { TimesFromCmdLine = 0; }
int  Get_Times_From_CmdLine() { return(TimesFromCmdLine); }
/* ********************************************************************	*/
