/* FILE ps.c */
     /* Postscript driver */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#define DEBUG		0

#define DSC	1	/* Allows PC gv to zoom, & good for latex	*/

#define Y_PAGE	11.0
#define X_PAGE	 8.5
#define EDGE	 0.5				/* Margin on each side	*/

#define	CENTER		1

/* FILE *out = stdout;		ORIGINAL VERSION */

FILE *out = NULL;

static int stroked;				/* Stroked path?	*/
static double scale;				/* Vs max == 1.		*/
static int orientation;				/* Takes 1 of 3 values	*/
#define NEW_PLOT	2
#define LANDSCAPE	1
#define PORTRAIT	0

/* PUBLIC FUNCTIONS */
void openpl(FILE *output);
void closepl();
void space(int x00, int y00, int x01, int y01);
void move(int x, int y);
void box(int x00, int y00, int x01, int y01);
void filledbox(int x00, int y00, int x01, int y01);
void erase();
void arc(int xc, int yc, int x00, int y00, int x01, int y01);
void circle(int x, int y, int r);
void point(int x, int y);
void SetColor(int red, int green, int blue);
void SetLineWidth(int width);
void linemod(char *s);
void fill(int level);
int  Is_PlotOpen(void);

void label(char *s);
void fontsize(int size);
void fontname (char *s);
void rotate (int w, int h, int angle);
void alabel(int x_justify, int y_justify, char *s);


/* PRIVATE FUNCTIONS */
static void Set_font();
static void Stroke();
static void Report_Info();
/* ********************************************************************	*/
/* ********************************************************************	*/

/* FUNCTION openpl */
	 /* Write postscript header, set defaults, save graphics state	*/
void openpl(FILE *output) {
   out = output;				/* Hook up streams	*/
   orientation = NEW_PLOT;			/* Flag to space()	*/
   stroked = 1;					/* Initialize		*/
   scale = 1;					/* Initialize		*/

#if DSC
   fputs("%!PS-Adobe-2.0 epsf-1.2\n", out);
   				/* Only DSC can zoom on PC ghostview	*/
   fputs("%%Pages: 1\n", out);			/* DSC commands		*/
   /* fputs("%%BoundingBox: 167 0 447 757\n", out);  Original version	*/
   fputs("%%BoundingBox: 867 0 447 757\n", out); /* Doesn't matter if too big?*/
   fputs("%%Page: 1 1\n", out);			/* DSC commands		*/
#else
   fputs("%!\n", out);
#endif
   fputs("106 45 {dup mul exch dup mul add 1.0 exch sub } setscreen\n", out);
   fputs("gsave\n\n", out);
   fputs("/M { moveto } bind def\n", out);
   fputs("/L { lineto } bind def\n", out);
   fputs("/R { rlineto } bind def\n", out);
   fputs("/H { 0 rlineto } bind def\n", out);
   fputs("/V { 0 exch rlineto } bind def\n", out);
   fputs("/S { stroke } bind def\n", out);
   fputs("/lw { setlinewidth } bind def\n", out);
   fputs("/tr { translate } bind def\n", out);
   fputs("/sh { show } bind def\n", out);
   fputs("/sf { scalefont setfont } bind def\n", out);
   fputs("/DeviceRGB setcolorspace\n", out);	/* Versus DeviceGray	*/
   fputs("true setstrokeadjust\n", out);	/* Fine point (manual!)	*/
   	/* Improve grey scale, especially on ghostscript output */
   Set_font();
   }
/* ********************************************************************	*/

/* FUNCTION Is_PlotOpen */
int Is_PlotOpen() { return(out != NULL); }
/* ********************************************************************	*/

/* FUNCTION closepl */
	 /* Tell postscript to show page, then restore graphics state	*/
void closepl(void) {
   Report_Info();				/* Details into file	*/
   fputs("S\n", out);				/* Stroke current path	*/
   fputs("showpage\n", out);
   fputs("grestore\n", out);
   }
/* ********************************************************************	*/
/* ********************************************************************	*/

/* FUNCTION space */
	 /* Set up rotation, scaling and offset per request */
void space(int x00, int y00, int x01, int y01) {
   static int prev_x0 = 0, prev_y0 = 0;
   static int x_range = 0, y_range = 0;
   static float height, width;			/* Current fig, inches	*/

   Stroke();
   if (orientation == NEW_PLOT) {		/* Set up for new figure*/
      prev_x0 = prev_y0 = 0;
      x_range = y_range = 0;
      }

#ifdef WRITE_THIS
   if ((x01-x00) == 0) {	/* Width not specified: use full page	*/
	/* Used by text calls that want access to entire page,
	 *  INDEPENDENT of the space that the figure takes up (which is
	 *  determined by number of rows & columns + ratio of HEIGHT to
	 *  trial duration, a complex result).
	 *  Will set width of page equal to 1000.			*/
	 *  NOTE: Assumes that y_range is unchanged from previous call! */
	 */
      if (y01-y00 != y_range)
	 fprintf(stderr, "ps.c space(): ignoring new y_range!!\n");
      x00 =
      fprintf(out, "\n%d %d tr\n", prev_x0-x00, prev_y0-y00);
      fprintf(out, " scale\n"
      stroked = 1;
      return;
      }
#endif


   if ( (orientation == NEW_PLOT) ||		/* New plot?		*/
        (x_range != (x01-x00)) ||		/* Changed x/y scaling?	*/
	(y_range != y01-y00)) {			/* Redo scaling/rotate:	*/

	/* initmatrix fools programs that want to embed */
      /* fputs("initmatrix\n", out);		* Return to init state	*/
      fputs("grestore gsave\n", out);		/* Return to init state	*/
      fputs("/DeviceRGB setcolorspace\n", out);	/* Versus DeviceGray	*/
      fputs("true setstrokeadjust\n", out);	/* Fine point (manual!)	*/

      x_range = x01-x00;			/* New ranges		*/
      y_range = y01-y00;

      if (y_range > x_range) {			/* What orientation?	*/
         orientation = PORTRAIT;			/* Portrait	*/
	 height = Y_PAGE - 2 * EDGE;
	 width = X_PAGE - 2 * EDGE;
      } else {
         orientation = LANDSCAPE;			/* Landscape	*/
         fprintf(out, "90 rotate\n");
         fprintf(out, "0 %0.f tr\n", -X_PAGE*72);
	 height = X_PAGE - 2 * EDGE;
	 width = Y_PAGE - 2 * EDGE;
         }   

     if ((width/x_range) < (height/y_range))	/* Use smaller scale	*/
        scale = (width*72.) / x_range;		   /* Based on width	*/
     else
        scale = (height*72.) / y_range;		   /* Or base on height	*/
	

     fprintf(out, "%.6f %.6f scale\n", scale, scale);
     fprintf(out, "%d %d tr\n\n", -(prev_x0=x00), -(prev_y0=y00));

#    if (CENTER)
		/* Half (total height/width in pix minus pix used)	*/
     if (orientation == PORTRAIT)
        fprintf(out, "%d %d tr\n\n",
		(int)(.5+((72./scale * X_PAGE) - x_range)/2),
		(int)(.5+((72./scale * Y_PAGE) - y_range)/2));
     else
        fprintf(out, "%d %d tr\n\n",
		(int)(.5+((72./scale * Y_PAGE) - x_range)/2),
		(int)(.5+((72./scale * X_PAGE) - y_range)/2));
#    if (DEBUG && CENTER)
        {
        extern void move(int x, int y);
        extern void cont(int x, int y);
        move(x00+100, y00+100);
        cont(x00+100, y01-100);
        Stroke();
        move(x00+100, y00+100);
        cont(x01-100, y00+100);
        Stroke();
        }
#	endif
#     endif
     }

   if ((prev_x0 != x00) || (prev_y0 != y00)) {		/* Translate?	*/
     fprintf(out, "\n%d %d tr\n", prev_x0-x00, prev_y0-y00);
     prev_x0 = x00;
     prev_y0 = y00;
     }
   stroked = 1;

   if (DEBUG) {
    fprintf(stdout,"%%%%%% DEBUG: space(%d %d %d %d) scale:%.5f, %s\n",
	x00,y00,x01,y01,scale, (orientation==LANDSCAPE)?"landscape":"portrait");
    fprintf(stdout, "\n");
    }
   }
/* ********************************************************************	*/
/* ********************************************************************	*/
static int x_was;
static int y_was;

/* FUNCTION move */
	 /* Move without drawing line; first stroke previous line */
void move(int x, int y) {
	Stroke();
	fprintf(out, "%d %d M\n", x_was=x, y_was=y);
	stroked = 0;
	}
/* ********************************************************************	*/

/* FUNCTION cont */
	 /* Move and draw line */
void cont(int x, int y) {
	if (x == x_was) {
	   fprintf(out, "%d V\n", y-y_was);
	   y_was = y;
	} else if (y == y_was) {
	   fprintf(out, "%d H\n", x-x_was);
	   x_was = x;
	} else {
	   fprintf(out, "%d %d R\n", x-x_was, y-y_was);
	   x_was = x;
	   y_was = y;
	   }
	/* fprintf(out, "%d %d L\n", x,y); */
	}
/* ********************************************************************	*/

/* FUNCTION box */
	 /* Draw a box */
void box(int x00, int y00, int x01, int y01) {
	Stroke();
        fprintf(out, "%d %d %d %d rectstroke\n", x00, y00, x01-x00, y01-y00);
	/* fprintf(out, "%d %d %d %d %d %d %d %d %d %d M L L L L fill\n",
	 			x00,y00, x00,y01, x01,y01, x01,y00, x00,y00); */
	}
void filledbox(int x00, int y00, int x01, int y01) {
	Stroke();
        fprintf(out, "%d %d %d %d rectfill\n", x00, y00, x01-x00, y01-y00);
	}
/* ********************************************************************	*/

/* FUNCTION erase */
	 /* Clear page */
void erase(void) {
	fputs("clippath 1 setgray fill\n", out);
        fputs("/DeviceRGB setcolorspace\n", out);	/* Need:setgray	*/
	}

/* ********************************************************************	*/

/* FUNCTION arc */
	 /* Draw a arc; not used; may not work!! */
void arc(int xc, int yc, int x00, int y00, int x01, int y01) {
	double radius = sqrt( (double)(xc-x00)*(xc-x00) + (yc-y00)*(yc-y00));
	double theta1 = asin( ((y00-yc) / radius) * (2 * 3.14159265 / 360));
	double theta2 = asin( ((y01-yc) / radius) * (2 * 3.14159265 / 360));
		
	if (x01 == x00) fprintf(out," "); /* prevent 'unused variable' warning*/
	Stroke();
	fprintf(out, "%d %d M     %d %d %f %f %f arc\n",
		0, y00,     xc,yc, radius, theta1, theta2);
	stroked = 0;
	}
/* ********************************************************************	*/

/* FUNCTION circle */
	 /* Draw a circle */
void circle(int x, int y, int r) {
	Stroke();
	fprintf(out, "%d %d M   %d %d %d %d %d arc\n",
			x+r, y, x, y, r, 0, 360);
	stroked = 0;
	}
/* ********************************************************************	*/

/* FUNCTION point */
	 /* Draw a point (filled circle of unspecified radius) */
void point(int x, int y) {
	circle(x,y,1);
	fill(1);
	}
/* ********************************************************************	*/
/* ********************************************************************	*/
static int colors[3] = {0,0,0};

/* FUNCTION SetColor */
	 /* color(0,0,0)==black,   color(0xffff,0xffff,0xffff)==white	*/
void SetColor(int red, int green, int blue) {
	Stroke();
	if (colors[0]!=red || colors[1]!=green || colors[2]!=blue) {
	   if ( (red   == 0 || red   > 0xFF00) &&	/* 0 or ~1?	*/
		(green == 0 || green > 0xFF00) &&
		(blue  == 0 || blue  > 0xFF00)) {
	      if (red)   red   = 0xFFFF;		/* Round up	*/
	      if (blue)  blue  = 0xFFFF;
	      if (green) green = 0xFFFF;
	      fprintf(out, "%d %d %d setcolor\n",	/* Use int	*/
		(colors[0]=red) ?   1:0,  /* int: less chars than float	*/
		(colors[1]=green) ? 1:0,
		(colors[2]=blue) ?  1:0);
	   } else
	      fprintf(out, "%.2f %.2f %.2f setcolor\n",
		(colors[0]=red)  /(float)0xFFFF,
		(colors[1]=green)/(float)0xFFFF,
		(colors[2]=blue) /(float)0xFFFF);
	   }
	}
/* ********************************************************************	*/

/* FUNCTION SetLineWidth */
	 /* Extension.  Line widths, independent of scaling params(?)	*/
void SetLineWidth(int width) {
	static int prev_width = -1;
	Stroke();
	if (prev_width != width)
	   fprintf(out, "%d lw\n", 10*(prev_width=width));
	}
/* ********************************************************************	*/

void linemod(char *s) {
	static char prev_line;		/* Use 4th char (is unique)	*/

	Stroke();
	if (*(s+3) == prev_line)
	   return;

	switch (*(s+3)) {
	 case 'i':				/* solid	*/
		fputs("[] 0 setdash\n", out);
		break;
	 case 'g':				/* longdashed	*/
		fputs("[15 34] 40 setdash\n", out);  /* Was 7 9	*/
		break;
	 case 'c':				/* disconected	*/
		fputs("[1 15] 17 setdash\n", out);
		break;
	 case 'd':				/* dotdashed	*/
		fputs("[11 2 1 2] 17 setdash\n", out);
		break;
	 case 't':				/* dotted	*/
		fputs("[4 6] 12 setdash\n", out);
		break;
	 case 'r':				/* shortdashed	*/
		fputs("[2 14] 17 setdash\n", out);
		break;
	 default:
	   fprintf(stderr, "Unrecognizable line pattern <%s>\n", s);
	 }

	prev_line = *(s+3);			/* Set to 4th char	*/
	return;
	}
/* ********************************************************************	*/

/* FUNCTION fill */
	 /* 0:no fill  1: black fill   0xffff: white fill	*/
	 /* Instead of using black level, use color saturation:	*/
	 /*   1:use current color   0x000f: VERY light current color	*/
	 /*  FILL LEVELS OTHER THAN 0 and 1 ARE NOT IMPLEMENTED!	*/
	 /* color(0,0,0)==black,   color(0xffff,0xffff,0xffff)==white	*/
void fill(int level) {
	if (level == 0)				/* NO fill		*/
	   fputs("S\n", out);
	else if (level == 1)			/* Current color fill	*/
	   fprintf(out, "fill\n");
	else			/* Diluted fill (eg, red --> pink)	*/
	   fprintf(out, "fill\n");		/* NOT IMPLEMENTED	*/
	stroked = 1;
	}
/* ********************************************************************	*/
/* ********************************************************************	*/

/* FUNCTION Stroke */
static void Stroke() {
     if (! stroked) {
	fputs("S\n", out);
	stroked = 1;
	}
     }
/* ********************************************************************	*/
/* ********************************************************************	*/

/* SUB-FILE font */
     /* Deal with fonts, strings */
/* ********************************************************************	*/

/* FUNCTION label */
	 /* Print a text string on the screen */
void label(char *s) {
	fprintf(out, "(%s) sh\n", s);
	stroked = 1;
	}
/* ********************************************************************	*/

char font_name[80] = "Helvetica-Bold";
int text_rotate = 0;			/* Degrees of text rotation	*/
int font_size    = 20000.;
double font_scaling = 1.;

/* FUNCTION Set_font */
	 /* Tell postscript to install new font(size)	*/
	 /* Correct so font size is independent of page scaling	*/
static void Set_font() {
	if (.9*font_size/scale <= 0)
  	   fprintf(out,"/%s findfont 1 sf\n", font_name);
	else
  	   fprintf(out,"/%s findfont %0.f sf\n", font_name, .9*font_size/scale);
   	}
/* ********************************************************************	*/

/* FUNCTION fontsize */
	 /* Set font size to size*/
	 /* BUG: if rescaled since last fontsize() call, size can be the same
	    but can end up wrong */
void fontsize(int size) {
  if (font_size != size) {
     font_size = size;
     Set_font();
     }
  }
/* ********************************************************************	*/

/* FUNCTION fontname */
	 /* Postscript wants cap'd names; plot lib doesn't. Take either	*/
void fontname (char *s) {
  char *dash;
  if (*s >= 'a' && *s <= 'z')			/* Capitalize 1st letter*/
     *s = *s - 'a' + 'A';
  dash = strchr(s, '-');				/* Find a dash	*/
  if (dash++ != NULL)
     if (*dash >= 'a' && *dash <= 'z') 
         *dash = *dash - 'a' + 'A';			/*  Cap letter!	*/
  if (strcmp(font_name, s) != 0) {			/* Change?	*/
     strcpy(font_name, s);
     Set_font();
     }
  }
/* ********************************************************************	*/

/* FUNCTION rotate */
	 /* Set text rotation angle.  Note: w and h are ignored! */
void rotate (int w, int h, int angle) {
  if (w==1000*h) angle++;	/* Dummy; prevent 'unused parameter' warning*/
  text_rotate = angle;
  }
/* ********************************************************************	*/

/* FUNCTION alabel */
	 /* label with x_justify l(left)/r/c; y_justify b(bottom)/c/t */
void alabel (int x_justify, int y_justify, char *s) {

  if (text_rotate)
     fprintf(out, "%d rotate\n", text_rotate);

  if (x_justify != 'l' || y_justify != 'b') {
     switch (y_justify) {
	/* NEED TO USE charpath (with bool == FALSE, I think) then pathbbox */
    	case 'b': /* bottom */
	   /* fputs("pop ", out);		* Pop y size off stack	*/
      	   break;
    	case 'c': /* centered */
	   /* fputs("2 div 0 exch rmoveto ", out); * Move up by 1/2	*/
      	   break;
    	case 't': /* top */
	   /* fputs("0 exch rmoveto ", out); 	* Move up full height	*/
      	   break;
    	}
     switch (x_justify) {
    	case 'c': 				    /* centered		*/
	   fprintf(out, "(%s) stringwidth\n", s);   /* x,y size on stack*/
	   fputs("pop 2 div neg 0 rmoveto ", out);  /* Move to left	*/
      	   break;
    	case 'r': 				   /* right justified	*/
	   fprintf(out, "(%s) stringwidth\n", s);  /* x,y size on stack	*/
	   fputs("exch neg exch rmoveto ", out);	   /* Left full width	*/
      	   break;
    	case 'l': /* left justified */
      	   break;
    	}
     }

  label(s);
  if (text_rotate)
     fprintf(out, "%d rotate\n", -text_rotate);		/* Restore	*/
  }
/* ********************************************************************	*/
/* ********************************************************************	*/

/* FUNCION Report_Info */
	/* Should be replaced with "Postscript_File_Comment()"	*/
static void Report_Info() {
	extern char *Report_Anal_Header_Info(int line);
	extern char *Report_Layout_Info();
	extern char *Report_Histo_Info();
	extern char *Get_CallCommand();

	fprintf(out, "\n\n");
	fprintf(out, "%%%%    DETAILS OF FILE CREATION   %%%%\n");
	fprintf(out, "%%%%     %s\n", Report_Anal_Header_Info(1));
	fprintf(out, "%%%%     %s\n", Report_Layout_Info());
	fprintf(out, "%%%%     %s\n", Report_Histo_Info());
	fprintf(out, "%%%%     %s\n", Get_CallCommand());
	fprintf(out, "\n\n");
	}
/* ********************************************************************	*/
