/* FILE main */
     /* Core of analysis program */

#include "deffs.h"
#include "config.h"
#include <fcntl.h>					/* Open/close	*/
#include <ctype.h>					/* isalpha()	*/
#include <string.h>
#include <stdlib.h>					/* atoi()	*/
#include <sys/resource.h>				/* setrlimit()	*/
#include <errno.h>
#include <time.h>					/* srand()	*/

#define MALLOC	1	/* malloc() not working on eye-hand! LHS 10/05	*/

#define	DEBUG	0

/* PUBLIC								*
 *  main								*
 *  Get_CallCommand		Snapshot of current cmdline		*
 *  Report_Anal_Header_Info	Snapshot of program constants		*
 *									*
 * PRIVATE								*
 *  Do_File		Process one file				*
 *  Do_Options		Process cmdline options				*
 *  Init_Options	(Re-) Init cmdline options			*
 *  Parse_Options	Break up cmd line into option-sized chunks	*
 */


static int Output = 0;          /* For #define of types, see deffs.h	*/
static int BatchFile = 0;	/* Passing file with list of files	*/
static int SET_INTERVAL_BY_PEAK_VARIANCE = 0;			/* Flag	*/
static int WriteFlag = 0;		/* Write data to a new file?	*/
static int ReportFlag = 0;		/* Report stack,class,etc	*/
static int Skip_BadIntervalFlag = 0;	/* Rather than truncating intvl	*/
static int Skip_BadIntervalExtra = 0;	/*  Check beyond end of interval*/
static char *OutputFileSuffix;
static FILE *OwnStream = NULL;		/* Vs pipe to stdout		*/
static int ConcatenateFiles = 0;	/* Put them all together?	*/

static char CallCommand[60000];

static void Do_File(char *file);
static int  Do_Options(int arrc, char **argv);
static void Init_Options();
static void Parse_Options(char *entry, int *argc, char ***argv);
static void Report(char *file, int value);
/* ********************************************************************	*/

/* FUNCTION main */
	 /* Analyze data */
int  main(int argc, char **argv) {
	int i;
	int args_left;				/* Replaces argc	*/

	if (argc==1) {
	   Print_Usage();
	   exit(0);
	   }

	{
	/* Set limit on core dump size in case of error	*/
	/* (Speeds up core dump, and we don't use cores)*/

  	extern int errno;                      /* see man 2 intro      */
        struct rlimit  our_limit;              /* see man getrlimit    */
	struct rlimit *our_ptr = &(our_limit);

	our_ptr->rlim_cur = 0;			/* Max core size, bytes */
	our_ptr->rlim_max = 0;
	if (setrlimit(RLIMIT_CORE,our_ptr) != 0)
	   fprintf(stderr, "\nErr %d in setting core limit", errno);
	}

	srand(time(NULL));			/* Random numbers	*/

	Read_User_Config_File();		/* Get user defaults	*/

	for (i=0; i<argc; i++) {		/* Save calling string	*/
	   strcat(CallCommand, *(argv+i));	/* One arg at a time	*/
	   strcat(CallCommand, " ");		/* Spaces between args	*/
	   }

	Init_Options();				/* Set default state	*/
	args_left =				/* Save for !batch	*/
	 Do_Options(argc, argv);		/* Set global options	*/


	if (ConcatenateFiles) {
#	   include <libgen.h>			/* basename()		*/
#	   define CAT_SIZE 30000
	   char CatCmd[CAT_SIZE]; /* cat file1 ... fileN > /tmp/fileN.concat	*/
	   char CatBaseName[300];	/* Base file for concat'd files	*/
	   char CatFileName[300]; 	/* Prepend /tmp, add '.concat'	*/

	   argv += argc - args_left;		/* Skip over arguments	*/
           if (args_left > CAT_SIZE/20)
               Exit("Too many files", "main.c");
	   sprintf(CatCmd, "cat ");

	   if (BatchFile) {		/* Pull filenames from a file	*/
	      char *entry;  		/* Ptr to batch file entries	*/
	      Open_BatchFile();				/* Open: input.c*/
	      entry = Next_BatchFile_Entry();  		/* Ptr to ~	*/
	      sprintf(CatBaseName, "%s", basename(entry));
      						      /* Use 1st file	*/
	      do {
	         int ii;
	         for (ii=0; entry[ii]; ii++)
		   if (entry[ii] == ' ')
		     Exit("Cannot have arguments in batch with -mf", "main");
	         strcat(CatCmd, entry);  /* Add file to call*/
                 strcat(CatCmd, " ");
	         entry = Next_BatchFile_Entry();  	/* Ptr to ~	*/
	        } while (entry != NULL);		/* EOF: all done*/

	    } else {			/* Pull filenames from cmdline	*/
	      while (args_left-- > 0) {		/* Walk thru files	*/
               strcat(CatCmd, *(argv++));
               strcat(CatCmd, " ");
	       }

	      sprintf(CatBaseName, "%s", basename(*--argv)); /*last file*/
	      }

	   sprintf(CatFileName, "/tmp/%s.concat", CatBaseName);
               		/* Concats all files into /tmp/file.concat 	*/
	   sprintf(CatCmd, "%s > %s", CatCmd, CatFileName);
	   /* fprintf(stderr, "%s\n", CatCmd); */	/* Debug	*/
	   if (strlen(CatCmd) > CAT_SIZE)
	      Exit("Cat command is too long", "main.c");
	   system(CatCmd);
 	   Do_File(CatFileName);
	   sprintf(CatCmd, "rm -f %s", CatFileName);
	   /* -f: file already rm'd by macro that re-calls grab (lfp.c)?*/
	   system(CatCmd);
	} else if (BatchFile) {
	   char *CallCommandEnd = CallCommand + strlen(CallCommand);
	   char **LocalArgv;			/* From batchfile	*/
	   int    LocalArgc;			/* Apply to only 1 file	*/

	   Open_BatchFile();				/* Open: input.c*/

	   while (1) {
	      /* extern char *Next_BatchFile_Entry(); */
	      char *entry = Next_BatchFile_Entry();  /* Ptr to ~	*/

	      if (entry == NULL)		/* EOF: all done	*/
		 break;
	      strcpy(CallCommandEnd, entry);  	/* Add opts,file to call*/
	      if (DEBUG) {
	         fprintf(stderr, "Complete call is %s\n", CallCommand);
	         fprintf(stderr, "Local CmdLine is %s\n", entry);
		 }
	      
			/* Init/Do_Options redundant 1st time thru loop	*/
	      Init_Options();			/* Re-init all options	*/
	      Do_Options(argc, argv);		/* Reset global options	*/
	      Parse_Options(entry, &LocalArgc, &LocalArgv); /* Set local*/
	      Do_Options(LocalArgc, LocalArgv); /* Set local options	*/
	      if (ConcatenateFiles)
		  fprintf(stderr, "Must write code to concat w/in batch\n");
	      Do_File(*LocalArgv);		/* 1st arg = filename	*/
	      if (ConcatenateFiles)
		  ConcatenateFiles = 0;
	      }
	} else {
	      argv += argc - args_left;	/* Skip over arguments	*/
	      while (args_left-- > 0)		/* Walk thru files	*/
	         Do_File(*(argv++));
	      }

	exit(0);	/* _exit() sometimes avoids seg fault ???!	*/
	}
/* ********************************************************************	*/

/* FUNCTION Do_File */
	 /* Process one file	*/
static void Do_File(char *file) {
	int File = Open_InputFile(file);

	if (File <= 0)				/* Open & read first hdr*/
	   return;				/* Go away if open fails*/
	if (ReportFlag) {
	   Report(file, ReportFlag);
	   return;
	   }

	if (WriteFlag) {			/* Process& rewrite file*/
	   Rewind_InputFile();			/* Insurance		*/
	   if (WriteFlag != 3)			/* Odd special case	*/
	      Open_OutputFile(file, OutputFileSuffix);

	   InitAlignTimes();			/* If >1 file being done*/
	   Set_Alignment_From_CmdLine("");	/* No fail on error	*/
	      	/* Like -ax: exclude trials from output file if align er*/

	   if (WriteFlag >= 2) {
	      while (Read_Next_Trial(SKIP_DATA)) {
	          if (Set_AlignTime() == SKIP_TRIAL)	/* Valid align?	*/
		     continue;				/* No! Skip it	*/

	          UnRead_Data(File);	/* Expected by write_output	*/
		  if (WriteFlag == 3)
		     Force_Zero_SkipTime(1);
	          if (Output & (TEXT_MACRO))		/* 300's: sort!	*/
	             if (Do_Macro(file) == 0) {		/* Do sort macro*/
			Skip_Over_Data();
			continue;
	      		}
		  if (WriteFlag != 3)
   	             Write_OutputFile();		/* Reread &write*/
	          }
	      Close_InputFile();
	      Close_OutputFile();
	      return;
	      }

	   while (Read_Next_Trial(SKIP_DATA)) {
	       int Frames = FrameCount_Header();	/* Can get chg'd*/
	       int Spikes = SpikeCount_Header(-1);	/* - by Get_Algn*/
	       int Touch1 = TouchCount_Header(1);	/* - so save!	*/
	       int Touch2 = TouchCount_Header(2);	/* - so save!	*/
	       int Button = ButtonCount_Header();

	       int align = Set_AlignTime();		/* Can chg hdr!	*/

	       if (align == SKIP_TRIAL)			/* Valid align?	*/
		  continue;				/* No! Skip it	*/
	       if (Bad_Interval(align, 0)) {		/* Check -i args*/
		  Warning("Skipping trial - missing requested interval");
		  continue;				/* Bad - Skip it*/
	          }	/* This really should be in Read_Next_Trial()	*/

	       Set_FrameCount_Header(Frames);
	       Set_SpikeCount_Header(Spikes);
	       Set_TouchCount_Header(1, Touch1);
	       Set_TouchCount_Header(2, Touch2);
	       Set_ButtonCount_Header(Button);

	       UnRead_Data(File);	/* Expected by write_output	*/

	       if (Output & (TEXT_MACRO))		/* e.g.,'sort92'*/
	          Do_Macro(file);			/* Do the macro	*/
   	       Write_OutputFile();			/* Reread &write*/
	       }
	   Close_InputFile();
	   Close_OutputFile();
	   return;
	   }

	Set_TrialTypes();			/* Do before all else!	*/

	if (SET_INTERVAL_BY_PEAK_VARIANCE) {	/* Reset int'vl & redo!	*/
	   Init_Histogram();
	   while (Read_Next_Trial(WITH_DATA))
	      Cumulate_Interval_And_Histogram();
	   Set_Interval_By_Peak_Variance();
	   Rewind_InputFile();
	   }
	Init_Histogram();	/* Do for ALL outputs; FOLLOWS Set_Intvl*/
	Init_Intervals();

	if (!(Output & TEXT_MACRO))	/* Added 1-22-2015; I think is ok*/
	   Init_Graphics();	/* Used by histo AND macros; init stuff	*/
		/* If the 'if' is ok, delete comment about 'used by macros'*/

	if (!(Output & (TEXT_MACRO|SPECIAL_GRAPHICS_MACRO))) {
	   Begin_Graphics(file, OwnStream);	/* Open ps* file	*/
	   if ((Output & XY_INDIVIDUAL_PLOTS) == XY_INDIVIDUAL_PLOTS)
	       Init_XY_Plot();
	   Layout_Trials(Output);	/* (row,col),axes,etc of @ fig	*/
	   Do_Class_Titles(Output);
	   Do_Header_Title();
	 } else if (!(Output & TEXT_MACRO)) {	/* Added 'if' 1-22-2015	*/
	   Begin_Graphics(file, OwnStream);	/* Open ps* file	*/
	   }

	Rewind_InputFile();
	while (Read_Next_Trial(WITH_DATA)) {

	   if (Skip_BadIntervalFlag && 			/* Check -i args*/
	    Bad_Interval(Get_AlignTime(), Skip_BadIntervalExtra)) {
	       Warning("Skipping trial - missing requested interval");
	       continue;                             /* Bad - Skip it*/
	       }

	   if (Output & INDIVIDUAL_PLOTS) {
	      if ((Output & XY_INDIVIDUAL_PLOTS) == XY_INDIVIDUAL_PLOTS)
	         Do_Individual_XY_Plot();	/* Behavioral data, XvsY*/
	      else
	         Do_Graphics();			/* All data, verus time	*/
	      }
	   Cumulate_Interval_And_Histogram();	/* Used by most outputs	*/
	   }

	Calc_Spikes_Per_Interval();		/* Used by most outputs	*/

	Rewind_InputFile();			/* Go back to 1st trial	*/
	Read_Next_Trial(WITH_DATA);


	if (!(Output & (TEXT_MACRO|SPECIAL_GRAPHICS_MACRO))) {
	   Do_Vertical_Lines_On_Graph();
	   Do_Histograms(Output);
	   }

	if (Output & (TEXT_MACRO | GRAPHICS_MACRO | SPECIAL_GRAPHICS_MACRO))
	   Do_Macro(file);			/* And rm if no data	*/
	if (!(Output & TEXT_MACRO))
	   End_Graphics();

	Close_InputFile();
	return;
	}
/* ********************************************************************	*/

/* FUNCTION Get_CallCommand */
	 /* Return pointer to calling command */
char *Get_CallCommand() { return(CallCommand); }
/* ********************************************************************	*/
/* ********************************************************************	*/

/* FUNCTION Do_Options */
	 /* And return number of options REMAINING (current argc value)	*/
static int Do_Options(int argc, char **argv) {
   int set_interval_by_cmdline = 0;		/* 'v' and 'i' clash?	*/

   while (argc-- > 1 && (**(++argv) == '-')) {
      int arg1 = -1, arg2;
      char ch;
      int args = sscanf(*argv, "-%c%d:%d", &ch, &arg1, &arg2);
      char temp;
      char *this_arg = *argv + 2;	/* Point to 2nd letter after '-'*/

      if (DEBUG == 2)
         fprintf(stderr, "Next arg is %s (count = %d)\n", *argv, argc);

      switch (ch) {
	 case 'b': BatchFile = 1;
		   if (*this_arg != 0)			/* If non-blank	*/
		     Set_BatchFile_Name(this_arg);   	/* Get name	*/
		   break;


	 case 'i': Set_Interval_From_CmdLine(arg1, arg2, 1);
		   /* Output |= SUMARY_PLOT;	 * Implied by 'i'	*/
		   if (! (arg1==0 && arg2 == 0))	/* Not if both 0*/
		      set_interval_by_cmdline = 1;
		   break;
	 case 'I': Set_Interval_From_CmdLine(arg1, arg2, 2);
		   /* Output |= SUMARY_PLOT;	 * Implied by 'I'	*/
		   if (args < 3) {
		      fputs("Use 'B' to set bg using only ending time", stderr);
		      Exit("'I' wants both begin AND end of 2nd interval",
		      		"Do_Options");
		      }
		   break;

	 case 'B': if (args > 2) {
		      fputs("Use 'I' to set begin/end of 2nd interval", stderr);
		      Exit("'B' wants just an ENDING time for bg interval",
				"Do_Options");
		      }
		   if (args == 1)
		      Exit("'B' wants an ENDING time for bg interval",
				"Do_Options");
	           Set_bg_Interval_From_CmdLine(arg1);
		   break;
		   
          case 'v': switch (arg1) {
		     case 1:  SET_INTERVAL_BY_PEAK_VARIANCE = 1; break;
		     case 2:  Set_MonteCarlo_TrialType();           break;
		     default: SET_INTERVAL_BY_PEAK_VARIANCE = 1; break;
		     }
		    break;

	 case 'X':		/* X: skip all BUT these; x:skip these	*/
	 case 'x': { char type = 't';		/* -x3 : trial (default)*/

		   if (args == 1)		/* eg, -xs3:5,not -x2:4	*/
		      args = sscanf(this_arg,"%c%d:%d",&type,&arg1,&arg2);
						/* this_arg:Strip -x,X	*/

		   if (type == '=') {
		      Seen_Skip();
		      break;
		      }

		   if (type == 'e') {		/* Special: event	*/
		     int EventType,from=FAIL,to=FAIL,which=TIME,occur=1,stack=0;
		      EventType = EventNameToNumber(++this_arg);

		      while (*this_arg!='.' &&	/* Advance past type	*/
			     *this_arg!= 0  && 	/* End of the word	*/
			     *this_arg!=',' &&
			     *this_arg!='#' &&
			     *this_arg!='_')
			 this_arg++;

		      if (*this_arg == '_') {	     /* Range given:	*/
		         args = sscanf(++this_arg,"%d:%d",&from,&to);
		         if (args == 0)
			    Exit("Bad -xe syntax", "Do_Options");
		         if (args == 1)
		            to = from;		/* Set range = 1 value	*/
		         while (isdigit(*(++this_arg))) ; /* Skip arg1	*/
			 if (*this_arg == ':')		  /* Skip :   ?	*/
		            while (isdigit(*(++this_arg))) ; /*Skip arg2?*/
 
			 if (*this_arg == ',') {
			    which = atoi(++this_arg);
		            while (isdigit(*(++this_arg))) ; /* end of #*/
			    }
			 }

		      if (*this_arg == '.') {
			 occur = atoi(++this_arg);
		         while (isdigit(*(++this_arg))) ; /* to end of #*/
		         }

		      if (*this_arg == '#')
			 stack = atoi(++this_arg);
		      Store_EventSkip(ch,EventType,from,to,which,occur,stack);
		      break;
		      }

		   if (strchr(*argv, ',') != NULL) {
		      fprintf(stderr, "\n',' not allowed in x field:\n\n");
		      Print_Help('x');
		      exit(0);
		      }

		   if (type == 'b') {		/* Special: stack.class	*/
		      args = sscanf(this_arg,"%c%d.%d",&type,&arg1,&arg2);
		      if (args != 3) {
		         fprintf(stderr, "-xb and -Xb syntax is: -xb56.3\n\n");
			 Print_Help('x');
			 exit(0);
			 }
		      }

		   if (type == 'i') {		/* Special:interval flag*/
		      /* Skip if trial does not contain full interval	*
		       * Can add extra time, too (e.g., -xi500)	*/
		      /* Set_Alignment_From_CmdLine("x");	*/
		      Skip_BadIntervalFlag = 1;
		      if (args == 1)
			 Skip_BadIntervalExtra = 0;
		       else if (args == 2)
			 Skip_BadIntervalExtra = arg1;
		       else
			 Exit("-xi can have either one or no arguments",
			      "Do_Options");
		      break;
		      }

		   if (args == 2)		/* If ==3, have RANGE	*/
		      arg2 = arg1;		/* Set range = 1 value	*/
		   if (ch == 'x')
		      Store_Skip(type, arg1, arg2);
		   else			/* NOTE: don't mix 'x' and 'X'!	*/
		      Store_No_Skip(type, arg1, arg2);
		   if (type != 't')		/* if 's', 'c' or 'b'	*/
		      Seen_Skip();		/* skip.c: ordering!	*/
		   break;
		   }


	 case 'a': Set_Alignment_From_CmdLine(this_arg);
		   break;
	 case 'r': Set_Show_Raster_FromCmdLine(arg1);
		   break;
	 case 'f': Set_Unfilled_Histos_From_CmdLine();
		   break;			/* Default is filled	*/
	 case 'l': Add_Vertical_Lines_On_Graph(arg1);
		   break;

	 case 'd': Set_dataValue_From_CmdLine(arg1);
		   break;		/* Used by various macros (-o)	*/

	 case 'D': Set_DataValue_From_CmdLine(arg1);
		   break;		/* Used by various macros (-o)	*/

	 case 'o': if (args > 1)	/* number after -o: macro number*/
		      Output = Set_Macro_Output(arg1);
		   else {		/* letter after -o: macro param	*/
		      int arg3;
		      args = sscanf(this_arg,
				      "%c%d:%d:%d", &ch,&arg1,&arg2,&arg3);
		      if (args == 1)		/* Just the flag	*/
			 Set_oValue(1, ch, 1);	/*  Set to '1'		*/
		      if (args >= 2)
			 Set_oValue(1, ch, arg1);
		      if (args >= 3)
			 Set_oValue(2, ch, arg2);
		      if (args == 4)
			 Set_oValue(3, ch, arg3);
		      }
		   break;

	 case 'g':  switch (arg1) {
			 case 1:  temp = XY_INDIVIDUAL_PLOTS;	goto SET;
			 case -1:	/* Default: no value given	*/
	 	 	 default: temp = INDIVIDUAL_PLOTS;	goto SET;
			 }
		
	 case 'G':  switch (arg1) {
			 case 1:  temp = VARIANCE_SUM_PLOT;	goto SET;
	 		 case 2:  temp = POLAR_SUM_PLOT;	goto SET;
			 case -1:	/* Default: no value given	*/
	 	 	 default: temp = SUMMARY_PLOT;		goto SET;
			 }

	 SET:     switch (*(this_arg+(args>1))) {	/* Next char	*/
/* next line: used to be 'NULL'; chg'd on compiler err, 12-15-02 LHS	*/
		      case 0:			/* Default:	*/
		      case '+':			/* This, too	*/
			 Output |= temp;
			 break;
		      case '=':			/* Only this	*/
			 Output = temp;
			 break;
		      case '~':			/* Not this	*/
			 Output &= ~temp;
			 break;
		      default:			/* Anything else*/
			 Exit("Unknown output modifier", "Do_Options");
		      }
		   if (DEBUG)
		      fprintf(stderr, "Output now %d (hex:%x) <did %c>\n",
			  Output,Output,ch);
		   break;


	 case 'L':				/* Layout	*/
	 	   Set_Layout_From_CmdLine(this_arg);
		   break;
 	 case 't':				/* Times	*/
		   Set_Times_From_CmdLine(arg1, arg2);
		   break;

	 case 'T': Set_Title_From_CmdLine(this_arg);
		   break;
	 case 'h': Set_Histo_Scale_From_CmdLine(arg1);
		   break;


	 case 'S': Open_InputFile(*(++argv));
	           Set_Alignment_From_CmdLine("r");	/* Record begin	*/
		   if (arg1 <= 0)		/* If no #, do trial 1	*/
		       arg1 = 1;
		   while (--arg1 >= 0)		/* Go to arg1-th header	*/
	              Read_Next_Trial(SKIP_DATA);   /* Read header only	*/
		   PrintHeader();		/* Print out this stack	*/
		   Close_InputFile();
		   exit(0);

	 case 'c': 			/* CHANGE (usually in header)	*/
	     if (args == 2) {				/* Just -c#	*/
		  NoTrialNumber:
		   Open_InputFile(*(++argv));		/* Hdr config	*/
	           Set_Alignment_From_CmdLine("r");	/* Record begin	*/
		   if (args == -1)			/* no #? trial 1*/
		       arg1 = 1;
		   while (--arg1 >= 0)		/* Go to arg1-th header	*/
	              Read_Next_Trial(SKIP_DATA);   /* Read header only	*/
		   PrintDetailedHeader();		/* But no stack	*/
		   Close_InputFile();
		   exit(0);
	           }
	     if (args == 1)		/* Could be -cU3[.:]2, or -c	*/
		 args = sscanf(this_arg,"%c%d%c%d", &ch, &arg1, &temp, &arg2);
						/* this_arg:Strip -c	*/
	     if (args == -1)
		goto NoTrialNumber;

	     switch (ch) {			/* character after 'c'	*/
		 case 'U':			/* Unit number:run #	*/
		    if (arg1 > 0)
		       Set_UnitNumber(arg1);
		    if (args > 2)
		       Set_RunNumber(arg2);
		    break;
		 case 's':			/* Trial stamp		*/
		    AddToTrialStamp(arg1);	/* Ought to be in hdr	*/
		    break;
		 default: 
		    Exit("unknown -c option", "Do_Options");
		 }
             break;

	 case 's': 		 /* If '-ss6', give 2nd panel's touches */
		   if (isalpha(*this_arg))	/* -s<letter><number>	*/
		       arg1 = atoi(this_arg+1);
		   if (arg1 <= 0)		/* Print switche	*/
		       arg1 = 1;		/* If no #, do trial 1	*/
	 	   Open_InputFile(*(++argv));
	           Set_Alignment_From_CmdLine("r");	/* Record begin	*/
		   while (--arg1 >= 0)	/* Go to arg1-th header	*/
	               Read_Next_Trial(SKIP_DATA); /* Read header only	*/
		   Set_AlignTime();	/* Else data read may fail	*/
		   Read_Skipped_Data();
		   if (isalpha(*this_arg)) {		/* 2nd letter?	*/
		      PrintArm(2);		/* Board 2 touches	*/
		    } else {
		      PrintArm(1);		/* Print touches	*/
		      PrintButtons(0);		/* Print buttons	*/
		      }
		   Close_InputFile();
		   exit(0);

	 case 'H': Print_Help(*this_arg);
		   exit(0);

	 case 'R': ReportFlag = arg1;
		   break;

	case 'W':				/* Open X window	*/
		/* Redirect stdout to stdin of (exec'd) ghostview:	*/
	   /* fclose(stdout); * Make (freed) descriptor 0 (stdout) avail*/
	   OwnStream = 			/* Flag to Begin_Graphics() 	*/
	    popen(         /* Fortunately, internal 'pipe' uses fd 0!   */
              (*this_arg == 'p')  ?     /* Portrait or landscape?       */
#ifdef DIAG
"gv -nolabels -magstep -4 -portrait -geometry 100x100 -" :
"gv -nolabels -magstep -4 -landscape -geometry 100x100 -",
#else
"gv " :
"gv ",
#endif
                "w"),      /* Open new fd for writing -- uses stdout!   */
           setbuf(OwnStream, NULL);     /* Unbuffered I/O               */
           break;

	 case 'z':   		      /* Zoom width by reducing height	*/
	    {
	    extern void Set_Config_Parameter();
	    if (arg1 == 0)
	        arg1 = 2;
	    Set_Config_Parameter("HEIGHT", HEIGHT/arg1);
	    Set_Config_Parameter("HISTO_OFFSET", HISTO_OFFSET/arg1);
	    break;
	    }

         case 'e':              /* Show each trial individually(Xwindow)*/
         {
           int  AlreadyThere = 0;                       /* For searches */
           char RecallSearch[12];               /* For '/': repeat prev */

           if (Open_InputFile(*(++argv)))       /* Open, read 1st hdr   */
              Exit("Can't open 'e' file", "Do_Options");

                /* Redirect stdout to stdin of (exec'd) ghostview:      */
	   OwnStream =
            popen(
#ifdef DIAG
              (*(*(argv-1)+2) == 'p')  ?        /* Portrait or landscape*/
                   "gv --orientation=portrait  --scale=-3 " :
                   "gv --orientation=landscape --scale=-3 " ,
#endif
 "gv",
                "w");      /* Open new fd for writing -- uses stdout!   */
  
           setbuf(OwnStream, NULL);     /* Unbuffered I/O               */
           setbuf(stdout, NULL);     	/* Unbuffered I/O               */

           Set_Show_Raster_FromCmdLine(-1);
           Rewind_InputFile();
           if (!SHOW_SINGLES)
              fprintf(stderr, "Set SHOW_SINGLES to TRUE!\n");

           /* CONNECT APPROPRIATE GRAPHICS DEVICE TO STDOUT */
           openpl(OwnStream);   /* If using postscript,  do prologue    */
           /* setvbuf(stdin, NULL, _IONBF, 55); * 55 is ignored */

           while (1) {
              char CmdLine[231] = "";
              int  CmdInteger = 1;
              int  GotInteger;                          /* Boolean      */
              int  i;

              if (!AlreadyThere)
                 Read_Next_Trial(SKIP_DATA);

              Init_Histogram();
              Set_TrialTypes_For_Single_Trial();  /* Set trial to type 0*/
              Init_Graphics();  /* Used by histo & therefor by ascii out*/

	      Read_Skipped_Data();
	      erase();					/* Clear screen	*/

	      Layout_Single_Trial();	/* (row,col),axes,etc of @ fig	*/
	      if ((Output & XY_INDIVIDUAL_PLOTS) == XY_INDIVIDUAL_PLOTS) {
	         Init_XY_Plot();
		 Do_Individual_XY_Plot();	/* Behavioral data, XvsY*/
		 Do_Individual_XY_Plot();	/* Behavioral data, XvsY*/
	      } else {
	         Do_Graphics();
	         Do_Graphics();
	         }
	      Cumulate_Interval_And_Histogram();
	      Do_Single_Trial_Title();
	      Do_Histograms(Output);
	      Do_Vertical_Lines_On_Graph();
	      Do_Histograms(Output);		/* Multiples:make it show*/
	      Do_Vertical_Lines_On_Graph();
	      Do_Histograms(Output);
	      Do_Vertical_Lines_On_Graph();
	      Do_Histograms(Output);
	      Do_Vertical_Lines_On_Graph();

	      fflush(stdout);	/* Doesn't work: ghostview bufs input?	*/
	      fflush(OwnStream);	/* Hence the multi calls above	*/

	      for (i=0; i<4; i++)
		 *(CmdLine+i) = 0;

	      fgets(CmdLine, 230, stdin);
	      GotInteger = sscanf(CmdLine+1, "%d", &CmdInteger);

	      switch (GotInteger) {
		 case -1:				/* No 2nd char	*/
		    CmdInteger = 1;			/*  & fall thru	*/
		 case  1:				/* Found an int	*/
		    break;
		 case 0:				/* No match	*/
	            GotInteger = sscanf(CmdLine+2, "%d", &CmdInteger);
	            if (GotInteger != 1)
		       CmdInteger = 1;
		    break;
		 } 

	      AlreadyThere = 0;

	      switch (*CmdLine) {
	         case 'q':
	         case 'x':
		 case '':  goto EXIT;

		 case 0:     break;

		 case 's':   if (CmdInteger < 0) {
				CmdInteger += TrialNumber_Input();
		 		Rewind_InputFile();
				}

			     while (CmdInteger-- > 0)
				Read_Next_Trial(SKIP_DATA);
			     AlreadyThere = 1;
			     break;

		 case '/':   
		 case 'n':
		    if (*(CmdLine+1) == '\n') {	/* '/' alone?	*/
		 	strcpy(CmdLine, RecallSearch);	 /* Get	*/
	            	GotInteger = sscanf(CmdLine+2, "%d", &CmdInteger);
	            	if (GotInteger != 1)
		       	    CmdInteger = 1;
		     } else
		        strncpy(RecallSearch,CmdLine,10);/* Save */

		     while (1)  {
			int trial = Read_Next_Trial(SKIP_DATA);
			if (trial == 0) {
			      fputs("Not found\n", stderr);
	 		      Rewind_InputFile();
			      goto NOT_FOUND;
			      }
			switch (*(CmdLine+1)) {
			   case 't': if (trial == CmdInteger)
					goto FOUND;
				     break;
			   case 's': if (StackNumber_Header()==CmdInteger)
					goto FOUND;
				     break;
			   case 'c': if (TableNumber_Header()==CmdInteger)
					goto FOUND;
				     break;
			   }
			}
		     NOT_FOUND:
			break;
		     FOUND:
			AlreadyThere = 1;
			break;

		 /* BETTER: search for stack.column type, e.g.,
		    /4   : goto next stack 4
		    /4.2 : goto next stack 4, column 2
		    /.2  : goto next column 2
		    /[47].[2:4] : goto next stack 4 or 7, col 2,3 or 5
		    NOTE: goto top of while loop *
		    */
		 case '':  	erase();
				break;
		 case 'B': 	Rewind_InputFile();
				break;
		 case 'h': case '?':

				fputs(
"ESC:   exit                                 CR:    next\n",
				stderr); fputs(
"  B:   goto beginning (trial 1)            / or n: repeat previous search\n",
				stderr); fputs(
" ^L:   clear                               /t#:    find trial #\n",
				stderr); fputs(
" +#:   show next # trials NO!              /s#:    find next stack #\n",
				stderr); fputs(
" s#:   skip next # trials (may be <0)      /c#:    find next class #\n",
				stderr);
			break;
		 default:    break;
		 }
	      }

	   EXIT:
	   Close_InputFile();
	   /* pclose(OwnStream);	 * Close graphics pipe: hangs!	*/
	   exit(0);
	   }


	 case 'm': switch (*this_arg) {
		     case 'f':
			ConcatenateFiles = 1;
			break;
		     case 'c':
		        Set_MergeTables(*argv+3);	/* strip 2 char	*/
		        break;
		     case 's':
		        Set_MergeStacks(*argv+3);	/* strip "-m"	*/
			break;
		     case '=':
			break;				/* Seen merge	*/
		     default:
		        Exit("Unknown merge modifier", "Do_Options");
		     }
		   if (*this_arg != 'f')  /* merge files does not count	*/
		      Seen_Merge();	  /* skip.c: set priority rule	*/
		   break;

	 case 'M':  Print_Macro_Numbers(arg1);
		    break;

	 case 'C': { 		/* Print or set one or all parameters	*/
		    char *equal = index(this_arg, (int)'=');/* Ptr to 1st =*/
		    if (equal != 0) {			/* Found an '='	*/
		        extern void Set_Config_Parameter(char *name, int value);
			arg1 = atoi(equal+1);		/* Get value	*/
			this_arg[equal-this_arg] = 0;	/* Truncate at =*/
		        Set_Config_Parameter(this_arg, arg1);
		     } else
		       Print_Config_Parameters(this_arg);
		    break;
		    }

	 case 'w':  WriteFlag = 1;
	 	    if (*this_arg == '#') {
		       Set_OutputFile_TrialLimit(atoi(++this_arg));
		       while (isdigit(*(++this_arg))) ;	/* Goto end of #*/
		       }
		    if (*this_arg == '-') {
		       WriteFlag = 2;			/* No alignment	*/
		       this_arg++;
		       if (*this_arg == '-') {		/* Double dash?	*/
		          WriteFlag = 3;		/* No alignment	*/
		          this_arg++;
		          }
		       }
		    if (*this_arg)			/* suffix?	*/
		       OutputFileSuffix = this_arg;	/* Get suffix	*/
		    break;

	 case 'O':  Set_OutputFile_ShiftTime(arg1);
		    break;

	 case 'U':  	/* -Uunit or -Uunit:run ; '0' means don't change*/
		    	/* OBSELETE AS OF 2013! -- use '-cU' or '-cu' */
		    if (arg1 > 0)
		       Set_UnitNumber(arg1);
		    if (args > 2)
		       Set_RunNumber(arg2);
		    break;

	 case 'u':  Set_UnitChannel_Header(arg1);
		    break;

	 default:   Print_Usage();
		    exit(0);
	 }
      }

   if (SET_INTERVAL_BY_PEAK_VARIANCE && set_interval_by_cmdline)
       Exit("Can't have both 'i' and 'v' flags", "Do_Options");

   if (Output == NO_OUTPUT) {			/* No output selected?	*/
      if (SET_INTERVAL_BY_PEAK_VARIANCE || set_interval_by_cmdline)
          Output |= STANDARD_SUM_PLOT;
      else
          Output = INDIVIDUAL_PLOTS;		/* Default: show all	*/
      }

   return(argc);
   }
/* ********************************************************************	*/

/* FUNCTION Init_Options */
	 /* Exceptions are noted */
static void Init_Options() {
	static char DefaultSuffix[] = "+";

	OutputFileSuffix = DefaultSuffix;
        Init_Skip();				/* Set all 'n' to zero	*/
	Init_Merge();
	Set_dataValue_From_CmdLine(0);
	Set_DataValue_From_CmdLine(0);
	Clear_oValues();
	Init_Vertical_Lines_On_Graph();
	Reset_Intervals();
	Output = NO_OUTPUT;
	SET_INTERVAL_BY_PEAK_VARIANCE = 0;
	Set_Title_From_CmdLine("");		/* No title!		*/
	Set_Histo_Scale_From_CmdLine(0);	/* Turn it off		*/
	Unset_Times_From_CmdLine();		/* Reset		*/
	Set_OutputFile_TrialLimit(0);
	Set_OutputFile_ShiftTime(0);
	Set_Alignment_From_CmdLine("X");	/* Abort on align err	*/
	Set_UnitNumber(0);			/* head.c		*/
	Init_Align();

	/* NOT RESET:							*
	 * b BatchFile:  Want this flag to remain true!			*
	 * i Interval:	 Global -I or -i will automatical override next	*
	 * I Interval:   Ditto: not reset 				*
	 * a Alignment	 If not specified in global,remains set in batch*
	 * r Show_Raster: Once turned on, remains set.			*
	 * f No-fill histograms
	 * L Layout as at acquisition time AND OTHER LAYOUT OPTIONS (Lr, etc)
	 * t Set times
	 * h Histo scale
	 * mf Concatenate files
	 */
	}
/* ********************************************************************	*/

const char Separators[] = " 	";	/* A space or a tab	*/
/* FUNCTION Parse_Options */
	 /* Parse line into arguments; set local argc and argv */
static void Parse_Options(char *entry, int *argc, char ***argv) {
	static char *Arguments[40];	/* Pointers to 40 arguments	*/
	static int count = 0;		/* Remember #, free'm next call	*/
	char *next;			/* Temporarily hold each token	*/
#if (MALLOC == 0)
	static char StorageSpace[40][40]; /* Instead of using malloc	*/
#endif

#if (MALLOC)
	while (count > 0)
	   free(Arguments[--count]);
#else
	count = 0;
#endif
					/* Separators: space, tab	*/
	next = strtok(entry, Separators);	/* 1st token: filename	*/
	Arguments[0] = 
#if (MALLOC)
		malloc(1+strlen(next));	/* Alloc space		*/
#else
		StorageSpace[0];
#endif
	/* +1 : strlen ignores \0 at end, but strcpy copies the '\0'!	*/
	strcpy(Arguments[0], next);		/* Store filename as 0	*/

	while ((next=strtok(NULL,Separators)) != NULL) {/* Get next arg	*/
#if (MALLOC)
	    Arguments[++count] = malloc(1+strlen(next));/* Alloc space	*/
#else
	    ++count;
	    Arguments[count] = StorageSpace[count];
#endif
	    strcpy(Arguments[count], next);		/* Store arg	*/
	    }


	*argc = count + 1;		/* Args (count) + filename (1)	*/
	*argv = Arguments;
	}
/* ********************************************************************	*/

/* FUNCTION Report_Anal_Header_Info */
	char *Report_Anal_Header_Info(int line) {
		static char Info[280];
		switch (line) {
		    case 1: sprintf(Info,
		      " HEIGHT %d HISTO_OFFSET %d", HEIGHT, HISTO_OFFSET);
	    case 2: break;
	    case 3: break;
	    case 4: break;
	    }
	return(Info);
	}
/* ********************************************************************	*/

/* FUNCTION Report */
	 /* Print info about stack, class, etc	*/
static void Report(char *file, int value) {
   Set_Alignment_From_CmdLine("r");	/* Record begin	*/
   Set_TrialTypes();
   switch (value) {
      case -1:		/* Default: no value given */
      case  0:
      case  1:
	while (Read_Next_Trial(WITH_DATA)) {
	   char RunStatus[20];
	   switch (RunStatus_Header()) {
	      case START_ERR:strcpy(RunStatus,"StartErr");break;
	      case MONK_ERR: strcpy(RunStatus,"RunErr");  break;
	      case SUCCESS:  strcpy(RunStatus,"Success"); break;
	      case WIGGLE_ERR:  strcpy(RunStatus,"WiggleErr"); break;
	      default: sprintf(RunStatus,"UnknownErr%d",
		 			RunStatus_Header());
	      }
	   if (value == 1)
	      fprintf(stdout, "%3d.%-3d ",
		UnitNumber_Header(), RunNumber_Header());

   	   fprintf(stdout, "%3d:  %2d %2d (%d) [%s]\n", TrialNumber_Input(),
	  	    Get_TrialType_Info(Get_TrialType(), STACK),
	  	    Get_TrialType_Info(Get_TrialType(), CLASS),
	  	    TrialNumber_Header(),
		    RunStatus);
	   }
	break;

      case 10:				/* Same as 0 but only digits	*/
   	fprintf(stdout, "unit run order stack class trial status\n");
	while (Read_Next_Trial(WITH_DATA))
   	   fprintf(stdout, "%3d %3d %3d  %2d %2d %d %d\n", 
		    UnitNumber_Header(), RunNumber_Header(),
		    TrialNumber_Input(),
	  	    Get_TrialType_Info(Get_TrialType(), STACK),
	  	    Get_TrialType_Info(Get_TrialType(), CLASS),
	  	    TrialNumber_Header(),
		    RunStatus_Header());
	break;

      case 102:
      case 2:
	{ int i;
	if (Is_Batch() || value==102)
	   fprintf(stdout, "%3d.%-3d ", UnitNumber_Header(),RunNumber_Header());
   	fprintf(stdout,  "Stacks:  %d   ", List_Length(STACK));
	for (i=0; i<List_Length(STACK);i++)
	   fprintf(stdout, " %d", List_Element(i,STACK));
	fprintf(stdout, "\n");
	if (Is_Batch())
	   fprintf(stdout, "%3d.%-3d ", UnitNumber_Header(),RunNumber_Header());
   	fprintf(stdout,"Classes: %d   ", List_Length(CLASS));
	for (i=0; i<List_Length(CLASS);i++)
	   fprintf(stdout, " %d", List_Element(i, CLASS));
	fprintf(stdout, "\n");
	if (Is_Batch())
	   fprintf(stdout, "%3d.%-3d ", UnitNumber_Header(),RunNumber_Header());
	fprintf(stdout,"Monkey:  %s\n", MonkName_Header());
	break;
	}

      case 3:
        while (Read_Next_Trial(SKIP_DATA))
           fprintf(stdout, "%3d.%-3d %3d:  (%d)\n", 
		   UnitNumber_Header(), RunNumber_Header(),
		   TrialNumber_Input(), TrialNumber_Header());
	break;

      case 4:		/* Useful for "grab -R4 * | grep ' 202 '"*/
      case 400:
	{ int i;
	if (Is_Batch())
	   fprintf(stdout, "%3d.%-3d ", UnitNumber_Header(),RunNumber_Header());
	fprintf(stdout, "%s ", file);
	for (i=0; i<List_Length(STACK);i++)
	   fprintf(stdout, " %d", List_Element(i,STACK));

	if (value == 400) {		/* Pad out to 20 values	*/
	   while (i++ < 20)
	      fputs(" 0", stdout);	/*  Use '0' to pad	*/
	   }

	fprintf(stdout, "\n");
        break;
	}

      case 5:		/* Give start time for each trial	*/
	{ float  Start;
	Rewind_InputFile();
        Read_Next_Trial(SKIP_DATA);
	Start = Time_From_Header();	/* seconds	*/
	Rewind_InputFile();
        while (Read_Next_Trial(SKIP_DATA))
           fprintf(stdout, "%4d:  %12s  %6.1f sec (%3d min %2d sec)\n", 
		   TrialNumber_Header(),
		   Time_Header(),
		   Time_From_Header() - Start,
		   (int)((Time_From_Header() - Start)+30) / 60, /*round*/
		   (int)((Time_From_Header() - Start)+30) % 60  /*round*/
		   );
	}
	break;

      case 6:		/* Print out the DELAY/PAUSE times, per trial	*/
      	{
#	include "event.h"
        while (Read_Next_Trial(SKIP_DATA)) {
	   EVENT *E = Stack_Ptr_Header();

           fprintf(stdout, "%3d", TrialNumber_Header());
	   while ((++E)->type != END_STACK)
	       if (E->type == DELAY)
                  fprintf(stdout, "  %5d", E->two);
	   E = Stack_Ptr_Header();
	   while ((++E)->type != END_STACK)
	       if (E->type == PAUSE)
                  fprintf(stdout, "  %5d", E->three);
           fprintf(stdout, "\n");
	   }
	break;
	}

      case 7:		/* Give error type for each trial	*/
   	fprintf(stdout, "order trial errtype status time\n");
        while (Read_Next_Trial(SKIP_DATA))
           fprintf(stdout, "%4d  %4d  %3d  %7d  %5d\n", 
		   TrialNumber_Input(),
		   TrialNumber_Header(),
		   ErrType_Header(),
		   RunStatus_Header(),
		   	/* Report time re: start, unless start error	*/
		   EventExtract(ERROR_STACK_BEGIN, TIME, 1) -
		     ((RunStatus_Header()==-3) ?	/* Start error?	*/
		    	0 :				/* Negate time!	*/
		        EventExtract(START_HERE, TIME, 1))); /* re:start*/
	break;

      case 8:
        while (Read_Next_Trial(SKIP_DATA))
	   PrintHeader();
	break;

      case 9:
        while (Read_Next_Trial(SKIP_DATA))
	   fprintf(stdout, "%3d: %d %d %d %d %d %d\n",
		   TrialNumber_Header(),
		   Get_TouchBound_Header(1),
		   0,0,0,0,0);
	break;

      case 11:
        while (Read_Next_Trial(SKIP_DATA))
	   fprintf(stdout, "%d %d %d    %s\n", 
		Year_From_Header(), Month_From_Header(), Day_From_Header(),
		Time_Header());
	break;

      case 12: {
	int i;
	if (Is_Batch())
	   fprintf(stdout, "%3d.%-3d ", UnitNumber_Header(),RunNumber_Header());
	for (i=0; i<Count_TrialTypes(); i++)
	    fprintf(stdout, "%3d %2d   %4d\n",
		   Get_TrialType_Info(i, STACK),
		   Get_TrialType_Info(i, CLASS),
		   Get_TrialType_Info(i, NUMBER));
	break;
	}
      case 13: {
	int i,j;
	int sum;
	int values[40][40];
	for (i=0; i<40; i++)
	   for (j=0; j<40; j++)
	     values[i][j] = 0;
	for (i=0; i<Count_TrialTypes(); i++)
	     values[List_Index(Get_TrialType_Info(i, STACK), STACK)-1]
		   [List_Index(Get_TrialType_Info(i, CLASS), CLASS)-1] = 
		   	Get_TrialType_Info(i, NUMBER);
	if (List_Length(CLASS) > 1) {	/* Header of class#, "sum"	*/
	   fprintf(stdout, "   ");
	   for (j=0; j<List_Length(CLASS); j++)
	      fprintf(stdout, "%5d", List_Element(j, CLASS));
	   fprintf(stdout, "   sum\n");
	   }
	for (i=0; i<List_Length(STACK); i++) {
	   fprintf(stdout, "%3d", List_Element(i, STACK));
	   sum = 0;
	   for (j=0; j<List_Length(CLASS); j++) {
	       fprintf(stdout, "%5d", values[i][j]);
	       sum += values[i][j];
	       }
	   if (List_Length(CLASS) > 1)
	       fprintf(stdout, " %5d", sum);
	   puts("");					/* Line feed	*/
	   }
	break;
	}

      case 14:
        while (Read_Next_Trial(SKIP_DATA))
	   fprintf(stdout, "%d\n", EventExtract(TRIAL_STAMP, 1, 1));
	break;


      case 20:
        while (Read_Next_Trial(SKIP_DATA))
   	   fprintf(stdout, "%5d   %3d  %2d %2d %d\n", 
		    EventExtractTrialStamp(),
		    TrialNumber_Input(),
	  	    Get_TrialType_Info(Get_TrialType(), STACK),
	  	    Get_TrialType_Info(Get_TrialType(), CLASS),
	  	    TrialNumber_Header());
	break;

      case 41: fprintf(stdout, "%d\n", List_Length(STACK)); break;
      case 42: fprintf(stdout, "%d\n", List_Length(CLASS)); break;
      case 43: fprintf(stdout, "%d %d\n", 
	   			       List_Length(STACK),
	   			       List_Length(CLASS)); break;

      default:
   	   fprintf(stdout, "Unknown -R flag (%d)\n", value);
      }
   Close_InputFile();
   return;
   }
/* ********************************************************************	*/
