/* FILE layout */
     /* Lay out figures on page	*/

#include "deffs.h"
#include "config.h"
#include <stdlib.h>
#include <ctype.h>

#define DEBUG 			0	/* Levels 0 thru 2		*/
#define INVERT_ROW		0	/* Reverse order of row (usu 11)*/
#define	SWAP_ROW		0	/* Reflect (x,y => -x,-y) (usu 3*/
#define OLD_LAYOUT_FROM_STACK	0	/* Prior to about 6/96:4x3 array*/

/* ********************************************************************	*/
/*  PUBLIC:
   Layout_Trials		Call once from main: set figure locations
   Layout_Single_Trial		Used with 'e' flag: single plot
   Set_Layout_From_CmdLine	Set rows & columns
   Class_Column			Return col# for class
   Stack_Row			Return row# for stack
   Report_Layout_Info		Return string of constant names and values
   if_layout_from_stack		Return boolean (for graph.c: rasters)
   if_overlay_stacks		Return boolean (for histo.c: histo fill)
   if_overlay_classes		Return boolean (for histo.c: histo fill)
   if_rotate_layout		Return boolean (for histo.c:Summary())

 PRIVATE:
   FillFromCmdLine
   Do_LayoutFromStack
   Do_Layout_From_CmdLine
 */
/* ********************************************************************	*/

static int LayoutFromStack = 0;
static int DidLayout = 0;
static int FilledStackToRow=0, FilledTableToCol=0;	/* Flags	*/
static int OverlayStacks = 0;			/* All stacks on 1 row	*/
static int OverlayClasses = 0;			/* All classes on 1 col	*/
static int RotateLayout = 0;			/* Swap rows and cols	*/

#  define MAX_STACK_NUMBER	2 * MAX_STACKS	/* Can chg stack # to lg'r value */
static int StackToRow[MAX_STACK_NUMBER];	/* Index=stack,value=row*/
static int TableToCol[MAX_STACK_NUMBER];

static int  FillFromCmdLine(int *into, char *from);
static void FillFromList(   int *into, int   List);
static void Do_Layout_From_CmdLine(int sumflag);
static void Do_LayoutFromStack();
/* ********************************************************************	*/

/* FUNCTION Set_Layout_From_CmdLine */
	 /* Format graphical output using cmd line requests	*/
	 /* NOTES:  no more than 1 "-Ls", no more than 1 "-Lc" !!!	*/
void Set_Layout_From_CmdLine(char *CmdLine) {
	static int LayoutErr = 0;
	int i;

	if (DEBUG)
	   fprintf(stderr, "Layout from command line\n");
	if (LayoutErr)				/* Err on  prev "-L" ?	*/
	   return;				/* Then skip all "-L"	*/

	if (DidLayout == 0) {			/* First call?		*/
	   for (i=0; i<MAX_STACK_NUMBER; i++) 	/* Init all to "-1"	*/
	      StackToRow[i] = TableToCol[i] = -1;	/* -1 == unused	*/
	   FilledStackToRow = FilledTableToCol = 0;	/* Initialize	*/
	   DidLayout = 1;
	   }

	switch (*CmdLine) {
	   case 's':			/* 'Stacks' (from command line)	*/
		LayoutErr = FillFromCmdLine(StackToRow, CmdLine+1);
		FilledStackToRow = 1;
		break;	

	   case 'c':			/* 'Columns/classes' (cmd line)	*/
		LayoutErr = FillFromCmdLine(TableToCol, CmdLine+1);
		FilledTableToCol = 1;
		break;

	   case 'o':	/* Overlay stacks(classes): 1 fig / class(stack)*/
		if (*(CmdLine+1) == 'c')
		   OverlayClasses = 1;
		else				/* Default:overlay stack*/
		   OverlayStacks = 1;			/* Set flag	*/
		break;

	   case 0:			/* No char: layout as in stack	*/
		LayoutFromStack = 1;
		return;

	   case 'r':
		RotateLayout = 1;
		break;

	   default:
		fprintf(stderr, "Unknown layout key (%c)!\n", *CmdLine);
		LayoutErr = 1;
		break;
	   }

	if (LayoutErr) {
	   LayoutFromStack = 1;
	   fprintf(stderr, "Err in layout (from command line)\n");
	   }
	return;
	}
/* ********************************************************************	*/

/* FUNCTION Layout_Trials */
	 /* Front end to formatting graphical output: what figures where*/
void Layout_Trials(int Output) {
	if (Output & INDIVIDUAL_PLOTS) {	/* g, ?G/P/V o(graphics)*/
	   if (LayoutFromStack)
	      Do_LayoutFromStack();
	    else {
	      if (FilledStackToRow==0)
	         FillFromList(StackToRow, STACK);
	      if (FilledTableToCol==0)
	         FillFromList(TableToCol, TABLE);

	      DidLayout = FilledStackToRow =	/* Reset for next time	*/
			  FilledTableToCol = 0;
	      Do_Layout_From_CmdLine(Output&(SUMMARY_PLOT|GRAPHICS_MACRO));
	      }
	} else if (Output&GRAPHICS_MACRO) {	
						/* Single SUM_FIG (ROC)	*/
		/* Eventually query macro file for number of cells to add*/
		/* Till then, default to a single	*/
		/* Eventually allow it to be added to -g!	*/
	       Set_TrialType_Info(Count_TrialTypes(), CELL, 1);
	       Set_TrialType_Info(Count_TrialTypes(), ROW, 1);
	       Set_TrialType_Info(Count_TrialTypes(), COL, 1);
           Set_Dimensions_For_Coords(1, 1);
	} else if (Output & SUMMARY_PLOT) {	/* Want 2 sum-figs	*/
	   int i;				/* Set 1st SUM_FIGS	*/
	   for (i=0; i<SUM_FIGS; i++) { 	/* to cells 1:SUM_FIGS	*/
	       Set_TrialType_Info(Count_TrialTypes()+i, CELL, i+1);
	       Set_TrialType_Info(Count_TrialTypes()+i, ROW, 1);
	       Set_TrialType_Info(Count_TrialTypes()+i, COL, i+1);
	       }
	   if (FilledStackToRow==0)
	       FillFromList(StackToRow, STACK);
	   if (FilledTableToCol==0)
	       FillFromList(TableToCol, TABLE);
           Set_Dimensions_For_Coords(1, SUM_FIGS); /* 1 row,SUM_ cols	*/
	   if (DEBUG) fprintf(stderr, "Set for sum plot only (1x%d)\n",SUM_FIGS);
	} else					/* Text-only macro	*/
	   return;
}
/* ********************************************************************	*/

/* FUNCTION Layout_Single_Trial */
	 /* Set up as 1x1 */
void Layout_Single_Trial() {
        Set_Dimensions_For_Coords(1, 1);

	Set_TrialType_Info(0, CELL, 1);
	Set_TrialType_Info(0, ROW,  1);
	Set_TrialType_Info(0, COL,  1);
	}
/* ********************************************************************	*/

/* FUNCTION Stack_Row */
	 /* Return row number given stack number */
int Stack_Row(int stack) {
	return(StackToRow[stack]);
	}
/* ********************************************************************	*/

/* FUNCTION Class_Column */
	 /* Return column number given class number */
int Class_Column(int class) {
	return(TableToCol[class]);
	}
/* ********************************************************************	*/

/* FUNCTION Report_Layout_Info */
	 /* Return snapshot of constants for inclusion in output file	*/
char *Report_Layout_Info() {
	static char Info[120];
	sprintf(Info, "layout: INVERT_ROW %d SWAP_ROW %d",
		INVERT_ROW, SWAP_ROW);
	return(Info);
	}
/* ********************************************************************	*/

/* FUNCTION if_layout_from_stack */
	 /* Return boolean */
int if_layout_from_stack() { return(LayoutFromStack); }
/* ********************************************************************	*/

/* FUNCTION if_overlay_stacks */
/* FUNCTION if_overlay_classes */
	 /* Return boolean for OverlayStacks / OverlayClasses */
int if_overlay_stacks() { return(OverlayStacks); }
int if_overlay_classes() { return(OverlayClasses); }
/* ********************************************************************	*/

/* FUNCTION if_rotate_layout */
	 /* Return boolean */
int if_rotate_layout() { return(RotateLayout); }
/* ********************************************************************	*/
/* *****  PRIVATE (static) FUNCTIONS **********************************	*/
/* ********************************************************************	*/

/* FUNCTION Do_LayoutFromStack */
	 /* Set rows & columns for page & for each trial type; label cells*/
static void Do_LayoutFromStack() {
   int i;
# if OLD_LAYOUT_FROM_STACK
   int fourth_col = 0;
   int third_row = 0;

   i = -1;
   while (Get_TrialType_Info(++i, NUMBER) > 0) {	/* Survey all types	*/
      if ( !SHOW_SINGLES && Get_TrialType_Info(i,NUMBER) <= 1) { /* 1 trial?*/
	 Set_TrialType_Info(i,ROW, -1);		/* Don't plot singles	*/
	 Set_TrialType_Info(i,COL, -1);
	 fprintf(stderr, "layout.c: Skipping single (%d.%d)\n",
		Get_TrialType_Info(i,STACK), Get_TrialType_Info(i,CLASS));
      } else {					/* Use specified cell	*/
         int cell = Get_TrialType_Info(i,CELL);
	 int at;
	 if (cell == FAIL)   
	     cell = 1;

	 at = (cell%4==0) ? 4 : cell%4;		/* Col: from 1:n-th cell*/
	 if (at == 4)
	    fourth_col = 1;
         Set_TrialType_Info(i,COL, at);		/* cell is 1:n	*/

	 at = 1 + (cell-1)/4;
	 if (at == 3)
	    third_row = 1;
	 Set_TrialType_Info(i,ROW, at);

	 if (Get_TrialType_Info(i,STACK) == SWAP_ROW) {
	    Set_TrialType_Info(i,COL, 4-Get_TrialType_Info(i,COL));
	    Set_TrialType_Info(i,ROW, 4-Get_TrialType_Info(i,ROW));
	    if (DEBUG) fprintf(stderr, "SWAP stack 'SWAP_ROW'");
	    }
	 if (Get_TrialType_Info(i,STACK) == INVERT_ROW) {
	    static int HaveNotToldYou = 1;
	    Set_TrialType_Info(i,COL,1+List_Length(TABLE)-Get_TrialType_Info(i,COL));
	    if (HaveNotToldYou) {
	       fprintf(stderr, "Inverting row %d\n", INVERT_ROW);
	       HaveNotToldYou = 0;
	       }
	    }

	 if (DEBUG) fprintf(stderr, "cell %2d ==>  row %2d, col %2d\n",
	      cell, Get_TrialType_Info(i,COL), Get_TrialType_Info(i,ROW));
	 }
      }

   Set_Dimensions_For_Coords(2+third_row, 3+fourth_col);
# else
   i = -1;
   while (Get_TrialType_Info(++i, NUMBER) > 0) {/* Survey all types	*/
      if ( !SHOW_SINGLES && Get_TrialType_Info(i,NUMBER) <= 1) { /* 1 trial?*/
	 Set_TrialType_Info(i,ROW, -1);		/* Don't plot singles	*/
	 Set_TrialType_Info(i,COL, -1);
	 fprintf(stderr, "layout.c: Skipping single (%d.%d)\n",
		Get_TrialType_Info(i,STACK), Get_TrialType_Info(i,CLASS));
      } else {					/* Use specified cell	*/
         int cell = Get_TrialType_Info(i,CELL);

	 if (cell == FAIL)   
	     cell = 1;

         Set_TrialType_Info(i,ROW, (cell%3==0) ? 3:cell%3); /*Row:1:n-th cell*/
	 Set_TrialType_Info(i,COL, 1 + (cell-1)/3);

	 if (DEBUG) fprintf(stderr, "cell %2d (%2d.%2d) ==>  row %2d, col %2d\n",
	      cell, 
	      Get_TrialType_Info(i,STACK), Get_TrialType_Info(i,CLASS),
	      Get_TrialType_Info(i,COL), Get_TrialType_Info(i,ROW));
	 }
      }

   Set_Dimensions_For_Coords(3, 5);
# endif
   }
/* ********************************************************************	*/

/* FUNCTION Do_Layout_From_CmdLine */
static void Do_Layout_From_CmdLine(int sumflag) {
	int rows = 0, cols = 0;
	int i;

	if (OverlayStacks) {
	   for (i=0; i<List_Length(STACK); i++) {
	       int stack = List_Element(i,STACK);
	       StackToRow[stack] = 1;		/* All at row 1	*/
	       }
	   /* Set_Unfilled_Histos_From_CmdLine();	* "-f"	option	*/
	   rows = 1;
	} else if (OverlayClasses) {
	   for (i=0; i<List_Length(CLASS); i++) {
	       int class = List_Element(i,CLASS);
	       TableToCol[class] = 1;		/* All at col 1	*/
	       }
	   /* Set_Unfilled_Histos_From_CmdLine();	* "-f"	option	*/
	   cols = 1;
	} else
	 for (i=0; i<MAX_STACK_NUMBER; i++)
	   if (StackToRow[i] != -1)
	      rows++;

	if (DEBUG)
	   fprintf(stderr,"Will set rows to %d (%s) (not counting summary)\n",
		rows, (rows>0)? "value from cmdline" : "sequential order");

	for (i=0; i<Count_TrialTypes(); i++) {
           if ( !SHOW_SINGLES && Get_TrialType_Info(i,NUMBER) <= 1)
	  	 { 				/* Just 1 trial?	*/
	       Set_TrialType_Info(i,ROW, -1);	/* Don't plot singles	*/
	       Set_TrialType_Info(i,COL, -1);
	       fprintf(stderr, "layout.c: Skipping single (%d.%d)\n",
		   Get_TrialType_Info(i,STACK), Get_TrialType_Info(i,CLASS));
	       }
	   Set_TrialType_Info(i,		/* Set each row number:	*/
	      (RotateLayout)?COL:ROW,		/*   If rotate, set col	*/
	      (rows > 0) ?			/* # of rows specified?	*/
		StackToRow[Get_TrialType_Info(i, STACK)] :
		List_Index(Get_TrialType_Info(i,STACK), STACK));
	   }


	if (rows == 0)				/* Not set by cmd line?	*/
	    rows = List_Length(STACK);		/* Put one stack / row	*/


	if (! OverlayClasses)
	   for (i=0; i<MAX_STACK_NUMBER; i++)
	      if (TableToCol[i] != -1)
	         cols++;
	if (DEBUG)
	   fprintf(stderr,"Will set cols to %d (%s)\n",
		cols, (cols>0)? "value from cmdline" : "sequential order");

	for (i=0; i<Count_TrialTypes(); i++) {
           if (!SHOW_SINGLES && Get_TrialType_Info(i,NUMBER)<=1) { /* 1 trial?*/
	       Set_TrialType_Info(i,ROW, -1);	/* Don't plot singles	*/
	       Set_TrialType_Info(i,COL, -1);
	       continue;
	       }
	   Set_TrialType_Info(i,		/* Set each col number:	*/
		(RotateLayout)? ROW:COL,	/*   If rotate, set row	*/
	        (cols > 0) ?			/* # of cols specified?	*/
		  TableToCol[Get_TrialType_Info(i, TABLE)] :
		  List_Index(Get_TrialType_Info(i,TABLE), TABLE));
	   if (Get_TrialType_Info(i,STACK) == INVERT_ROW) {
	      static int HaveNotToldYou = 1;
	      Set_TrialType_Info(i, (RotateLayout)? ROW:COL,
		1+List_Length(TABLE)-Get_TrialType_Info(i,COL));
	      if (HaveNotToldYou) {
	         fprintf(stderr, "Inverting row %d\n", INVERT_ROW);
		 HaveNotToldYou = 0;
		 }
	      }
	   }

	if (cols == 0)
	   cols = List_Length(TABLE);

	if (sumflag) {	/* Could use sumflag to pass number of SUM_FIGS	*/
	   int i;

	   if (RotateLayout)
	      cols++;				/* Space for sum figs	*/
	   else
	      rows++;				/* Space for sum figs	*/
	   for (i=0; i<SUM_FIGS; i++) { /* to cells total + 1:SUM_FIGS	*/
	       Set_TrialType_Info(Count_TrialTypes() + i, CELL, i + rows*cols);
	       Set_TrialType_Info(Count_TrialTypes() + i, ROW,
					(RotateLayout)?cols:rows);
	       Set_TrialType_Info(Count_TrialTypes() + i, COL, i+1);
	       }
	   if (RotateLayout) {
	      if (rows < SUM_FIGS)
	          rows = SUM_FIGS;
 	    } else
	      if (cols < SUM_FIGS)
	          cols = SUM_FIGS;
	   }

	for (i=0; i<Count_TrialTypes(); i++)
	   Set_TrialType_Info(i, CELL, i + rows*cols);

	if (RotateLayout)
           Set_Dimensions_For_Coords(cols, rows);
	else
           Set_Dimensions_For_Coords(rows, cols);
	if (DEBUG)
	   fprintf(stderr,"Set to %d by %d\n", rows, cols);
	}
/* ********************************************************************	*/
#  define PUSH_PAST_NUMBER()	while (isdigit(*from)) from++;

/* FUNCTION FillFromCmdLine */
	 /* Read command line, fill array with stack->row OR table->col	*/
	 /* Return 1 if fails */
static int FillFromCmdLine(int *Into, char *from) {
	int CurrentNumber = 1;			/* Of column or row	*/

	while (1) {
	   int n = atoi(from);

 	   PUSH_PAST_NUMBER();

	   switch (*from) {	/* Next char:		*/
	      case 0:   Into[n] = CurrentNumber++;
			return(0);

	      case ',': Into[n] = CurrentNumber++;
			from++;
			break;

	      case ':': {
			int m = atoi(++from);

			if (n < m)
			  while (n <= m)
			   Into[n++] = CurrentNumber++;
			 else
			  while (n >= m)
			   Into[n--] = CurrentNumber++;

		   	PUSH_PAST_NUMBER();

			if (*from == 0)
			   return(0);
			if (*(from) == ',') {
			   from++;
			   break;
			   }
			/* Anything else:  ERROR!  Fall thru */
			}
				
	      default:
			fprintf(stderr, "Bad layout separator (%c)!\n", *from);
			return(1);
	      }
           }
	}
/* ********************************************************************	*/

/* FUNCTION FillFromList */
	 /* Using list, fill array with stack->row OR table->col	*/
static void FillFromList(int *into, int List) {
	int count = 1;		/* 'Fill' with all the stacks	*/
	int i;

	for (i=0; i<MAX_STACK_NUMBER; i++) 	/* Init to -1	*/
	   into[i] = -1;			/*    == unused	*/
	for (i=0; i<List_Length(List); i++) {	/* Now enter 'm	*/
#	  ifdef DO_NOT_KNOW_WHY_THIS_MATTERS
	   if (warned==0 && List_Element(i,List) > MAX_STACK_NUMBER) {
	      Warning("Exceeded max stack #; summary figure may fail");
	      warned++;
	      }
#	  endif
	   into[List_Element(i,List)] = count++;
	   }
	}
/* ********************************************************************	*/
