/* FILE histo */
     /* Care and feeding of spike histograms */
     /* See interval.c for functions/structures relating to INTERVAL	*/
#include "deffs.h"
#include "config.h"

#define GAUSSIAN 1
/* Like the old average shifted histogram (convolve with a triangle),
 * but a bit more precise.  Bin size has essentially no effect (no effect?)
 * on filtering.  To convert from SD to -3dB point: -3dB = 136 / SD(ms)
 */

#define SHOW_INTERVAL_IN_NON_SUM	1	/* If 1,show in raw data*/
#define PRINT_OUT_VALUES		(STATS_ON_SUM_FIG==2)

#define VARIANCE_CODE			1	/* Mostly at end of file*/
#define INTERVAL_BY_VARIANCE_METHOD	1	/* For getting interval	*/
     			/* 1: Use *any* interval (of max SD)	*/
     			/* 2: Background (prior to T2)		*/
     			/* 3: T2+50 to T2+800, T3+50 to T3+800	*/
     		        /* 4: Look between -d and -D times      */
     			/* 5: 3, minus bg (in Fill_Var_Histo())	*/

#define DEBUG				0
#define SUM_SCALE /* integer */		2	/* In summary figure	*/
#define POLAR_SCALE /* integer */	0.5	/* In polar summaries	*/

/* ********************************************************************	*/
/* Do_Histograms		Plot 1 histo for each class
 * Init_Histogram		Initialize structures
 * Cumulate_Histogram		Add in 1 trial's data to a histo
 * Show_Histogram		Plot 1 histogram
 * Set_Histo_Scale_From_CmdLine
 * Set_Unfilled_Histos_From_CmdLine
 * Get_Histo_BinSize		Return the bin size (ms) (interval.c)
 * Report_Histo_Info		Return string for include in postscript
 * Set_Interval_By_Peak_Variance    See VARIANCE_CODE
 *
 * PRIVATE:
 * Setup_Histograms		Determine vertical scaling
 * Do_Individual_Histograms	One histo for each trialtype
 * Plot_Histo_Of_Variance	Plot variance histogram (across types)
 * Plot_Histo_Of_Mean		Plot mean histo (across types, trials)
 * Plot_Summary_Figure		Plot mean+/-SE as function of class#
 * Plot_Polar_Summary		Plot class means on polar plot
 * Histo_ScaleBars			Draw scale bars
 * Fill_Variance_Histogram	Internal work for variance histos, intervals
 */

/* ********************************************************************	*/
/* ********************************************************************	*/

#if (GAUSSIAN==0)
     	/* Kernel is a triangle with base = 2*ASH-1, height=ASH	*/
     	/* Keep REAL_BIN_SIZE an integer multiple of ASH	*/
/* ********************************************************************	*/

#define MAX_TIME	10000			/* How long (ms)	*/
#define REAL_MAX_BINS	(MAX_TIME/REAL_BIN_SIZE)   /* ms/bin	*/

#define MAX_BINS function_MAX_BINS()
int function_MAX_BINS() { return(REAL_MAX_BINS * ASH); }

#define BIN_SIZE function_BIN_SIZE()
int function_BIN_SIZE() { return(REAL_BIN_SIZE / ASH); }
#define MAX_MAX_BINS	16000	/* MAX_BINS is parameter;this is abs max*/

#endif

#if (GAUSSIAN)
#  define   GAUSSIAN_SCALE 10000.		/* Large so coefs not 0	*/
     /* Eventually switch to floating point so can ignore all this!	*/
#  define   BIN_SIZE function_BIN_SIZE_MS()	/* grab_config value	*/
#  define   MAX_BINS	20000
#  define   MAX_MAX_BINS	MAX_BINS
/* Originally, we declared a huge array, but only used a subset, based
 * on bin size.  That only helps in initializing the array - trivial	*/
#  define MAX_SD	100
#  define THE_SD (SMOOTH_SD/(double)BIN_SIZE)	/* Adjust for bin size	*/
static int Gaussian[3*MAX_SD];		/* Allocate lots of space	*/
#endif


/* NOTE: Max_time, Real_bin_size and Real_max_bins are unused variables!*/
/* ********************************************************************	*/
static int Histogram[MAX_TYPES+1][MAX_MAX_BINS];  /* Store ASH	*/
static int Trials[MAX_TYPES+1];	/* '+1':MAX,MAX+1=mean,var hists*/
static int Fill_Histos = 1;		/* Boolean: solid or outline?	*/

static int HistoScale_From_CmdLine = 0;	/* Full scale, in Hz		*/
     /* Negative value is a percentage multiplier			*
      *	-h-200		Calc usual scale and then double	*
      *	-h-50		Calc usual scale and then halve		*
      */
static void Plot_Histo_Of_Mean();
static void Plot_Summary_Figure(int indi_flag);
static void Plot_Polar_Summary(int indi_flag);
static void Setup_Histograms();
static void Do_Individual_Histograms();
static void Histo_ScaleBars();

#if VARIANCE_CODE
static int  Fill_Variance_Histogram(float *Sum, float *Sum2);
static void Plot_Histo_Of_Variance();
#endif
/* ********************************************************************	*/
/* ********************************************************************	*/

/* FUNCTION Do_Histograms */
void Do_Histograms(int Output) {
     if ((Output & XY_INDIVIDUAL_PLOTS) != XY_INDIVIDUAL_PLOTS)
        Setup_Histograms();		/* Need for global plots, too	*/

     if (Count_TrialTypes() == 0) {
         fprintf(stderr, "No trials to plot\n");
         return;
         }

     if (Output & INDIVIDUAL_PLOTS)
        if ((Output & XY_INDIVIDUAL_PLOTS) != XY_INDIVIDUAL_PLOTS)
           Do_Individual_Histograms();

     if (Output & SUMMARY_PLOT) {
        Plot_Histo_Of_Mean();

        if ((Output & POLAR_SUM_PLOT) == POLAR_SUM_PLOT)
           Plot_Polar_Summary(Output & INDIVIDUAL_PLOTS);
#if VARIANCE_CODE
        else if ((Output & VARIANCE_SUM_PLOT) == VARIANCE_SUM_PLOT)
           Plot_Histo_Of_Variance();
#endif
        else		/* Std sum fig dependent on INDIVIDUAL_PLOTS	*/
 	      Plot_Summary_Figure(Output & INDIVIDUAL_PLOTS);
        }
     }
/* ********************************************************************	*/

/* FUNCTION Do_Individual_Histograms */
     /* Plot histogram for each trialtype	*/
static void Do_Individual_Histograms() {
     int i = -1;
     extern int if_overlay_classes();

     while (++i < Count_TrialTypes()) {
        SetColorByTrialType(i);
        Show_Histogram(i);
        Color(0,0,0);
        }
     }
/* ********************************************************************	*/

/* FUNCTION Init_Histogram */
      /* Initialize average shifted histograms */
void Init_Histogram() {
     int type,bin;

#	if (GAUSSIAN)
       int i;
       if (THE_SD > MAX_SD)
          Exit("SD too large", "Init_Histogram");
       for (i=0; i<(int)(3*THE_SD); i++)
           Gaussian[i] = (int)(0.5 + GAUSSIAN_SCALE /
     	 (THE_SD*sqrt(2*3.1416)) * exp(-((double)i*i)/(2.*THE_SD*THE_SD)));
#	endif
#	if (GAUSSIAN==0)
       if (REAL_BIN_SIZE % ASH)
        Exit("Make REAL_BIN_SIZE a multiple of ASH!", "histo.c");
#	endif

     for (type=0; type<MAX_TYPES; type++) {		/* Init bins	*/
         for (bin=0; bin<MAX_BINS; bin++)
     	    Histogram[type][bin] = 0;
         Trials[type] = 0;
         }
     }
/* ********************************************************************	*/

/* FUNCTION Cumulate_Interval_And_Histogram */
      /* Add 1 trial's data to histo & tally spikes in interval	*/
      /*  Note: wrap back when hit ends so that an entire spike is
          recorded (otherwise spike at 1st bin would count only 1/2)	*/
void Cumulate_Interval_And_Histogram() {
     int SpikeCount = SpikeCount_Header(-1);
     int type = Get_TrialType();
     int i = 0;
     extern short Spikes[];
     short *spikes = Spikes;
     int *HistoEnd = Histogram[type] + MAX_BINS;
     int *HistoBegin = Histogram[type];
     int BeginInterval1 = Get_Interval_Begin(1) + Get_ZeroTime();
     int EndInterval1   = Get_Interval_End(1)   + Get_ZeroTime();
     int BeginInterval2 = Get_Interval_Begin(2) + Get_ZeroTime();
     int EndInterval2   = Get_Interval_End(2)   + Get_ZeroTime();
     int spike_count1 = 0;
     int spike_count2 = 0;

     if (DurationTime()/BIN_SIZE  < MAX_BINS+1)
        HistoEnd = Histogram[type] + DurationTime()/BIN_SIZE + 2;
        /* Extra 2 bins: tails off spike count at end a bit	*/

     while (i++ < SpikeCount) {
        int *bin = Histogram[type] + *spikes/BIN_SIZE;
        int j = 0;

#	   if (GAUSSIAN)
        *bin += Gaussian[0];				/* Center bin	*/
        while (++j < (int)(3*THE_SD)) {		/* Flanking bins*/
           if (bin + j < HistoEnd)
              *(bin + j) += Gaussian[j];		/*  On right	*/
           else					   /* Wrap back	*/
     	 *(HistoEnd - ((bin+j)-HistoEnd) - 1) += Gaussian[j];
           if (bin - j >= HistoBegin)
              *(bin - j) += Gaussian[j];		/*  On left	*/
           else					   /* Wrap back	*/
     	 *(HistoBegin - ((bin-j)-HistoBegin)) += Gaussian[j];
           }
#	   endif

#	   if (GAUSSIAN == 0)
        *bin += ASH;					/* Center bin	*/
        while (++j < ASH) {				/* Flanking bins*/
           if (bin + j < HistoEnd)
              *(bin + j) += ASH-j;			/*  On right	*/
           else					   /* Wrap back	*/
     	 *(HistoEnd - ((bin+j)-HistoEnd) - 1) += ASH-j;
           if (bin - j >= HistoBegin)
              *(bin - j) += ASH-j;			/*  On left	*/
           else					   /* Wrap back	*/
     	 *(HistoBegin - ((bin-j)-HistoBegin)) += ASH-j;
           }
#	   endif

        if (*spikes >= BeginInterval1 && *spikes<EndInterval1)
           spike_count1++;
        if (BeginInterval2 != FAIL)		/* Is there a 2nd intvl	*/
           if (*spikes >= BeginInterval2 && *spikes<EndInterval2)
              spike_count2++;

        spikes++;				/* Get next spike	*/
        }

     Cumulate_Interval(spike_count1, spike_count2, type);
     (Trials[type])++;				/* Tally trial	*/
     }
/* ********************************************************************	*/

/* FUNCTION Set_Histo_Scale_From_CmdLine */
/* FUNCTION Get_Histo_Scale_From_CmdLine */
void Set_Histo_Scale_From_CmdLine(int x) {
     HistoScale_From_CmdLine = x;
     }
int  Get_Histo_Scale_From_CmdLine() {
     return(HistoScale_From_CmdLine);
     }
/* ********************************************************************	*/

/* FUNCTION Set_Unfilled_Histos_From_CmdLine */
      /* Leave histograms as unfilled outlines	*/
void Set_Unfilled_Histos_From_CmdLine() {
     Fill_Histos = 0;
     }
/* ********************************************************************	*/

/* FUNCTION Get_Histo_BinSize */
      /* For interval.c: bin size in ms */
int Get_Histo_BinSize() { return (BIN_SIZE); }
/* ********************************************************************	*/

#if GAUSSIAN
#  define HISTOtoHZ(i)	((1000. * (i)) / (BIN_SIZE * GAUSSIAN_SCALE))
#endif
#if (GAUSSIAN==0)
#  define HISTOtoHZ(i)	((1000. * (i)) / (BIN_SIZE * ASH*ASH))
#endif
#  define HZtoPIX(i)	((int) ((i) * scale))
#  define HISTOtoPIX(i)	(HZtoPIX(HISTOtoHZ(i)))
#  define MIN_HZ	20	/* Min full scale; this is what bar is	*/

static float scale;		/* Convert Hz to pixels	(HZtoPIX())	*/
static int scale_bar;		/* Height in Hz (draw with HZtoPIX())	*/

     /* HISTOtoHZ converts counts in the average shifted histogram
      * (whose kernel has area=ASH*ASH OR GAUSSIAN_SCALE) into spikes per second.
      * HZtoPIX scales spikes per second onto the plot (add offset).
      */

/* FUNCTION Setup_Histograms */
      /* Set correct scaling based on max counts across all histos	*/
      /* Calculate interval means and SE's				*/
static void Setup_Histograms() {

     int max_trials = 1;			/* To alloc raster space*/
     float max_counts_per_trial = 1;		/* Convert to max_hz	*/
     int max_hz;				/* Highest firing seen	*/
     int bins = DurationTime() / BIN_SIZE;
     int type;

     extern int SHOW_RASTER;			/* From graph.c		*/

     /* CALCULATE MAX COUNTS FOR ANY BIN, ANY TRIAL */
     for (type=0; type<MAX_TYPES; type++) {	/* Go thru all used figs*/
        register int bin;

        if (Trials[type] == 0)			/* End of used	*/
           break;
        if (Trials[type] == 1)			/* Skip singles	*/
           continue;
        if (Trials[type] > max_trials)		/* Find max trials/cell	*/
              max_trials = Trials[type];
        for (bin=0; bin<bins; bin++) 		/* Find max spikes/bin	*/
           if (Histogram[type][bin]/Trials[type] > max_counts_per_trial)
      	      max_counts_per_trial = Histogram[type][bin] / Trials[type];
        }	

     /* WHAT IS MAX RATE TO BE DISPLAYED (max_hz)?  DETERMINES SCALE BAR*/
     if (HistoScale_From_CmdLine > 0)		/* CmdLine request?	*/
        max_hz = HistoScale_From_CmdLine;	/* Make scale bar exact	*/
     else if (max_trials == 1)			/* Only a single trial	*/
        max_hz = 300;
     else {
        max_hz = (int) HISTOtoHZ(max_counts_per_trial);
        max_hz = (1+max_hz/MIN_HZ) * MIN_HZ;	/* Round up to pwr of 2	*/
        if (HistoScale_From_CmdLine < 0)	/* Alter after prev calc*/
           max_hz    = (int) (max_hz * (-HistoScale_From_CmdLine)/100);
        }
     scale_bar = max_hz;	/* Used to be 80% of max;100% is easier	*/

     /* GIVEN AVAILABLE PIXELS, SET Hz->pixel SCALE FACTOR	*/
      {
      int max_pix = HEIGHT-HISTO_OFFSET-HEIGHT/20;
     	/* Total pix minus 'used for analog display' minus margin */
      if (SHOW_RASTER)			/* If there's a raster:	*/
         /* Old calc: allows histo to overlap bottom half of rasters	*/
         /* max_pix -= max_trials*10-15;	 *  Allow for it's size	*/
         max_pix -= max_trials * RASTER_TIC_HEIGHT;
      scale = max_pix / (float) max_hz;	/* scale = pixels / hz	*/

      Histo_ScaleBars();
      }
}
/* ********************************************************************	*/

/* FUNCTION Histo_ScaleBars */
static void Histo_ScaleBars() {
     int total = RightEdgeCoord();		/* # of horiz pixels	*/
     int scale_bar_in_pix = HISTO_OFFSET + HZtoPIX(scale_bar);
     char tag[20];

     Setup_Coords_For_TrialType(TOP_LEFT_CELL);
     SetLineWidth(3);
   	fontsize(15);

     move(-total/35,  scale_bar_in_pix/2 + HISTO_OFFSET/2);
     sprintf(tag, "%d", scale_bar);
   	alabel('r', 't', tag);
     move(-total/35,  scale_bar_in_pix/2 + HISTO_OFFSET/2-250);
   	alabel('r', 't', "sp/s");

     move(-total/50,  HISTO_OFFSET);
     cont(-total/40,  HISTO_OFFSET);
     cont(-total/40,  scale_bar_in_pix);
     cont(-total/50,  scale_bar_in_pix);

     SetLineWidth(1);
     }
/* ********************************************************************	*/

/* FUNCTION Show_Histogram */
      /* Plot one histogram */
void Show_Histogram(int type) {
     int duration = DurationTime();

     if (Setup_Coords_For_TrialType(type) == -1)
        return;					/* Don't draw it*/

     if (Trials[type] <= 1 - SHOW_SINGLES) {
        fprintf(stderr,"%d trial of type %d: skipping\n",Trials[type],type);
        return;
        }
     fill(0);
     if (Fill_Histos) {
        move(duration, HISTO_OFFSET);
        cont(0, HISTO_OFFSET);			/* Draw 0 line	*/
     } else {
        SetLineWidth(2);
        move((SMOOTH)?BIN_SIZE/2:0, HISTO_OFFSET +
     		HISTOtoPIX(Histogram[type][0]/(float)Trials[type]));
        }

      {
      int bin;
      int x = (SMOOTH) ? -BIN_SIZE/2 : 0;	/* Smooth: add -1/2 width */
      for (bin=0; bin<duration/BIN_SIZE; bin++) {
     		/* Ghostview limits continuous 'cont's to about 390 */
         if (SMOOTH == 0) {
             cont(x, HISTO_OFFSET + 
     		HISTOtoPIX(Histogram[type][bin]/(float)Trials[type]));
             cont(x+=BIN_SIZE, HISTO_OFFSET + 
     		HISTOtoPIX(Histogram[type][bin]/(float)Trials[type]));
         } else
             cont(x+=BIN_SIZE, HISTO_OFFSET + 
     		HISTOtoPIX(Histogram[type][bin]/(float)Trials[type]));
         }
      }
     if (Fill_Histos) {
        extern int if_overlay_stacks(), if_overlay_classes();  /* layout.c */
        cont(duration, HISTO_OFFSET);
        fill( (if_overlay_stacks()||if_overlay_classes())
     				? 0xffff : 1);	/* Not written!	*/
     } else
        fill(0);
#	if (SHOW_INTERVAL_IN_NON_SUM)
        Show_Interval();			/* Spike-count interval	*/
#	endif
     SetLineWidth(1);
     }
/* ********************************************************************	*/

/* FUNCTION Plot_Histo_Of_Mean */
      /* Plot in (first) unused cell(s)  OR  on separate page */
static void Plot_Histo_Of_Mean() {
     int LastType = Count_TrialTypes();	/* 1st empty: calc avg	*/
     int Bins = DurationTime() / BIN_SIZE;	/* How many bins/histo?	*/
        int bin, type;

     for (type=0; type<LastType; type++) {	/* Calc avg histo	*/
        int *from  = Histogram[type];
        int *to    = Histogram[LastType];

        if ((Trials[type]==1) && ! SHOW_SINGLES)
           continue;				/* Skip singles		*/
        for (bin=0; bin<Bins; bin++)
           *(to++)  += *(from++);
        Trials[LastType] += Trials[type];
        }

     Show_Histogram(LastType);		/* Show average histo	*/
     Graph_TimeLine();			/* For last (avg'd) fig	*/
     Picture();

     Graph_Analog_Data();			/* Show 1 set of analog	*/
     Show_Interval();			/* Spike-count interval	*/

     if (Setup_Coords_For_TrialType(LastType) != -1) {
           move(10, (int) HEIGHT*.9);
           fontsize(15);
           alabel('c', 'b', "Mean response");
        }

#	ifdef MARK_AVERAGE_ON_EACH_CLASS
      for (type=0; type<LastType; type++) {
        if (! (Trials[type] > 1 || SHOW_SINGLES))
           continue;				/* Skip singles		*/
        Setup_Coords_For_TrialType(type);
        Show_Interval();
        }
# 	endif
      }
/* ********************************************************************	*/

#define SUM_OFFSET	(HISTO_OFFSET)

/* FUNCTION Plot_Summary_Figure */
      /* Plot mean+/-SEM of mean firing rate as function of class #	*/
      /* NOTE: scaling is x2 relative to ALL OTHER FIGS if desired	*/
static void Plot_Summary_Figure(int indi_flag) {
     int LastType = Count_TrialTypes();	/* Next empty: is sumfig*/
     int Bins = DurationTime() / BIN_SIZE;	/* How many bins/histo?	*/
        int type=0, i;
     int Classes = List_Length(CLASS);
     int Stacks = List_Length(STACK);
     int r,c;				/* Row & col counter	*/
     char tag[40];
     float Get_Interval_Mean(), Get_Interval_SEM();
     int DidLabel[40] = {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0};
     int Last_X_Coord = (Classes-1)*Bins*BIN_SIZE/Classes;
     extern int if_rotate_layout();
# ifdef ADD_THIS_SOME_TIME
     Do classes 1:max in the summary
     int MaxClass = 1;

     if (Sequential) {
       for (c in 1:Classes)
         if (List_Element(c, CLASS) > MaxClass)
            MaxClass = List_Element(c, CLASS);
       Last_X_Coord = (MaxClass-1)*Bins*BIN_SIZE/Classes;
       }
#endif

     Setup_Coords_For_TrialType(LastType+1);	/* Next empty cell:	*/

     SetLineWidth((Stacks<4) ? 4 : 2);

     if (if_rotate_layout()) {
      Last_X_Coord = (Stacks-1)*Bins*BIN_SIZE/Stacks;
      for (c=0; c<Classes; c++) {		/* For each class:	*/
        if (Class_Column(List_Element(c,CLASS)) == -1) /* In layout?	*/
           continue;
        SetColorByStackIndex(c);	/* Not a stack index at all!	*/
        for (r=0; r<Stacks; r++)
           for (type=0; type<LastType; type++)  /* Find matching fig	*/
     	 if (Match_List(type,CLASS,c) && Match_List(type,STACK,r)
     	                      && Get_TrialType_Info(type,NUMBER) > 1)
     	     goto A_FOUND_FIRST;
        A_FOUND_FIRST:			/* Begin the line	*/
        move((r*Bins*BIN_SIZE)/Stacks,
     	SUM_OFFSET + SUM_SCALE*HZtoPIX(Get_Interval_Mean(type,1)));
        while (++r < Stacks)		/* Do rest of line:	*/
           for (type=0; type<LastType; type++)
              if (Match_List(type,CLASS,c) && Match_List(type,STACK,r)
     	                && Get_TrialType_Info(type,NUMBER) > 1) {
     	    cont((r*Bins*BIN_SIZE)/Stacks, SUM_OFFSET +
     		    SUM_SCALE*HZtoPIX(Get_Interval_Mean(type,1)));
     	    break;
     	    }
        fill(0);
        }
     } else {			/* Not a rotated layout:	*/
      for (r=0; r<Stacks; r++) {		/* For each stack:	*/
        if (Stack_Row(List_Element(r,STACK)) == -1)	/* In layout?	*/
           continue;
        SetColorByStackIndex(r);
        for (c=0; c<Classes; c++)		/*  For each class	*/
           for (type=0;type<LastType;type++)	/*   Find matching fig	*/
     	 if (Match_List(type,STACK,r) && Match_List(type,CLASS,c)
     	             && Get_TrialType_Info(type,NUMBER) > 1)
     	     goto FOUND_FIRST;
        FOUND_FIRST:				/* Begin the line	*/
        move((c*Bins*BIN_SIZE)/Classes,
     	SUM_OFFSET + SUM_SCALE*HZtoPIX(Get_Interval_Mean(type,1)));
        while (++c < Classes)		/* Do rest of line:	*/
           for (type=0; type<LastType; type++)
              if (Match_List(type,STACK,r) && Match_List(type,CLASS,c)
     	             && Get_TrialType_Info(type,NUMBER) > 1) {
     	    cont((c*Bins*BIN_SIZE)/Classes,
                   SUM_OFFSET + SUM_SCALE*HZtoPIX(Get_Interval_Mean(type,1)));
     	    break;
     	    }
        fill(0);
        }
      }
     linemod("solid");
     SetLineWidth((Stacks<4) ? 2 : 1);

     if (PRINT_OUT_VALUES)
         fprintf(stderr, "stack class   mean  SEM    N\n");

     for (type=0; type<LastType; type++) {  /* Now do points & err bars	*/
        int x;				    /* x coordinate		*/
        float mean = Get_Interval_Mean(type,1);
        float SEM =  Get_Interval_SEM(type,1);
        int class = Get_TrialType_Info(type, CLASS);
        int stack = Get_TrialType_Info(type, STACK);
        int label_index = (if_rotate_layout())?
     			List_Index(stack, STACK) : class;

        if (! (Trials[type] > 1 || SHOW_SINGLES))
           continue;				/* Skip singles		*/
        if (Stack_Row(stack) == -1)
           continue;

        x = (if_rotate_layout()) ? 
     	((List_Index(stack, STACK) -1)*Bins*BIN_SIZE)/Stacks :
     	((List_Index(class, CLASS) -1)*Bins*BIN_SIZE) / Classes;
        x += 5*List_Index(stack, STACK);	/* Offset a couple pix	*/
        SetColorByTrialType(type);
        /* circle(x-80, SUM_SCALE * HZtoPIX(mean)-200,25); */
        move(x, SUM_OFFSET + SUM_SCALE * HZtoPIX(mean+SEM));
        cont(x, SUM_OFFSET + SUM_SCALE * HZtoPIX(mean-SEM));
        fill(0);

        if (!indi_flag || STATS_ON_SUM_FIG) {
	    int position = (if_rotate_layout()) ?
     		List_Index(Get_TrialType_Info(type,CLASS), CLASS) :
     		List_Index(Get_TrialType_Info(type,STACK), STACK) ;
	    int height = SUM_OFFSET - HEIGHT * position/12.;

            if (PRINT_OUT_VALUES)
              fprintf(stderr, "  %4d %2d    %5.1f %4.1f   %d\n",
            	    		stack, class, mean, SEM, Trials[type]);

   	    fontsize(12);
            move(x, height);
            sprintf(tag, "%.1f", mean);
   	    alabel('l', 'b', tag);

   	    fontsize(10);
            move(x, height - HEIGHT/30);
            sprintf(tag, "%.1f,%d", SEM, Trials[type]);	/* DIAG:rounds!	*/
   	    alabel('l', 'b', tag);
            }

        if (DidLabel[label_index] == 0) {
   	   fontsize((if_rotate_layout())? 8 : 15);
           Color(0,0,0);
           move(x, SUM_OFFSET - 150);
           sprintf(tag, "%d", (if_rotate_layout())? stack : class);
   	   alabel('c', 'c', tag);
           move(x, SUM_OFFSET); 
           cont(x, SUM_OFFSET-50);
           DidLabel[label_index] = 1;
           }
        }

     Color(0,0,0);					/* In black:	*/
     fontsize(15);
     SetLineWidth(1);

     move(0, SUM_OFFSET);				/* x axis	*/
     cont(Last_X_Coord, SUM_OFFSET);

     if (!indi_flag || STATS_ON_SUM_FIG) {		/*X axis label	*/
        move(DurationTime() - 200, SUM_OFFSET - 600);
        alabel('l', 'c', "Mean/sd,n");
     } else if (if_rotate_layout()) {
        move(Last_X_Coord/2, SUM_OFFSET - 600);
        alabel('c', 'c', "Stack");
     } else {
        move(Last_X_Coord/2, SUM_OFFSET - 600);
        alabel('c', 'c', "Class");
        }

     i = -1;					/* Draw y axis tics:	*/
     if (fabs(scale) > 0.01)			/* Avoid infinite loop	*/
      while (1) {
        int height = (++i * HZtoPIX(scale_bar) * SUM_SCALE) / 4;

        if (height >= (int)(HEIGHT * .8))
           break;
        move(DurationTime(),     SUM_OFFSET + height);
        cont(DurationTime()+100, SUM_OFFSET + height);
        move(DurationTime()+150, SUM_OFFSET + height-HEIGHT/25);
        sprintf(tag, "%d",       ((i*scale_bar+2)/4));
        alabel('l','c',tag);
        }
     fill(0);

     SetLineWidth(1);
     move(DurationTime(), SUM_OFFSET);		/* Draw y axis	*/
     cont(DurationTime(), SUM_OFFSET + (i-1)*HZtoPIX(scale_bar)*SUM_SCALE/4);

     Setup_Coords_For_TrialType(TITLE_SPACE);
     Color(8,0,0);					/* In red:	*/
     if (!indi_flag) {
        move(RIGHT_TEXT_MARGIN, TopCoord(10));
           alabel('l', 'b', "Mean +/- 1 SEM firing");
        }
     move(RIGHT_TEXT_MARGIN, TopCoord(0));
     if (Get_Interval_Begin(2) != Get_Interval_End(2))
        sprintf(tag, "%d:%d & %d:%d ms",
     	Get_Interval_Begin(1), Get_Interval_End(1),
     	Get_Interval_Begin(2), Get_Interval_End(2));
      else
        sprintf(tag, "From %d to %d ms",
     	Get_Interval_Begin(1), Get_Interval_End(1));
     alabel('l', 'b', tag);
     Color(0,0,0);
     }
/* ********************************************************************	*/

/* FUNCTION Plot_Polar_Summary */
      /* Plot mean+/-SEM of mean as f() of class ON POLAR PLOT	*/
      /* Use 1/3 of sum_scale					*/
      /* Use -d or -D to force the number of classes			*/
static void Plot_Polar_Summary(int indi_flag) {
     int LastType = Count_TrialTypes();	/* Next empty one	*/
     int type=0, i;
     int Classes = List_Length(CLASS);
     int Stacks = List_Length(STACK);
     int r,c;				/* Row & col counter	*/
     char tag[40];
     int x_center = DurationTime() / 2;
     int y_center = HEIGHT / 2;
     float Get_Interval_Mean(), Get_Interval_SEM();

     Setup_Coords_For_TrialType(LastType+1);	/* Next empty cell:	*/

     /* move(x_center, HEIGHT); cont(x_center, 0);	 * Cross-hairs	*/
     /* move(0, y_center); cont(DurationTime(), y_center);		*/

     if (Classes < 8)	/* Fewer than 8, but one is 9:16, then 16!*/
        for (c=0; c<Classes; c++)
        if (List_Element(c,CLASS) > 8 && List_Element(c,CLASS<=16)) {
     	   Classes = 16;
     	   break;
     	   }
     if (Get_DataValue_From_CmdLine())
        Classes = Get_DataValue_From_CmdLine();
      else if (Get_dataValue_From_CmdLine())
        Classes = Get_dataValue_From_CmdLine();

#	define Angle(class) 				\
      (-(class-1) * 2 * 3.14 / 			\
        ((Classes <= 8)  ? 8. : ((Classes <= 16) ? 16. : (float)Classes)))
/* Minus 1: Get_TrialType_Info(type,CLASS) returns 1-n; we want 0-(n-1)	*/
/*  negative because directions  2 thru 9 go CW, not CCW 		*/


     for (r=0; r<Stacks; r++) {		/* For each stack:	*/
        float rate;
        int start_x=0, start_y=0;		/* Remember 1st point	*/
        float max = 0.; int max_at = 0;	/* For finding max	*/
        int first = 1;
        /* GRAND_MEAN: int avg_rate=0,avg_count=0; * Cross conditions*/

        SetLineWidth((Stacks<4) ? 7 : 2);
        if (Stack_Row(List_Element(r,STACK)) == -1)	/* In layout?	*/
           continue;
        SetColorByStackIndex(r);

        c = -1;
        while (++c < Classes)		/* Do rest of line:	*/
         for (type=0; type<LastType; type++)
          if (Match_List(type,STACK,r) && Match_List(type,CLASS,c) &&
     	         Get_TrialType_Info(type,NUMBER) > 1) {
     	    double angle = Angle(Get_TrialType_Info(type,CLASS));

     	    rate = POLAR_SCALE*HZtoPIX(Get_Interval_Mean(type,1));
     	    /* GRAND_MEAN: avg_rate += rate; avg_count++; */
     	    if (rate > max) {		/* Find stack's max	*/
     	       max = rate;		/* Save where & how much*/
     	       max_at = Get_TrialType_Info(type,CLASS);
     	       }
     	    if (first) {
                    move(start_x = x_center + (int)(cos(angle) * rate),
                         start_y = y_center + (int)(sin(angle) * rate));
     	       first = 0;	/* Save the start, & use 'move'	*/
     	     } else
                    cont(x_center + (int)(cos(angle) * rate),
                         y_center + (int)(sin(angle) * rate));
     	    break;
     	    }
        cont(start_x, start_y);		/* Wrap back to start	*/
        fill(0);

        SetLineWidth(0);
        linemod("solid");

        for (type=0; type<LastType; type++)	/* Do error bars	*/
          if (Match_List(type,STACK,r) && 
     	     Get_TrialType_Info(type,NUMBER) > 1) {
     	    double angle = Angle(Get_TrialType_Info(type,CLASS));
     	    double Cosine = cos(angle) * POLAR_SCALE;
     	    double Sine   = sin(angle) * POLAR_SCALE;
     	    double SEM  = Get_Interval_SEM(type,1);
     	    rate = Get_Interval_Mean(type,1);
            move(x_center + (int) (Cosine * HZtoPIX(rate+SEM)),
                 y_center + (int) (Sine   * HZtoPIX(rate+SEM)));
       	    cont(x_center + (int) (Cosine * HZtoPIX(rate-SEM)),
     	    y_center + (int) (Sine   * HZtoPIX(rate-SEM)));
     	    fill(0);
     	    }

        		/* Circle showing grand mean (for each stack)	*/
        /* circle(x_center, y_center, avg_rate/avg_count);	* Mean	*/

     			/* Mark the maximum for each stack	*/
        {  double angle = Angle(max_at);	/* +1: macro subtracts 1*/
         Color(0,0,0);			/* Big BLACK mark	*/
         circle(x_center + (int) (cos(angle) * max),	/* Max point	*/
     	   y_center + (int) (sin(angle) * max),
     	   75);		/* (int)(SUM_SCALE*HZtoPIX(1)));	*/
         fill(1);
         SetColorByStackIndex(r);	/* Small coded mark inside	*/
         circle(x_center + (int) (cos(angle) * max),	/* Max point	*/
     	   y_center + (int) (sin(angle) * max),
     	   45);
         fill(1);
         }
        }

     Color(0,0,0);
     linemod("longdashed");
     SetLineWidth(0);
     circle(x_center, y_center, 15);
     fill(1);
     /* fill(1); */
     for (i=0; i<HEIGHT/5; i+=HZtoPIX(scale_bar/10))
        circle(x_center, y_center, i);

     Color(8,0,0);				/* Show scale	*/
     SetLineWidth(3);
     i -= HZtoPIX(scale_bar)/10;
     linemod("solid");
     move(x_center+i,	y_center+i/2);
     cont(x_center+i,	y_center-i/2);
     move(x_center+i/2,	0);
     sprintf(tag, "%d sp/s (x2?)", (int)(i/(float)HZtoPIX(1)));
     alabel('l', 'b', tag);
     linemod("solid");

     Setup_Coords_For_TrialType(TITLE_SPACE);
     fontsize(15);
     Color(8,0,0);					/* In red:	*/
     if (!indi_flag) {
        move(RIGHT_TEXT_MARGIN, TopCoord(10));
        alabel('l', 'b', "Mean +/- 1 SEM firing");
        }
     move(RIGHT_TEXT_MARGIN, TopCoord(0));
     if (Get_Interval_Begin(2) != FAIL)
        sprintf(tag, "(%d to %d) - (%d to %d) ms",
     	Get_Interval_Begin(1), Get_Interval_End(1),
     	Get_Interval_Begin(2), Get_Interval_End(2));
      else
        sprintf(tag, "From %d to %d ms",
     	    Get_Interval_Begin(1), Get_Interval_End(1));
     alabel('l', 'b', tag);
     Color(0,0,0);
     }
/* ********************************************************************	*/

/* FUNCTION Report_Histo_Info */
      /* Put creation info into ps file and invisibly on plot	*/
char *Report_Histo_Info() {
     static char Info[180];
#	if GAUSSIAN
     sprintf(Info,
"Histo: SHOW_INTERVAL_IN_NON_SUM %d SUM_SCALE %d SD %d",
     SHOW_INTERVAL_IN_NON_SUM, SUM_SCALE, SMOOTH_SD);
#endif
#	if (GAUSSIAN==0)
     sprintf(Info,
"Histo: SHOW_INTERVAL_IN_NON_SUM %d SUM_SCALE %d MAX_TIME %d REAL_BIN_SIZE %d ASH %d  SMOOTH %d",
     SHOW_INTERVAL_IN_NON_SUM, SUM_SCALE,
     MAX_TIME, REAL_BIN_SIZE, ASH, SMOOTH);
#endif

     /* Write invisibly on plot */
     Setup_Coords_For_TrialType(TITLE_SPACE);
     fontsize(4);
     Color(8,8,8);					/* In WHITE	*/
     move(LEFT_TEXT_MARGIN, TopCoord(0));	/* Near title bottom	*/
        alabel('l', 'b', Info);
     fontsize(15);
     Color(0,0,0);
     return(Info);
     }
/* ********************************************************************	*/
/* *******                    *****************************************	*/
/* *******   VARIANCE  CODE   *****************************************	*/
/* *******                    *****************************************	*/
/* ********************************************************************	*/

#if VARIANCE_CODE

#define VAR_WIDTH	250		/* Calc variance across ~ ms	*/
     /* Also, if use SD to set interval, is interval duration	*/
#define VAR_BIN  (VAR_WIDTH/BIN_SIZE)	/* Input bins in VAR_WIDTH	*/

#define SD_SCALING		80		/* Must be an int	*/
#define VARIANCE_SCALING	1.		/* Must be a  float	*/

/* FUNCTION Plot_Histo_Of_Variance */
      /* Variance as function of time across all fig types */
      /* If 'method 5', then background is first subtracted!	*/
static void Plot_Histo_Of_Variance() {
     float Sum[MAX_MAX_BINS];			/* Sum of spike counts	*/
     float Sum2[MAX_MAX_BINS];			/* Sum2 of spike counts	*/
     int LastType = Count_TrialTypes() + 1;	/* Next empty one	*/
     int Bins = DurationTime() / BIN_SIZE;	/* Bins per input histo	*/
     int bin;

     Trials[LastType] = Fill_Variance_Histogram(Sum, Sum2);

     for (bin=0; bin<VAR_BIN/2; bin++)
        Histogram[LastType][bin] = 0;
     for (; bin<Bins-VAR_BIN/2; bin++)
        Histogram[LastType][bin] = (int)
     		(SD_SCALING * SD(Sum[bin],Sum2[bin],Trials[LastType]));
     for (; bin<Bins; bin++)
        Histogram[LastType][bin] = 0;

     Show_Histogram(LastType);			/* Show variance histo	*/
     Graph_TimeLine();
     move(5, HEIGHT);
     fontsize(15);
     { char tag[80];
       sprintf(tag, "left panel: SD (x %d)", SD_SCALING);
       alabel('l', 'b', tag);
       }

# if (INTERVAL_BY_VARIANCE_METHOD != 5)
     /* Normalize to mean and replot	*/
     /* If method==5, then Sum[bin] has bg subtracted!! -- don't use!*/
     Color(8,0,0);				/* Red!			*/
     for (bin=VAR_BIN/2; bin<Bins-VAR_BIN/2; bin++)
        Histogram[LastType][bin] = (VARIANCE_SCALING *	/* Scaled	*/
           Histogram[LastType][bin] * Trials[LastType]) / Sum[bin];
     Show_Histogram(LastType);			/* Show variance histo	*/
     Color(8,0,0);				/* Red!			*/
     move(5, HEIGHT-100);
     fontsize(15);
     { char tag[80];
       sprintf(tag, "right: Normalized to mean (x %.f)", VARIANCE_SCALING);
       alabel('l', 'b', tag);
       }
     Color(0,0,0);				/* Normal color		*/
# endif

     Graph_Analog_Data();			/* Show 1 set of analog	*/
     }
/* ********************************************************************	*/

/* FUNCTION Set_Interval_By_Peak_Variance */
      /* Here instead of interval.c for convenience!	*/
      /* Minor semi-bug: if looks at a diff targ than what we're
         aligned on, won't be quite properly aligned! */
void Set_Interval_By_Peak_Variance() {
     float Sum[MAX_MAX_BINS];		/* Sum of spike counts	*/
     float Sum2[MAX_MAX_BINS];		/* Sum2 of spike counts	*/
     int LastType = Count_TrialTypes() + 1;	/* Next empty one	*/
     int Bins = DurationTime() / BIN_SIZE;
     int bin;

     float MaxSD = 0;
     int CenterBin = 0;
     int FirstBin, LastBin;

     Trials[LastType] = Fill_Variance_Histogram(Sum, Sum2);

#  if (INTERVAL_BY_VARIANCE_METHOD == 1)
     FirstBin = VAR_BIN/2 + 150/BIN_SIZE;	/* At least 150 ms in	*/
     LastBin = Bins - VAR_BIN/2;

     for (bin=VAR_BIN/2; bin<Bins-VAR_BIN/2; bin++) {
        float thisSD = SD(Sum[bin],Sum2[bin],Trials[LastType]);
        if (thisSD > MaxSD) {
           MaxSD = thisSD;
           CenterBin = bin;
           }
        }
#elif (INTERVAL_BY_VARIANCE_METHOD==4)
     		/* Specify the interval using -d to -D */
     FirstBin = Get_dataValue_From_CmdLine();
     LastBin  = Get_DataValue_From_CmdLine();

     if (FirstBin == 0 && LastBin == 0)
          Warning("Can set bounds with -d and -D");

     FirstBin /= BIN_SIZE;
     LastBin  /= BIN_SIZE;

     CenterBin = LastBin - VAR_BIN/2;

     for (bin=VAR_BIN/2; bin<Bins-VAR_BIN/2; bin++) {
        float thisSD = SD(Sum[bin],Sum2[bin],Trials[LastType]);
        if (thisSD > MaxSD) {
           MaxSD = thisSD;
           CenterBin = bin;
           }
        }
     fprintf(stderr, "DIAG: Need to get this working!!\n");
#elif (INTERVAL_BY_VARIANCE_METHOD==2)
     		/* Effect of position (PRIOR TO 2nd TARG)? */
     FirstBin = Get_TargetTime(1);
     LastBin  = Get_TargetTime(2);

     if (FirstBin == FAIL || LastBin == FAIL)
          Warning("Target time 1 or 2 missing!");

     FirstBin /= BIN_SIZE;
     LastBin  /= BIN_SIZE;

     CenterBin = LastBin - VAR_BIN/2;

     if (LastBin - VAR_BIN < 0)
        Warning("Pre-interval too short: FIX (using non-existent bins)!");
     else if (LastBin - VAR_BIN < FirstBin)
        Warning("Using 'control' data prior to 1st acquire!");
     else if (LastBin - VAR_BIN < FirstBin + 300/BIN_SIZE)
        Warning("Using 'control' data from within 300 ms of 1st acq!");

#elif (INTERVAL_BY_VARIANCE_METHOD==3 || INTERVAL_BY_VARIANCE_METHOD==5)
     	/* Look from targettime(2)+50 to 800 ms later; 	*/
     	/* 5: bg spikes subtracted (in Fill_Var_Histo()	*/
     FirstBin = (Get_TargetTime(2) + 50)  / BIN_SIZE + VAR_BIN/2;
     LastBin  = (Get_TargetTime(2) + 800) / BIN_SIZE - VAR_BIN/2;

     if (LastBin > Bins - VAR_BIN/2)
         LastBin = Bins - VAR_BIN/2;
     if (Get_TargetTime(3) != FAIL)		/* If a 3rd target	*/
         if (LastBin > Get_TargetTime(3)/BIN_SIZE - VAR_BIN/2)
             LastBin = Get_TargetTime(3)/BIN_SIZE - VAR_BIN/2;

     for (bin=FirstBin; bin<LastBin; bin++) {
        float thisSD = SD(Sum[bin],Sum2[bin],Trials[LastType]);
        if (thisSD > MaxSD) {
           MaxSD = thisSD;
           CenterBin = bin;
           }
        }

     if (Get_TargetTime(3) != FAIL) {
        FirstBin = (Get_TargetTime(3) + 50)  / BIN_SIZE + VAR_BIN/2;
        LastBin  = (Get_TargetTime(3) + 800) / BIN_SIZE - VAR_BIN/2;

        if (LastBin > Bins - VAR_BIN/2)
           LastBin = Bins - VAR_BIN/2;
        for (bin=FirstBin; bin<LastBin; bin++) {
           float thisSD = SD(Sum[bin],Sum2[bin],Trials[LastType]);
           if (thisSD > MaxSD) {
              MaxSD = thisSD;
              CenterBin = bin;
              }
           }
        }
#endif

     Set_Interval_By_Bin(CenterBin-VAR_BIN/2, CenterBin+VAR_BIN/2);
     }
/* ********************************************************************	*/

/* FUNCTION Fill_Variance_Histogram */
static int Fill_Variance_Histogram(float *Sum, float *Sum2) {
     int LastType = Count_TrialTypes();		/* Next empty one	*/
     int Bins = DurationTime() / BIN_SIZE;	/* Bins per input histo	*/
     int bin, type, k;
     int count = 0;

     for (bin=0; bin<Bins; bin++)		/* Initialize		*/
        Sum[bin] = Sum2[bin] = 0;

     for (type=0; type<LastType; type++) {	/* For each figure type	*/
        int background_spike_count = 0;	/* Method 5 only!	*/
        if (! (Trials[type] > 1 || SHOW_SINGLES))
           continue;				/* Skip singles		*/
        count++;

#if (INTERVAL_BY_VARIANCE_METHOD==5)		/* Subtract background	*/
        {
        int background_bin = Get_TargetTime(2)/BIN_SIZE - VAR_BIN/2 - 1;
        for (k=-VAR_BIN/2; k<VAR_BIN/2; k++)		/* Spikes in bg	*/
              background_spike_count += Histogram[type][background_bin+k];
        }
#endif

        for (bin=VAR_BIN/2; bin<Bins-VAR_BIN/2; bin++) { /* For @ bin*/
           int spike_count = 0;
           float normalized;

           for (k=-VAR_BIN/2; k<VAR_BIN/2; k++) /* Spikes in big bin	*/
     	      spike_count += Histogram[type][bin+k];
           spike_count -= background_spike_count;	/* Method 5!	*/
           normalized = spike_count / ((float) Trials[type] * VAR_BIN);
           Sum[bin]  += normalized;
           Sum2[bin] += normalized * normalized;
           }
        }
     return(count);
     }
/* ********************************************************************	*/
#else
void Set_Interval_By_Peak_Variance() {}		/* Dummy function	*/
#endif
