/* FILE event */
     /* Handle stacks (and events within stacks) */
#include "deffs.h"
#include "event.h"
#include "event.i"
#include "config.h"

#include <string.h>					/* strcmp()	*/

#define  DEBUG 		0		/* Levels 0 thru 2		*/

#define PRINT_ONLY_SOME_EVENTS	0	/* Skip windows, etc.		*/

#define UPDATE_1998_EVENTS	1
#define ADJUST_EVENTS		1	/* DIM 0/100 -> BLANK on/off	*/
					/* 2: print debug statement	*/
/* ********************************************************************	*/
/* ********************************************************************	*/
/*
PUBLIC:
  EventExtract		Return an element of a particular stack entry
  MultiEventExtract	Return an element of one of several types of entries
  EventExtractTapeOn
  EventExtractTapeOff
  EventExtractErrorBegin
  EventExtractTrialStamp	Ought to be in header!
  EventAddTrialStamp		Ought to be in header!
  AnyTargetEventExtract	Return an element of a 'target on' entry
  TargetEventExtract	Return element of a specific 'target on' entry
  AnyTargetOffEventExtract	Return an element of a 'target off' entry
  TargetOffEventExtract		Return element of a specific 'target off' entry
  WindowAcquireEventExtract	Return element of an ACQUIRE (eye/arm/head...)
  AcquireEventExtract		Return element of specific acquire
  Print_Events			Print out stack
  Shift_Time_In_EventStack	For -O(ffset) option (with -w(rite)
  Show_Events_For_XY_Plot	For -g1, show all events
  Event_Adjust
  CheckLengthOfEventStack	Check if stack is longer than data stream
  EventNameToNumber		For macros
PRIVATE
  Is_TargetType			Is it eventype TARGET_/ON,SAC,POLAR, etc?
 */

static void Print_Entry(EVENT *E);
static EVENT_INFO_TYPE *Info_Ptr(int type);
static int EventMatch(int types[], int match);
static int Is_TargetType(int x);
/* ********************************************************************	*/
/* *************  PUBLIC FUNCTIONS ************************************	*/
/* ********************************************************************	*/

/* FUNCTION EventExtract */
	/*  Return elements of header stack	*/
int EventExtract(int type, int which, int occurrence) {
    EVENT *E = Stack_Ptr_Header();

    if (occurrence >= LAST_OCCURRENCE) {
       EVENT *TOP = E;
       int count = occurrence - LAST_OCCURRENCE + 1;	/* How many from end?*/

       while ((++E)->type != END_STACK) ;		/* Find end	*/
       while (--count >= 0) {
          while ((--E)->type != type)
	     if (E <= TOP)
	         return(FAIL);
	  }
       goto RETURN_VALUE;
       }

    do {
       if (E->type == type) {
	  if (--occurrence == 0) {
	     RETURN_VALUE:
	     switch (which) {
		case TIME:   return(E->time);
		case ONE:    return(E->one);
		case TWO:    return(E->two);
		case THREE:  return(E->three);
		case TYPE:   return(E->type);
		default:     return(FAIL);
		}
	     }
	  }
     } while ((E++)->type != END_STACK);

    if (type==TAPE_ON)			/* No tape_on?  Return '0'	*/
       return(0);
    else if (type==TAPE_OFF)		/* No tape_off? Return end time	*/
       return((E-1)->time);		/* Time of END_STACK		*/

    return(FAIL);
    }
/* ********************************************************************	*/

/* FUNCTION MultiEventExtract */
	/*  Return elements of header stack	*/
	/* NOTE: MUST TERMINATE 'types[]' with a '0' entry!	*/
int MultiEventExtract(int types[], int which, int occurrence) {
    EVENT *E = Stack_Ptr_Header();

    if (occurrence >= LAST_OCCURRENCE) {
       EVENT *TOP = E;
       int count = occurrence - LAST_OCCURRENCE + 1;	/* How many from end?*/

       while ((++E)->type != END_STACK) ;		/* Find end	*/
       while (--count >= 0)
          while (! EventMatch(types, E->type))
	     if (--E <= TOP)
	         return(FAIL);
       goto RETURN_MULTI_VALUE;
       }

    do {
       if (EventMatch(types, E->type))
	  if (--occurrence == 0)
	    RETURN_MULTI_VALUE:
	     switch (which) {
		case TIME:   return(E->time);
		case ONE:    return(E->one);
		case TWO:    return(E->two);
		case THREE:  return(E->three);
		case TYPE:   return(E->type);
		default:     return(FAIL);
		}
     } while ((E++)->type != END_STACK);
    return(FAIL);
    }
/* ********************************************************************	*/

/* FUNCTION Is_TargetType */
	 /* Check if E->type is a target on type */
static int Is_TargetType(int x) {
	return(x==TARGET_ON || x==TARGET_SAC || x==TARGET_ON_POLAR ||
	      (x >= TARGET_RE_0 && x<= TARGET_RE_8_POLAR));
	}	
/* ********************************************************************	*/

/* FUNCTION EventExtractTapeOff */
int EventExtractTapeOff() {
	return(EventExtract(TAPE_OFF, TIME, 1));
	}
/* FUNCTION EventExtractTapeOn */
int EventExtractTapeOn() {
	return(EventExtract(TAPE_ON, TIME, 1));
	}
/* FUNCTION EventExtractErrorBegin */
int EventExtractErrorBegin() {
	return(EventExtract(ERROR_STACK_BEGIN, TIME, 1));
	}

/* FUNCTION EventExtractTrialStamp */
   /* Bit of a kluge.  Put in header, not stack!	*/
int  EventExtractTrialStamp() { return(EventExtract(TRIAL_STAMP, 1, 1)); }
void EventAddTrialStamp(int add) { 
	EVENT *E = Stack_Ptr_Header();
	E--;
	while ((++E)->type != END_STACK) {
	   if (E->type == TRIAL_STAMP) {
	       E->one += add;
	       return;
	       }
	   }
	}
/* ********************************************************************	*/

/* FUNCTION AnyTargetEventExtract */
	 /* Find a target event */
int AnyTargetEventExtract(int which, int occurrence) {
    /* return(MultiEventExtract(TargetTypes, which, occurrence)); */
    return(TargetEventExtract(-1, which, occurrence));
    }
/* ********************************************************************	*/

/* FUNCTION AnyTargetOffEventExtract */
	 /* Find a target off event */
int AnyTargetOffEventExtract(int which, int occurrence) {
    return(TargetOffEventExtract(-1, which, occurrence));
    }
/* ********************************************************************	*/

/* FUNCTION TargetEventExtract */
	 /* Find a target event that uses a particular target (-1:any)	*/
	 /*    target_on,   target_on_polar 	*
	  *    target_sac,  target_sac_polar 	*
	  *    target_re_#, target_re_#_polar	*/
int TargetEventExtract(int target, int which, int occurrence) {
    EVENT *E = Stack_Ptr_Header();

    if (occurrence >= LAST_OCCURRENCE) {
       EVENT *TOP = E;
       int count = occurrence - LAST_OCCURRENCE + 1;	/* How many from end?*/

       while ((++E)->type != END_STACK) ;		/* Find end	*/
       while (--count >= 0) {
	  while (! (Is_TargetType(E->type) && (target<0 || E->one==target)))
	     if (--E <= TOP)
	         return(FAIL);
	  }
       goto RETURN_MULTI_VALUE;
       }

    do {
       if (Is_TargetType(E->type) && (target<0 || E->one == target)) {
	  if (--occurrence == 0) {
	    RETURN_MULTI_VALUE:
	     switch (which) {
		case TIME:   return(E->time);
		case ONE:    return(E->one);
		case TWO:    return(E->two);
		case THREE:  return(E->three);
		case TYPE:   return(E->type);
		default:     return(FAIL);
		}
	    }
	}
     } while ((E++)->type != END_STACK);
    return(FAIL);
    }
/* ********************************************************************	*/

/* FUNCTION TargetOffEventExtract */
	 /* Find a target off event that uses a particular target	*/
int TargetOffEventExtract(int target, int which, int occurrence) {
    EVENT *E = Stack_Ptr_Header();

    if (occurrence >= LAST_OCCURRENCE) {
       EVENT *TOP = E;
       int count = occurrence - LAST_OCCURRENCE + 1;	/* How many from end?*/

       while ((++E)->type != END_STACK) ;		/* Find end	*/
       while (--count >= 0) {
	  while (! (E->type==TARGET_OFF && (target<0 || E->one==target)))

	     if (--E <= TOP)
	         return(FAIL);
	  }
       goto RETURN_MULTI_VALUE;
       }

    do {
       if (E->type==TARGET_OFF && (target<0 || E->one == target)) {
	  if (--occurrence == 0) {
	    RETURN_MULTI_VALUE:
	     switch (which) {
		case TIME:   return(E->time);
		case ONE:    return(E->one);
		case TWO:    return(E->two);
		case THREE:  return(E->three);
		case TYPE:   return(E->type);
		default:     return(FAIL);
		}
	    }
	}
     } while ((E++)->type != END_STACK);
    return(FAIL);
    }
/* ********************************************************************	*/
/* FUNCTION WindowAcquireEventExtract */
	 /* Find an acquire event that is associated with a window	*/
	/* Includes 'move' acquires */
int WindowAcquireEventExtract(int which, int occurrence) {
    EVENT *E = Stack_Ptr_Header();

    if (occurrence >= LAST_OCCURRENCE) {
       EVENT *TOP = E;
       int count = occurrence - LAST_OCCURRENCE + 1;	/* How many from end?*/

       while ((++E)->type != END_STACK) ;		/* Find end	*/
       while (--count >= 0)
          while (! ((E->type == ACQUIRE) && (E->one < MAX_WINDOW_BEHAVS)))
	     if (--E <= TOP)
	         return(FAIL);
       goto RETURN_VALUE;
       }

    do {
       if (E->type==ACQUIRE && E->one<MAX_WINDOW_BEHAVS) {
	  if (--occurrence == 0) {
	     RETURN_VALUE:
	     switch (which) {
		case TIME:   return(E->time);
		case ONE:    return(E->one);
		case TWO:    return(E->two);
		case THREE:  return(E->three);
		case TYPE:   return(E->type);
		default:     return(FAIL);
		}
	     }
	  }
     } while ((E++)->type != END_STACK);
    return(FAIL);
    }
/* ********************************************************************	*/

/* FUNCTION AcquireEventExtract */
	 /* Find acquire particular behav and/or target (-1:any)	*/
int AcquireEventExtract(int behav, int target, int which, int occurrence) {
    EVENT *E = Stack_Ptr_Header();

    if (occurrence >= LAST_OCCURRENCE) {
       EVENT *TOP = E;
       int count = occurrence - LAST_OCCURRENCE + 1;	/* How many from end?*/

       while ((E++)->type != END_STACK) ;		/* Find end	*/
       while (--count >= 0) {
	  E--;
          while (! ((E->type==ACQUIRE ||
		    E->type==BUTTON_ACQUIRE_AND ||
		    E->type==BUTTON_ACQUIRE_OR) &&
	         (behav <0 || E->one == behav) &&
	         (target<0 || E->two == target)))
	     if (--E <= TOP)
	         return(FAIL);
	  }
       goto RETURN_VALUE;
       }

    do {
       if ((E->type==ACQUIRE ||
	    E->type==BUTTON_ACQUIRE_AND ||
	    E->type==BUTTON_ACQUIRE_OR) &&
           (behav <0 || E->one==behav) &&
	   (target<0 || E->two==target)) {
	  if (--occurrence == 0) {
	     RETURN_VALUE:
	     switch (which) {
		case TIME:   return(E->time);
		case ONE:    return(E->one);
		case TWO:    return(E->two);
		case THREE:  return(E->three);
		case TYPE:   return(E->type);
		default:     return(FAIL);
		}
	     }
	  }
     } while ((E++)->type != END_STACK);
    return(FAIL);
    }
/* ********************************************************************	*/

/* FUNCTION Shift_Time_In_Event_Stack */
	 /* Change tape_on/off times before writing data to output file	*/
void Shift_Time_In_Event_Stack(int time) {
	EVENT *E = Stack_Ptr_Header();

	E--;
	while ((++E)->type != END_STACK)
	   E->time -= time;
	}
/* ********************************************************************	*/

/* FUNCTION EventMatch */
static int EventMatch(int types[], int match) {
	int i = -1;
	while (types[++i])
	   if (match == types[i])
	       return(1);
	return(0);
	}
/* ********************************************************************	*/

/* FUNCTION Print_Events */
	 /* Print scratch stack	*/
void Print_Events() {
   EVENT *E = Stack_Ptr_Header();
   int line = 0;
   int skip = 0;

   E--;
   while ((++E)->type != END_STACK) {
#      if PRINT_ONLY_SOME_EVENTS
        while (IS_SKIP_EVENT_TYPE(E->type))
          E += E->two + 1;			/* Jump over them	*/

        if((E->type >= 230 && E->type <= 240) ||  /* Eye windows, store	*/
           (E->type >= 330 && E->type <= 340) ||  /* Arm windows, store	*/
	   (E->type == 135) ||			  /* NEW_COLOR		*/
	   (E->type >= CHANNEL_COUNT && E->type <= REWARD_SCHEDULE) ||
	   		/* flow,err stacks,start,pick cols,store,rwd	*/
	   (E->type >= RANDOM_VALUE && E->type <= SKIP_HIGH_BOUND))
	   		/* randoms, *value, skips			*/
	  continue;
#      endif

       fprintf(stdout, "%2d %c ", ++line, (skip-->0)?'<':' ');
       Print_Entry(E);
       fprintf(stdout, "\n");

       if (IS_SKIP_EVENT_TYPE(E->type))
          skip = E->two;
       }
#  if (! PRINT_ONLY_SOME_EVENTS)
    fprintf(stdout, "%2d    %d  End\n", ++line, E->time);
#  endif
   }
/* ********************************************************************	*/

/* FUNCTION PRINT_STRING_FORMAT */
	 /* Return 1: end printout immediately	*/
static int PRINT_STRING_FORMAT(EVENT *E, char *format) {
    char **string = NULL;
    int value = -1;		/* Should always be overwritten		*/
    int length = 10000;

    switch (*format) {		/* 1st char says which string		*/
    	case 'O':		/* Handled slightly differently in reach*/
	case 'X':
	case 'S': break;

	case 'o': string = OffOrOn;  
		  length = sizeof(OffOrOn)/sizeof(*OffOrOn);
		  break;
	case 'B':				/* Added to the grab code */
	case 'b': string = BehavName;
		  length = sizeof(BehavName)/sizeof(*BehavName);
		  break;
	case 'c': string = ColorName;
		  length = sizeof(ColorName)/sizeof(*ColorName);
		  break;
	case 'C': 
		  string = ChannelName;
		  length = sizeof(ChannelName)/sizeof(*ChannelName);
		  break;
	case 'v': string = ValueName;	
		  length = sizeof(ValueName)/sizeof(*ValueName);
		  break;
	case 'T':			break;
	case 't': string = NamedTargets;	/* Not used	*/
		  break;
	case 'r': string = RGB_Name;    
		  length = sizeof(RGB_Name)/sizeof(*RGB_Name);
		  break;
	default: goto ERR_HANDLER;
	}

    switch (*(format+1)) {		/* 2nd char says which param #	*/
	case '1': value = E->one;   break;
	case '2': value = E->two;   break;
	case '3': value = E->three; break;
	default:  goto ERR_HANDLER;
	}

    switch (*format) {
       case 'X':				/* If value==off, stop	*/
       case 'O':				/* " "", print off&stop	*/
       	      if (value==OFF) {
	         if (*format == 'O')
		    fputs("Off", stdout);
	         return(1);
		 }
	      break;

       case 'S':	/* Print stack name: can't! */
	      break;

       case 'T':	/* Not available	*/
              fprintf(stdout, "(Targ %d)  ", value);
              break;

       default:
	      if (E->type == 76 && E->one == -1)
		  value = 0;		/* Gets flagged in reach; really 0 */
	      if ((value < 0) || value >= length)
		 fprintf(stdout, "value=%d (illegal) ", value);
	        else
	         fputs(string[value], stdout);
	      break;
       }
    return(0);

    ERR_HANDLER:
	putchar(*format);
	putchar(*(format+1));
	return(0);
    }
/* ********************************************************************	*/

/* FUNCTION Print_Entry */
	 /* Print a single event */
	 /* If type is negative, it wasn't actually run on this trial	*/
static void Print_Entry(EVENT *E) {
   EVENT_INFO_TYPE *EI;
   char *f;
   int Neg = 0;

   if ((int) E->type < 0) {
      E->type = -(E->type);
      Neg = 1;
      }

   EI = Info_Ptr(E->type);
   f  = EI -> format;

   printf("%5d  %s%-16s  ", (E->time == 65535) ? -1 : E->time,
		   	    (Neg) ? "-":"", EI->tag);

   --f;
   while (*(++f)) {
      switch (*f) {
         case SPECIAL_FORMAT:	goto SPECIAL_CASES;
	 case '%' :
	        switch (*(++f)) {	/* Push past '%'	*/
		   case '1':   printf("%d ", E->one);   break;	  /* %1	*/
		   case '2':   printf("%d ", E->two);   break;	  /* %2	*/
		   case '3':   printf("%d ", E->three); break;	  /* %3	*/
		   case '%':   putchar('%');    break;	  /* %%	*/
		   default:    
			       if (PRINT_STRING_FORMAT(E, f++))
		                  return;
		   }
		break; 

	 default:	putchar(*f);
	 }
      }
   return;


   SPECIAL_CASES:
   switch ((int) E->type) {

      case ACQUIRE:
      	   fprintf(stdout, "%s", BehavName[E->one]);
	   if ((E->one < MAX_WINDOW_BEHAVS) && (E->one != E->two))
	      fprintf(stdout, " Targ %d", E->two);
	   fprintf(stdout, " %d ms", E->three);
	   break;

      case START_AT:
	   switch (E->one) {
	      case 0:  fputs("immediately", stdout);		break;
	      case 1:  fputs("after 1 acquire", stdout);	break;
	      default: printf("after %d acquires", E->one);	break;
	      }
	    break;

      case SUMMARY:
            printf("cell:%d type:%d", E->one, E->two);
	    if (E->two == 1)
	       break;
	    printf("at:%d", E->three);
	    break;

      case DELAY:
	    if (E->one != 0)
		printf("%d lines ", (int) E->one);
	    printf("<= %d ms", E->two);	/* Lookup not allowed	*/
	    if (E->three == 0)
		E->three = 10;				/* Default	*/
	    if (E->three != 10)
		printf(" by %dms", E->three);		/* No lookup	*/
	    break;

      case NEW_COLOR:
      	    printf("Color # %d, %s", E->one-15, RGB_Name[E->two]);
	    if (E->two == 0)
	       printf(" same as VGA %s", ColorName[E->three]);
	    else
	       printf(" = %d", E->three);
	    break;

      default:
      case -1:
      	    printf(" (type %d) (%d,%d,%d)", E->type, E->one,E->two,E->three);
	    break;
      }
   return;
}
/* ********************************************************************	*/

/* FUNCTION Info_Ptr */
	 /* Return pointer to specific Event_Info entry	*/
static EVENT_INFO_TYPE *Info_Ptr(int type) {
      EVENT_INFO_TYPE *Use = Event_Info - 1;

      while ((++Use)->type >= 0)
         if (Use->type == type)
            return(Use);
      return(Use);
      }
/* ********************************************************************	*/

/* FUNCTION RemoveSpace */
        /* Return a copy of what you're sent minus any spaces   */
        /* Spaces foul up 'stack.exe' -- much tougher to parse  */
static char *RemoveSpace(char *from) {
	static char NoSpace[80];
	char *to = NoSpace;

	while (*from)
	   if (*from != ' ')
	      *(to++) = *(from++);
	   else
	      from++;
	*to = 0;
	return(NoSpace);
	}
/* ********************************************************************	*/

/* FUNCTION EventNameToNumber */
	 /* Given a name, return the number */
	/* Can't pass spaces through the command line */
int EventNameToNumber(char *name) {
        EVENT_INFO_TYPE *Use = Event_Info - 1;
	char stripped[80];
	int i;

	for (i=0; i<80; i++) {
	   if ((*(name+i) == 0)   || (*(name+i) == '.') ||
	       (*(name+i) == '_') ||
	       (*(name+i) == '#') || (*(name+i) == ','))
	      break;
	   else
	      stripped[i] = *(name+i);
	   }
	stripped[i] = 0;

        while ((++Use)->type >= 0) {
           if (strcmp(Use->tag, stripped) == 0)
              return(Use->type);
           if (strcmp(RemoveSpace(Use->tag), stripped) == 0)
              return(Use->type);
	   }
	return(FAIL);
	} 
/* ********************************************************************	*/
/* ********************************************************************	*/
/* FUNCTION SkipPicture */
static int SkipPicture(int target) {
   switch (target) {
      case 0:  return(PICTURE_TARGET_0 == 0);
      case 1:  return(PICTURE_TARGET_1 == 0);
      case 2:  return(PICTURE_TARGET_2 == 0);
      case 3:  return(PICTURE_TARGET_3 == 0);
      case 4:  return(PICTURE_TARGET_4 == 0);
      case 5:  return(PICTURE_TARGET_5 == 0);
      case 6:  return(PICTURE_TARGET_6 == 0);
      case 7:  return(PICTURE_TARGET_7 == 0);
      case 8:  return(PICTURE_TARGET_8 == 0);
      case 9:  return(PICTURE_TARGET_9 == 0);
     default: return(1);
     }
   }
/* ********************************************************************	*/
#define TENTHtoAD(tenth)	(2 * (tenth))

/* FUNCTION Show_Events_For_XY_Plot */
	 /* Show events for individual XY plot */
void Show_Events_For_XY_Plot() {
        EVENT *E;
	int at_h=0, at_v=0;

	for (E=Stack_Ptr_Header(); E->type!=END_STACK; E++) {
	   if (Is_TargetType(E->type)) {
	       if (SkipPicture(E->one))
		  continue;
	       at_h = TENTHtoAD(E->two);	/* Abs, not rel location*/
	       at_v = TENTHtoAD(E->three);
	       XY_Plot_Target(at_h, at_v, E->one);
	    } else if (E->type==TARGET_PURSUIT && (E->two!=0 || E->three!=0)) {
	       int time = 0;
	       EVENT *F = E;

	       if (SkipPicture(E->one))
		  continue;
	       while (++F != END_STACK)			/* Find stop	*/
		  if (F->type == TARGET_PURSUIT &&
			E->one == F->one && (F->two==0 && F->three==0)) {
		     time = F->time - E->time;		/* Purs duration*/
		     break;
		     }

	       F = E;				/* Find last pos of targ*/
	       while (--F > Stack_Ptr_Header())
		  if (Is_TargetType(F->type) && (F->one == E->one))
		     break;

	       XY_Plot_Pursuit(
	         TENTHtoAD(F->two),
	         TENTHtoAD(F->three),
	         TENTHtoAD(F->two + (E->two * time * 10 / 1000)),
	         TENTHtoAD(F->three + (E->three * time * 10 / 1000)));
		        /* Start + rate(d/s)*time(s)*tenth/deg / ms/s	*/
	       XY_Plot_Target(
	         TENTHtoAD(F->two + (E->two * time * 10 / 1000)),
	         TENTHtoAD(F->three + (E->three * time * 10 / 1000)),
		 E->one);
	       }
	  }
	}
/* ********************************************************************	*/

/* FUNCTION Adjust_Events */
	 /* Do some generic fixes */
void Adjust_Events(EVENT *E) {

	if (UPDATE_1998_EVENTS && Version() == -2)
	   fprintf(stderr, "Cannot do this!\n");

	if (ADJUST_EVENTS == 0)
	   return;

	E--;	/* Replace TARGET_DIM 0/100 with TARGET_BLANK on/off	*/
	while ((++E)->type != END_STACK)
	   if (E->type == TARGET_DIM && E->two == 0) {
	       int target = E->one;
	       E->type = TARGET_BLANK;
	       E->two = ON;

	       while ((++E)->type != END_STACK) {
	          if (E->type == TARGET_DIM &&
		      E->one == target &&
		      E->two == 100) {
		    if (ADJUST_EVENTS == 2)
		       fprintf(stderr, "Replace dim 100 with blank off\n");
		    E->type = TARGET_BLANK;
		    E->two = OFF;
		    break;
		    }
	         }
	      }
	}
/* ********************************************************************	*/

/* FUNCTION CheckLengthOfEventStack */
	 /* On non-successful trial, show which events weren't reached	*/
void CheckLengthOfEventStack(int data_length) {
    EVENT *E = Stack_Ptr_Header();

    data_length += EventExtract(TAPE_ON, TIME, 1);	/*Add start time*/
    E--;

    while ((++E)->type != END_STACK)
       if (E->time > data_length && E->time != (unsigned short) -1)
	       				/* E->time is unsigned!		*/
          E->type = -E->type;		/* Leave in data, but unreadable*/

    if (E->time > data_length)
        E->time = data_length;
    }
/* ********************************************************************	*/
