/* FILE calib.c */
     /* Reverse calibrations */

#include "deffs.h"

#include <string.h>
#include <ctype.h>
#include <unistd.h>
/* ******************************************************************** */

/* PUBLIC: 
 *
 *  Init_CalibFile
 *  Read_CalibFile
 *  Do_Calibration
 *
 * PRIVATE:
 *
 */
/* ******************************************************************** */

# define DEGtoAD(d)	( (d) * 20 )	

 static int CalibFileNumber = -1;		/* Last file read	*/
 static char *CalibFilePath;
 static FILE *CalibFile;

 static int CalibType = -1;			/* Unset	*/

#define CHAIR_EYE_CROSSTALK	Calibs[0]
#define EYE_LEFT_BOOST		Calibs[1]
#define EYE_DOWN_BOOST		Calibs[2]
#define HORIZONTAL_OFFSET	Calibs[3]
#define VERTICAL_OFFSET		Calibs[4]

/* Calib type 1: some are set.  2+3: none are used */
#define CHAIR_RIGHT_CROSSTALK	  ((CalibType==1)? Calibs[5] : 0)
#define EYE_CROSSTALK		  ((CalibType==1)? Calibs[6] : 0)
#define DOWNWARD_CROSSTALK	  ((CalibType==1)? Calibs[7] : 0)
#define DOWNWARD_CROSSTALK_OFFSET ((CalibType==1)? Calibs[8] : 0) /*in deg*/

/* Calib type 2: used.  1: not used */
#define H_GAIN	((CalibType>1)? Calibs[5] : 0)
#define V_GAIN	((CalibType>1)? Calibs[6] : 0)

/* Calib type 3: used.  1+2: not used */
#define H_ON_V_CROSSTALK	((CalibType==3)? Calibs[7] : 0)
#define V_ON_H_CROSSTALK	((CalibType==3)? Calibs[8] : 0)

#define MAX_CALIBS		  9
 static int Calibs[MAX_CALIBS];
 static int Warned = 0;

/* ******************************************************************** */

/* FUNCTION Init_CalibFile */
	 /* Init calib file number */
void Init_CalibFile(char *inputfilename) {
	CalibFileNumber = -1;
	CalibFilePath = inputfilename;
	}
/* ******************************************************************** */

#define Get_Next_Line						\
	   while (fgets(Line, 80, CalibFile) != NULL)		\
	      /* if ((Line[0] != '#') && (Line[1] != '\n')) */	\
	      if (isdigit(Line[0]) || Line[0]=='-')		\
	         break;
/* ******************************************************************** */
	
/* FUNCTION Read_CalibFile */
	 /* Read in the specified calib file */
void Read_CalibFile(int number) {
	if (CalibFileNumber != number) {
	   char FileName[105];
	   char Line[81];
	   int i;

	   if (number == ALREADY_CALIBRATED) {		/* Ran thru -w?	*/
	      CalibFileNumber = number;
	      return;
	      }

	   if (number <= 0) {		/* No calib file ?	*/
	      fprintf(stderr, "No calib file--call larry!!\n");
	      CalibFileNumber = number;
	      return;
	      }

	   CalibFileNumber = number;
	   sprintf(FileName, "calib.%d", number);
	   CalibFile = fopen(FileName, "r");

	   if (CalibFile == NULL) {	/* Try same directory as file	*/
	      int i;
	      for (i=strlen(CalibFilePath); i>0; i--)	/* Find slash	*/
	         if (*(CalibFilePath+i) == '/') {
		    strncpy(FileName, CalibFilePath, (size_t) i);
		    FileName[i] = 0;
	   	    sprintf(FileName, "%s/calib.%d", FileName, number);
	   	    CalibFile = fopen(FileName, "r");
		    break;		/* Tried your best, now go away	*/
		    }
	      }

	   if (CalibFile == NULL) {
	      fprintf(stderr, "Cannot find calibration file '%s'\n", FileName);
	      return;
	      }


	   Get_Next_Line;		/* Calibration type	*/
	   CalibType = atoi(Line);	/* Ignore it for now	*/
	   
	   for (i=0; i<MAX_CALIBS; i++)
	       Calibs[i] = 0;

	   for (i=0; i<MAX_CALIBS; i++) {
	       Get_Next_Line;
	       Calibs[i] = atoi(Line);
	       if (feof(CalibFile))
	          break;
	       }

	   fclose(CalibFile);
	   }
	}
/* ******************************************************************** */
#define CHAIRtoAD(raw)    (int)(DEGtoAD((raw)/12.42))

/* FUNCTION Do_Calibration */
	 /* Do reverse calibration */
void Do_Calibration(short *data, int channels, int frame) {
	static int turntable = 0;
	int UsingTurnTable = (ChairAt_Header() != FAIL);
	int EyeInHead;

	if (CalibFileNumber == ALREADY_CALIBRATED)	/* From -w	*/
	   return;

	UsingTurnTable &= (channels > 2);

		/* Has Chuck's box run out of offset?		*/
	*(data+H_EYE) += HORIZONTAL_OFFSET;
	*(data+V_EYE) += VERTICAL_OFFSET;

	if (UsingTurnTable) {	/* Recording table position/velocity?	*/
 	   if (frame == 0)				/* Init smoother*/
	      turntable = *(data+CHAIR_POSITION);
	   else						/* Smoothed copy*/
	      turntable = (turntable*2 + *(data+CHAIR_POSITION)) / 3;

		/* Cross-talk of chair position on h eye position	*/
		/*    (Calculated using raw turntable units)		*/
	   *(data+H_EYE) += 
	   	  (short)(turntable * (float)CHAIR_EYE_CROSSTALK/100.);

	   	/* Change SMOOTHED chair posit from raw units to ADs 	*/
	   *(data+CHAIR_POSITION) = (short) CHAIRtoAD(turntable);

	   /* Change chair velocity from tenths per s to AD units per s	*/
	   *(data+CHAIR_VELOCITY) *= 2;

	   	/* Calculate h eye position in the head */
	   EyeInHead = *(data+H_EYE) - *(data+CHAIR_POSITION);
	 } else
	   EyeInHead = *(data+H_EYE);

	/* Leftward eye calibration	*/
	if (EyeInHead < 0)
	   *(data+H_EYE) += (short) (EyeInHead * EYE_LEFT_BOOST / 100.);

	/* Downward eye calibration	*/
	if (*(data+V_EYE) < 0)
	   *(data+V_EYE) += (short) (*(data+V_EYE) * EYE_DOWN_BOOST / 100.);

	/* Crosstalk (required where there's a large vertical offset)	*/
	*(data+V_EYE) += *(data+H_EYE) * EYE_CROSSTALK/100.;

	if (CHAIR_RIGHT_CROSSTALK) {
	   if (ChairAt_Header() == REACH_FAIL)
	      *(data+H_EYE) += *(data+V_EYE) * CHAIR_RIGHT_CROSSTALK/100.;
           else if (Warned==0) {
	      fprintf(stderr, "Chair right calibration not written!  (%d)\n",
	             CHAIR_RIGHT_CROSSTALK);
	      Warning("");
	      Warned = 1;
	      }
	   }
	if (*(data+V_EYE) < 0)
	   *(data+H_EYE) +=  
   /* Adjust for horiz shift proportional to vertical AND horizontal, centered
    * at vert = 0 and horiz = OFFSET */
		   (*(data+H_EYE) - DEGtoAD(DOWNWARD_CROSSTALK_OFFSET)) *
		    -*(data+V_EYE) * DOWNWARD_CROSSTALK/10000.; 
	/* if at (0,-20) and values are -10 [deg] & 10 [%], then add to H_EYE:
	 *    10 deg * -20/20 * 10/100 = 10 * -1 * .1 = -1 deg	*/

	if (H_GAIN)
	   *(data+H_EYE) += *(data+H_EYE) * H_GAIN/100.;
	if (V_GAIN)
	   *(data+V_EYE) += *(data+V_EYE) * V_GAIN/100.;

	if (H_ON_V_CROSSTALK)
	   *(data+V_EYE) += *(data+H_EYE) * H_ON_V_CROSSTALK/100.;
	if (V_ON_H_CROSSTALK)
	   *(data+H_EYE) += *(data+V_EYE) * V_ON_H_CROSSTALK/100.;
	}
/* ******************************************************************** */
