/* FILE blink */
     /* Find blinks and remove them */

/* I've added "REMOVE_BLINKS" as a possible parameter in .grab_config.
 * Set it to 1 to (try to) remove blinks from the eye movement data.
 * It will replace the blink artifact with two straight lines, extending
 * the initial (pre-blink) eye position to the mid-point of the blink,
 * and then abruptly changing to the post-blink eye position and holding
 * that new position to the end of the blink.  The idea is that if there's
 * no change in eye position as a result of the blink, you'll be left with
 * a straight line.  If there *is* a change, you'll be left with an
 * instantaneous saccade of the appropriate size at the mid-point of the blink
 * artifact.
 */

#include "deffs.h"
#include "ad.h"

/* ********************************************************************	*/
 /* PUBLIC
   Remove_Blinks1	Detect blinks and remove them; return count
   Remove_Blinks2	Use input 0100 to remove blinks
  * PRIVATE
   Velocity		Return veloc at a point in time
   ReplaceEyeData	Replace a chunk of data with straight line
 * ********************************************************************	*/
#define POINTS		(32/ms)			/* 15 -> 60/ms */
#define PEAK_CRITERION	DEGtoAD(30)		/* 30,80 */
#define END_CRITERION   DEGtoAD(10)		/* 10,20 */
#define DEBUG		0

static float Velocity(short *ptr, int channels);
static void  ReplaceEyeData(int begin, int end);
/* ********************************************************************	*/

/* FUNCTION RemoveBlinks1 */
void RemoveBlinks1() {
	short *TOP, *END, *X, *Y;
	int channels = ChannelCount_Header();
	int ms = MsPerFrame_Header();
	int count = 0;

	extern short Data[MAX_FRAMES*MAX_CHANNELS];

	TOP = Data;
	TOP += (POINTS+1)*channels;		/* Leave safe margin	*/

	END = Data + (channels * FrameCount_Header());	/* Find end	*/
	END -= (POINTS+1)*channels;

	X = TOP + H_EYE;
	Y = TOP + V_EYE;

        while (1) {		/* Find a huge biphasic velocity	*/
	   if (fabs(Velocity(X, channels)) > PEAK_CRITERION ||
	       fabs(Velocity(Y, channels)) > PEAK_CRITERION) {
	      short *StartX = X, *StopX = X;	/* Putative blink	*/
	      short *StartY = X, *StopY = Y;	/* Putative blink	*/

#	      ifdef NOT_NEEDED
	      int Sign;				/* Positive veloc?	*/

	      if (fabs(Velocity(X, channels)) > PEAK_CRITERION)
		 Sign = Velocity(X,channels) > 0;
	      else if (fabs(Velocity(V, channels)) > PEAK_CRITERION)
		 Sign = Velocity(Y,channels) > 0;
#	      endif
 	
    	      if (DEBUG) 
	         fprintf(stderr, "Possible near %ld (%.0f %.0f %.0f)\n",
			     ms*(X-Data)/channels, Velocity(X,channels),
			     	 Velocity(StartX+32/ms,channels),
			     	 Velocity(StartX+120/ms,channels));
	      for (StartX=X, StartY=Y; 		/* Backup to find start	*/
		   StartX>X-100/ms*channels;
		   StartX-=channels, StartY-=channels) {
		  if (fabs(Velocity(StartX, channels)) < END_CRITERION &&
		      fabs(Velocity(StartY, channels)) < END_CRITERION)
		     break;			/* Found start of blink	*/
		  if (StartX <= TOP) 		/* Too far?		*/
		     break;
		  }


	      if (StartX <= X-100/ms*channels)
		 goto FALSE_ALARM;		/* Too far back?	*/
    	      if (DEBUG) 
	         fprintf(stderr, "Possible start: %ld (%.0f %.0f %.0f)\n",
			     ms*(StartX-Data)/channels,
			     Velocity(StartX,channels),
			     Velocity(StartY,channels), 0.);

	      for (StopX=X+50/ms*channels, StopY=Y+50/ms*channels; 
		   StopX<X+300/ms*channels; 
		   StopX+=channels, StopY+=channels) {
		  if (fabs(Velocity(StopX, channels)) < END_CRITERION &&
		      fabs(Velocity(StopY, channels)) < END_CRITERION)
		     break;
		  if (StopX >= END) 		/* Too far?		*/
		     break;
	          }
	      /* Most likely now at the turn-around point for blink	*/
	      /* So jump over that and test for biphasic */
	      /* Could check that there was a large change in posit	*/

	      StopX += 50/ms*channels;
	      StopY += 50/ms*channels;
	      if (fabs(Velocity(StopX, channels)) < PEAK_CRITERION &&
	          fabs(Velocity(StopY, channels)) < PEAK_CRITERION) {
		 if (DEBUG) fprintf(stderr, "Not a biphasic velocity\n\n");
		 goto FALSE_ALARM;
	         }

	      for ( ;
		   StopX < X+300/ms*channels; 
		   StopX+=channels, StopY+=channels) {
		  if (fabs(Velocity(StopX, channels)) < END_CRITERION &&
		      fabs(Velocity(StopY, channels)) < END_CRITERION)
		     break;
		  if (StopX >= END) 		/* Too far?		*/
		     break;
	          }


	      if (StopX >= X+300/ms*channels)
		 goto FALSE_ALARM;
    	      if (DEBUG) 
	         fprintf(stderr, "Stop: %ld (%.0f %.0f %.0f)\n",
			     ms*(StopX-Data)/channels,
			     Velocity(StopX,channels),
			     Velocity(StopY,channels), 0.);

	      /* Extend the selection? */
	      StartX -= 0/ms*channels;
	      StopX  += 0/ms*channels;

	      ReplaceEyeData((StartX-Data)*ms/channels,
			     (StopX-Data)*ms/channels);

	      count++;
	      X = StopX + 60/ms*channels;
	      Y = StopY + 60/ms*channels;
	     }

	    FALSE_ALARM:
	     X += channels;
	     Y += channels;
	     if (X >= END)
	        break;
	}
    if (DEBUG) fprintf(stderr, "%d blinks\n", count);
    /* return(count); */
    }
/* ********************************************************************	*/

/* FUNCTION Velocity */
	 /* Return change in posit about the time of the ptr */
static float Velocity(short *ptr, int channels) {
    int ms = MsPerFrame_Header();
    return((float) *(ptr+POINTS*channels) - *(ptr-POINTS*channels));
    }
/* ********************************************************************	*/
/* ********************************************************************	*/
#define START_MARGIN	(140/ms)	/* Multiply by 4, divide by ms	*/
#define END_MARGIN	(120/ms)

/* FUNCTION RemoveBlinks2 */
void RemoveBlinks2() {
	int ms = MsPerFrame_Header();
	int time_off = -END_MARGIN;
	int time_on;

	while ((time_on = 
		Get_Button_Time(1,64,1, time_off+END_MARGIN, 0)) != FAIL) {
	   time_off = Get_Button_Time(0,64,1, time_on, 0);
	   if (time_off == FAIL)
	       time_off = 30000000;		/* Infinitely large	*/
	   ReplaceEyeData(time_on - START_MARGIN, time_off+END_MARGIN);
	   }
	}
/* ********************************************************************	*/

/* FUNCTION ReplaceEyeData() */
	 /* Replace from Start to Stop in 2 pieces; discontinuity between*/
static void ReplaceEyeData(int begin, int end) {
	extern short Data[MAX_FRAMES*MAX_CHANNELS];
	int channels = ChannelCount_Header();
	int ms = MsPerFrame_Header();
	short *END = Data + (channels * FrameCount_Header());
	short *x = Data + ((begin-SkipSpikeTime())/ms)*channels;
	short *y = x + (V_EYE - H_EYE);
	short *endX = Data + ((end-SkipSpikeTime())/ms)*channels;
	short *endY = x + (V_EYE - H_EYE);
	short fillX, fillY;

	if (DEBUG)
	   fprintf(stderr, "Remove %d to %d ms\n", begin, end);
	if (x < Data + 4*channels) {	/* At start; use end to fill	*/
	   if (endX > END -4*channels + H_EYE) {
	      if (FrameCount_Header() > 20)	/* Skip fails/no data	*/
	         Warning("Entire trial is a blink (no eye data)!");
	      return;
	      }
	   fillX= (*endX + *(endX+(1*channels)) + *(endX+(2*channels)) + 
		  *(endX-(3*channels)) + 2) / 4;
	   fillY= (*endY + *(endY+(1*channels)) + *(endY+(2*channels)) + 
		  *(endY-(3*channels)) + 2) / 4;
	 } else {
	   fillX= (*x + *(x-(1*channels)) + *(x-(2*channels)) + 
		  *(x-(3*channels)) + 2) / 4;
	   fillY= (*y + *(y-(1*channels)) + *(y-(2*channels)) + 
		  *(y-(3*channels)) + 2) / 4;
	   }

	if (endX > END)
	    endX = END;
	for ( ; x <= x + (endX-x)/2;
		x+=channels, y+=channels) {
	   *x = fillX;
	   *y = fillY;
	   }

	if (endX <= END -4*channels + H_EYE) {	/* Get end-of-blink pos	*/
	   fillX = ( *(endX+0*channels) + *(endX+1*channels) +
		     *(endX+2*channels) + *(endX+3*channels) ) / 4;
	   fillY = ( *(endY+0*channels) + *(endY+1*channels) +
		     *(endY+2*channels) + *(endY+3*channels) ) / 4;
	 } else {	/* Ends with blink: use pre-blink eye posit	*/
	   endX = END;
	   endY = END + V_EYE - H_EYE;
	   }

	for ( ; x <= endX;
		x+=channels, y+=channels) {
	   *x = fillX;
	   *y = fillY;
	   }
	}
/* ********************************************************************	*/
