function [spikes] = load_spikes(trial_file, spike_file, time_interval, fs, trial_index)
% LOAD SPIKES: Load spikes as a an array of 0s (no spike) and 1s (spike).
%   Each number is an LFP sample index.
%
%   Input:
%       trial_file = the temp file with data for each trial
%                    (trial stamp, start time, trial type, stack, class)
%       spike_file = the temp file with spike times (ms) for each trial
%       time_interval = trial duration after start time (ms)
%       fs = LFP sampling rate (kHz).
%       trial_index = which trials to include (trials that have LFP).
%
%   Output:
%       spikes = array of 0s (no spike) and 1s (spike), each number is
%                an LFP sample index
%

    % Read spikes times from spike file
    spike_times = {};
    fid = fopen(spike_file);
    fgetl(fid);                 % Skip header
    while ~feof(fid)
        spike_times{numel(spike_times) + 1} = sscanf(fgetl(fid), '%d'); 
    end
    fclose(fid);

    % Load reach data for each trial
    [trial_stamps_reach, start_time, trial_type] = read_trial_file(trial_file);
	
	% Remove trials without LFP data
	trial_stamps_reach = trial_stamps_reach(trial_index);
	start_time = start_time(trial_index);
	trial_type = trial_type(trial_index);
    spike_times = spike_times(trial_index);

	% Convert spike times to array of 0s (no spike) and 1s (spike) for each
    %   LFP Sample 
    n_samples_per_interval = round(time_interval * fs);
    spikes = zeros(length(spike_times), n_samples_per_interval);
    for i_trial = 1:length(spike_times)
        spike_indexes = round(spike_times{i_trial} * fs) + 1;
        spike_indexes(spike_indexes > n_samples_per_interval) = [];
        spikes(i_trial, spike_indexes) = 1;
    end

end
