function powBurstLFP(suffix)

% LFP POWER - BURSTS
%   Find bursts in power.
%       Criteria:  Must exceed 2 sd of trial mean for 3+ cycles.
%       Burst def: Half-max to half-max.
%

% AUTHOR: Charles D Holmes
% EMAIL:  chuck@eye-hand.wustl.edu
%

% Turn off warnings
%warning off all

% Import Parameters
Params = readParams(suffix);

% Parse relevant params
freqArray  = Params.freqStart1:Params.freqStep1:Params.freqEnd1;
timeWindow = Params.time1;
timeStep   = Params.time2; 
evoked     = Params.evoked;
TW         = Params.freqStart2 / 10;

% Read data
[lfpTrials,fsLfp,trialType,~,trialNum] = ...
        readAndProcessLFPData(Params.datadirectory,Params.commandline{1},   ...
                              Params.trialfile,Params.timeInterval,         ...
                              Params.alphaPlex,Params.electrode1);

% Get all trial types and count nTrials
typelist = unique(trialType);
for iType = 1:length(typelist)
    nTrials(iType) = sum(trialType==typelist(iType));
end

% Handle stimulus locked response
[lfpTrials,trialType] = lfpLockedAndInduced(lfpTrials,trialType,evoked);

% Compute power spectrum, then average power per spectrum
spec = specgramMtm(lfpTrials,fsLfp*1e3,timeWindow,timeStep, ...
                    freqArray,'pow',TW);

% Get power traces and trial-wise stats (mean, std).
pow     = permute(sum(spec,2),[3,1,2]); % Time x Trial
powMean = mean(pow, 1);
powSd   = std(pow, [], 1);

% Find bursts of power (exceed 2 std).
powExceed = bsxfun(@gt,pow,powMean+2*powSd);
%powExceed = pow > powMean + 2*powSd;
tempZeros = zeros(1,size(powExceed,2));
tempDiff  = diff([tempZeros;powExceed;tempZeros],[],1);
[excessIdx(:,1),burstTrial] = ind2sub(size(tempDiff),find(tempDiff==1));
[excessIdx(:,2),~]          = ind2sub(size(tempDiff),find(tempDiff==-1));

% Truncate burst end times (currently 1 element too long)
runsToEnd              = excessIdx(:,2)==size(powExceed,1)+1;
excessIdx(runsToEnd,2) = size(powExceed,1);

% Find power excess (first step for finding bursts). 
% Excess must last > 3 cycles of smallest frequency.
minWidth    = (3*1000)/freqArray(1);
excessWidth = diff(excessIdx,[],2)*timeStep;
isBurst     = excessWidth > minWidth;
excessIdx   = excessIdx(isBurst,:);
burstTrial  = burstTrial(isBurst);

% For each burst (half-max to half-max).
burstArray = zeros(size(powExceed));
for iBurst = 1:size(burstTrial,1)
    excessWin        = excessIdx(iBurst,1):excessIdx(iBurst,2);
    [powMax,tempIdx] = max(pow(excessWin,burstTrial(iBurst)));
    maxIdx           = excessWin(1)-1+tempIdx;
    belowHalf        = pow(:,burstTrial(iBurst)) <= 0.5*powMax;
    burstStart       = find(belowHalf(1:maxIdx),1,'last');
    burstEnd         = find(belowHalf(maxIdx:end),1,'first') + maxIdx - 1;
    
    if isempty(burstStart), burstStart = 1;                 end
    if isempty(burstEnd),   burstEnd   = length(belowHalf); end
    burstArray(burstStart:burstEnd,burstTrial(iBurst)) = 1;
end

% Average for burst rate.
for iType = 1:length(typelist)
    isType             = trialType==typelist(iType);
    burstRate(:,iType) = mean(burstArray(:,isType),2);
end


%%%% DATA OUTPUT %%%%
fid = fopen(['data', suffix],'w');
fprintf(fid,'%s\n',char(Params.commandline{1}));
fprintf(fid,'%s\n',Params.datadirectory);
fprintf(fid,'%s\n',Params.paramsstr);
fprintf(fid,'%s\n',sprintf('%d ',nTrials));
fprintf(fid,'\n');
for iType = 1:length(typelist)
    fprintf(fid,'%e ',burstRate(:,iType));
    fprintf(fid,'\n');
end
fclose(fid);
