function matchPursuitLFP(suffix)
% LFP POWER
%
% Output: 
%  Power Matrix (trial type x freq x time).
%   Data is written to a text file. Each row is a time series of power for
%   a given frequency and trial type. Rows are grouped in blocks of trial type,
%   with each row being the next frequency.
%

% AUTHOR: Charles D Holmes
% EMAIL: chuck@eye-hand.wustl.edu
%


% Prep
addpath(genpath('/data/code/grab/2018/_macros/lfp/packages/MP/matlab'));

% Import Parameters
Params = readParams(suffix);
time_dec_pre = Params.time1;
time_dec_post = Params.time2;

% Read data
[lfp_trials, fs_lfp, trial_type, ~, trial_num] = ...
        readAndProcessLFPData(Params.datadirectory, Params.commandline{1},   ...
                              Params.trialfile, Params.timeInterval*2,       ...
                              Params.alphaPlex, Params.electrode1);


% Get all trial types and count nTrials
type_list = unique(trial_type);
for i_type = 1:length(type_list)
    n_trials(i_type) = sum(trial_type == type_list(i_type));
end

% Handle alignment locked responsed
lfp_trials = lfpLockedAndInduced(lfp_trials, trial_type, Params.evoked);

% Downsample lfps
[b_pre, a_pre] = cheby1(8, 0.05, 0.8/time_dec_pre);
lfp_filt = filtfilt(b_pre, a_pre, lfp_trials')';
lfp_filt = lfp_filt(:, 1:time_dec_pre:end);

% Get the number of points. We took twice the data we needed, so getting the highest power of two
%   will be like "rounding up".
n_points = 2^floor(log2(size(lfp_filt,2)));
if n_points == 2^log2(size(lfp_filt,2))
    n_points = n_points/2;
end

% Do atom deconstuction.
%   Data needs to be in TIME x TRIALS x CHANNELS
importData(lfp_filt.', 'mp_temp/', 'data/', [1,n_points], ...
           fs_lfp*1e3/time_dec_pre);
runGabor('mp_temp/', 'data/', n_points, 100)
gabor_info = getGaborData('mp_temp/', 'data/', 1);
rmdir('mp_temp/', 's')

% Contruct spectrograms from atoms
fun = @(x) reconstructEnergyFromAtomsMPP(x.gaborData, n_points, 1, []);
spec_temp = cellfun(fun, gabor_info, 'Un', 0);
spec = cat(3, spec_temp{:});

% Downsample spectrograms
[b_post, a_post] = butter(8, 0.8/time_dec_post);
spec_filt = permute(spec, [2, 1, 3]);
spec_filt = filtfilt(b_post, a_post, spec_filt);
spec_filt = permute(spec_filt, [2, 1, 3]);
spec_filt = spec_filt(:, 1:time_dec_post:end, :);

% Sort the data by trial type
for i_type = 1:numel(type_list)
    is_type = trial_type == type_list(i_type);
    spec_sorted(i_type,:,:) = mean(spec_filt(:,:,is_type), 3);
    spec_sorted_sd(i_type,:,:) = std(spec_filt(:,:,is_type), [], 3);
end

% Output data
fid = fopen(['data', suffix], 'w');
fprintf(fid, '%s\n', char(Params.commandline{1}));
fprintf(fid, '%s\n', Params.datadirectory);
fprintf(fid, '%s\n', Params.paramsstr);
fprintf(fid, '%s\n', sprintf('%d ', n_trials));
fprintf(fid, '%d %g %d %d\n', n_points, fs_lfp*1e3, time_dec_pre, ...
        time_dec_post);
for i_type = 1:size(spec_sorted, 1)
    for i_freq = 1:size(spec_sorted, 2)
        rowstr = sprintf('%.3g ', spec_sorted(i_type, i_freq, :));
        rowstr_sd = sprintf('%.3g ', spec_sorted_sd(i_type, i_freq, :));
        fprintf(fid, '%s\n', rowstr);
        fprintf(fid, '%s\n', rowstr_sd);
    end
end
fclose(fid);
