function timeSpikeLFP(suffix)
% TIME SPIKE LFP - Spike triggered average of LFP.
%

    % Read parameters file
    Params = read_params(suffix);
    time_start = Params.time_1;
    time_end = Params.time_2;
    pass_band = [Params.freq_start_1, Params.freq_end_1, Params.freq_step_1];
    if not(all(pass_band > 0))
        pass_band = [];
    end

    % Load lfp data
    [lfps, fs, trial_type, trial_index] = ...
            load_lfps(Params.data_directory, Params.command_line,   ...
                      Params.trial_file, Params.time_interval,      ...
                      Params.alpha_plex, Params.electrode_1,        ...
                      pass_band, Params.remove_mean);

    % Load spike data
    spikes = load_spikes(Params.trial_file, Params.spike_file,      ...
                         Params.time_interval, fs, trial_index);

    % Get all trial types and count nTrials
    type_list = unique(trial_type);
    n_trials = arrayfun(@(x) sum(trial_type == x), type_list);

    % Thin spikes: for each trial type, if there are more spikes than
    %   the thinning number, remove the excess.
    if Params.thin_spikes
        spikes = limit_spikes(spikes, trial_type, Params.n_spikes_limit,    ...
                              Params.seed_rng);
    end

    % Compute spike-triggered average
    sample_start = ceil(time_start * fs);
    sample_end = ceil(time_end * fs);
    for i_trial = 1:size(spikes, 1)

        % Find spikes that would allow their STA windows to fit in the
        %   interval.
        spike_indexes = find(spikes(i_trial, :) == 1);
        spike_indexes = spike_indexes(spike_indexes + sample_start > 0); 
        spike_indexes = spike_indexes(spike_indexes + sample_end    ...
                                      <= size(lfps, 2));

        % If there were no spikes, just make the average NaNs. 
        if isempty(spike_indexes)
            trial_sta(i_trial, :) = nan(1, sample_end - sample_start + 1);
        % If there ARE spikes, grab snippets around the spikes and average.
        else
            lfp_snippets = [];
            for i_spike = 1:numel(spike_indexes)
                next = size(lfp_snippets, 1) + 1;
                window = spike_indexes(i_spike) + (sample_start:sample_end);
                lfp_snippets(next, :) = lfps(i_trial, window);
            end
            trial_sta(i_trial, :) = mean(lfp_snippets, 1);
        end
    end

    % Now, average by trial type
    for i_type = 1:numel(type_list)
        is_type = trial_type == type_list(i_type);
        type_sta_mean(i_type, :) = nanmean(trial_sta(is_type, :), 1);
        type_sta_std(i_type, :) = nanstd(trial_sta(is_type, :), [], 1);
    end

    % Output data
    fid = fopen(['data', suffix], 'w');
    fprintf(fid, '%s\n', Params.command_line);
    fprintf(fid, '%s\n', Params.data_directory);
    fprintf(fid, '%s\n', Params.params_str);
    fprintf(fid, '%s\n', sprintf('%d ', n_trials));
    for i_type = 1:size(type_sta_mean, 1)
        fprintf(fid, '%s\n', sprintf('%10.5f ', type_sta_mean(i_type, :)));
        fprintf(fid, '%s\n', sprintf('%10.5f ', type_sta_std(i_type, :)));
    end
    fclose(fid);

end
