function powerLFP(suffix)
% LFP POWER - Compute LFP power for given time and frequency values.
%

    % Read parameters
    Params = read_params(suffix);
    freq = Params.freq_start_1 : Params.freq_step_1 : Params.freq_end_1;
    time_window = Params.time_1;
    time_step = time_window / 4;
    TW = Params.freq_start_2 / 10;
    n_tapers = floor(2 * TW - 1);

    % Load lfp data
    [lfps, fs, trial_type] =    ...
            load_lfps(Params.data_directory, Params.command_line,   ...
                      Params.trial_file, Params.time_interval,      ...
                      Params.alpha_plex, Params.electrode_1, [],    ...
                      Params.remove_mean); 

    % Get all trial types and count nTrials
    type_list = unique(trial_type);
    n_trials = arrayfun(@(x) sum(trial_type == x), type_list);

    % Compute spectrogram for each trial.
    if Params.chronux == 1

        addpath(genpath(['/data/code/grab/2018/_macros/lfp/chronux_2_12/',  ...
                         'spectral_analysis']))

        % Set up chronux parameters, which use sec and Hz for all values.
        movingwin = [time_window, time_step] / 1e3;
        Params_Chronux.tapers = [TW, n_tapers];
        Params_Chronux.Fs = fs * 1e3;
        Params_Chronux.fpass = freq([1, end]);

        [pxx, ~, freq_out] = mtspecgramc(lfps.', movingwin, Params_Chronux);
        pxx = permute(pxx, [3, 2, 1]);  % Trial x Freq x Time 
    else
        S = mt_dtft_gram(lfps, fs, time_window, time_step, freq, TW, n_tapers);
        pxx = mean(abs(S) .^ 2, 4) / fs / 1e3;
        freq_out = freq;
    end

    % Average by trial type.
    for i_type = 1:numel(type_list)
        is_type = trial_type == type_list(i_type);
        for i_freq = 1:length(freq_out)
            specgram_mean(i_type, i_freq, :) = ...
                    mean(pxx(is_type, i_freq, :), 1);
            specgram_std(i_type, i_freq, :) = ...
                    std(pxx(is_type, i_freq, :), [], 1);
        end
    end

    % Output data to a text file.
    fid = fopen(['data', suffix], 'w');
    fprintf(fid, '%s\n', char(Params.command_line));
    fprintf(fid, '%s\n', Params.data_directory);
    fprintf(fid, '%s\n', Params.params_str);
    fprintf(fid, '%s\n', sprintf('%d ', n_trials));
    fprintf(fid, '%s\n', sprintf('%g ', freq_out));
    for i_type = 1:size(specgram_mean, 1)
        for i_freq = 1:size(specgram_mean, 2)
            fprintf(fid, '%s\n',    ...
                    sprintf(['%10.5f '], specgram_mean(i_type, i_freq, :)));
            fprintf(fid, '%s\n',    ...
                    sprintf(['%10.5f '], specgram_std(i_type, i_freq, :)));
        end
    end
    fclose(fid);

end
