function cohSpikeLFP(suffix)
% WAVELET ANALYSIS
%   Determine phase consistency between LFP and spikes over the frequency
%   spectrum. LFPs are decomposed with the wavelet transform.
%

    % Read parameters
    Params = read_params(suffix);
    freq = Params.freq_start_1 : Params.freq_step_1 : Params.freq_end_1;

    % Read in LFP data
    % Load lfp data
    [lfps, fs, trial_type, trial_index] = ...
            load_lfps(Params.data_directory, Params.command_line,   ...
                      Params.trial_file, Params.time_interval,      ...
                      Params.alpha_plex, Params.electrode_1, [],    ...
                      Params.remove_mean);

    % Load spike data
    spikes = load_spikes(Params.trial_file, Params.spike_file,      ...
                         Params.time_interval, fs, trial_index);

    % Get all trial types and count nTrials
    type_list = unique(trial_type);
    n_trials = arrayfun(@(x) sum(trial_type == x), type_list);

    % Thin spikes: for each trial type, if there are more spikes than
    %   the thinning number, remove the excess.
    if Params.thin_spikes
        spikes = limit_spikes(spikes, trial_type, Params.n_spikes_limit,    ...
                              Params.seed_rng);
    end

    % Compute wavelet transform
    [freq_cent, wt] = wavelet_transform(lfps, fs, freq);

    % Compute PLV and coherence for each trial type
    for i_type = 1:numel(type_list)
        these_wt = wt(trial_type == type_list(i_type), :, :);
        these_spikes = spikes(trial_type == type_list(i_type), :);

		% Compute phase-locking values and coherence
		for i_freq = 1:numel(freq_cent)
			this_wt = permute(these_wt(:, i_freq, :), [1, 3, 2]);
			wt_spike = this_wt(find(these_spikes));
			coh(i_type, i_freq) = mean(wt_spike) / mean(abs(wt_spike)); 
			plv(i_type, i_freq) = mean(wt_spike ./ abs(wt_spike));
		end

        % PLV stats
        N = sum(these_spikes(:));
        R = abs(plv(i_type, :)) * N;
        pval(i_type, :) =   ...
                exp(sqrt(1 + 4*N + 4*((ones(size(R))*N).^2 - R.^2))     ...
                    - (1 + 2*N));
    end

    % Output data
    fid = fopen(['data', suffix], 'w');
    fprintf(fid, '%s\n', Params.command_line);
    fprintf(fid, '%s\n', Params.data_directory);
    fprintf(fid, '%s\n', Params.params_str);
    fprintf(fid, '\n');
    fprintf(fid, 'Freq  Coh  Coh_Angle  Sync  Sync_Angle  Pval  Type\n');
    for i_type = 1:length(type_list)
        fprintf(fid, '%g %g %g %g %g %g %d\n',     ...
                [freq_cent;
                 abs(coh(i_type, :)); angle(coh(i_type, :));
                 abs(plv(i_type, :)); angle(plv(i_type, :));
                 pval(i_type, :);
                 ones(size(freq_cent)) * type_list(i_type)]);
    end
    fclose(fid);
    
end
