function cohLFP(suffix)
% LFP COHERENCE - Compute LFP-LFP coherence for given time and 
%       frequency values.
%

    % Read parameters
    Params = read_params(suffix);
    freq = Params.freq_start_1 : Params.freq_step_1 : Params.freq_end_1;
    time_window = Params.time_1;
    time_step = time_window / 4;
    TW = Params.freq_start_2 / 10;
    n_tapers = floor(2 * TW - 1);

    % Load lfp data
    [lfps_1, fs, trial_type] =  ...
            load_lfps(Params.data_directory, Params.command_line,   ...
                      Params.trial_file, Params.time_interval,      ...
                      Params.alpha_plex, Params.electrode_1, [],    ...
                      Params.remove_mean);
    lfps_2 = load_lfps(Params.data_directory, Params.command_line,  ...
                       Params.trial_file, Params.time_interval,     ...
                       Params.alpha_plex, Params.electrode_2, [],   ...
                       Params.remove_mean);

    % Get all trial types and trial counts for each type.
    type_list = unique(trial_type);
    n_trials = arrayfun(@(x) sum(trial_type == x), type_list);

    if max(n_trials) ~= min(n_trials)
        warning('Trial counts vary across stack!')
    end

    % Generate spectral estimates
    if Params.chronux == 1

        addpath(genpath(['/data/code/grab/2018/_macros/lfp/chronux_2_12/',  ...
                         'spectral_analysis']))

        % Set up chronux parameters, which use sec and Hz for all values.
        movingwin = [time_window, time_step] / 1e3;
        Params_Chronux.tapers = [TW, n_tapers];
        Params_Chronux.Fs = fs * 1e3;
        Params_Chronux.fpass = freq([1, end]);
        Params_Chronux.trialave = 1;

        % Compute average by trial type.
        for i_type = 1:numel(type_list)
            is_type = trial_type == type_list(i_type);
            [C, phi, ~, ~, ~, ~, freq_out] =    ...
                    cohgramc(lfps_1(is_type, :).', lfps_2(is_type, :).',    ...
                             movingwin, Params_Chronux);
            coh_mag(i_type, :, :) = C.';    % Type x Freq x Time
            coh_angle(i_type, :, :) = phi.';
        end

    else

        % Spectra
        S1 = mt_dtft_gram(lfps_1, fs, time_window, time_step, freq, TW,     ...
                          n_tapers);
        S2 = mt_dtft_gram(lfps_2, fs, time_window, time_step, freq, TW,     ...
                          n_tapers);
        freq_out = freq;

        % Cross- and auto-spectra.
        S12 = mean(S1 .* conj(S2), 4);
        S11 = mean(abs(S1) .^ 2, 4);
        S22 = mean(abs(S2) .^ 2, 4);

        % Average by trial type
        for i_type = 1:numel(type_list)
            is_type = trial_type == type_list(i_type);
            for i_freq = 1:numel(freq_out)
                S12_mean = mean(S12(is_type, i_freq, :), 1);
                S11_mean = mean(S11(is_type, i_freq, :), 1);
                S22_mean = mean(S22(is_type, i_freq, :), 1);
                coh(i_type, i_freq, :) = S12_mean   ...
                                         ./ sqrt(S11_mean .* S22_mean);
            end
        end

        % Break down into magnitude and angle
        coh_mag = abs(coh);
        coh_angle = angle(coh);
    end

    % Output data
    fid = fopen(['data', suffix], 'w');
    fprintf(fid, '%s\n', char(Params.command_line));
    fprintf(fid, '%s\n', Params.data_directory);
    fprintf(fid, '%s\n', Params.params_str);
    fprintf(fid, '%s\n', sprintf('%d ', n_trials));
    fprintf(fid, '%s\n', sprintf('%g ', freq_out));
    for i_type = 1:size(coh_mag, 1)
        for i_freq = 1:size(coh_mag, 2)
            fprintf(fid, '%s\n',    ...
                    sprintf('%1.10f ', coh_mag(i_type, i_freq, :)));
            fprintf(fid, '%s\n',    ...
                    sprintf('%3.5f ', rad2deg(coh_angle(i_type, i_freq, :))));
        end
    end
    fclose(fid);

end
