/* FILE _macros/print.c */

#include "../Deffs.h"

/* ********************************************************************	*/
     /* Generic macro: print spike rates in specified interval,
	sorted by stack and class */
     /* Stack name: ounit#.run#
      * First line: unit run begin time end time # stacks # classes # trials
      * Then (for each class):
		 stack, table, body, head, targ, mean, se, n; then indi data
	   NOTE: 1st line gives maximum # of trials; for each class,
		 'n' gives actual # of trials.  Indi data is padded
		 out with 'NAs' to equal the maximum (otherwise would
		 be much more complex to read in data!).
           NOTE: mean is written as Hz; indi data is number of spikes
                 Hz = # spikes / interval
     */

/* ********************************************************************	*/
/* ********************************************************************	*/

/* FUNCTION Write_OutputFiles */
	 /* Write individual trial data out for Splus import	*/
	 /* Append any arguments to 'title' to end of file	*/
void Write_OutputFiles0() {
	int Types   = Count_TrialTypes();
	int MaxTrials = -1;
	int i;
	FILE *file;
   	char Name[80];

	Rewind_InputFile();		/* For consistent unit,run#	*/
   	sprintf(Name, "o%d.%d", UnitNumber_Header(), RunNumber_Header());
	file = Open_Macro_Output(Name, -1);

     	/* First line: unit.run begin(ms) end(ms) stacks classes trials	*/
	fprintf(file, "Unit.run  %d.%d\n",
   		UnitNumber_Header(), RunNumber_Header());
	fprintf(file, "\t%d stacks x %d classes\n",
		List_Length(STACK), List_Length(CLASS));
	fprintf(file, "\tInterval: %d to %d ms",
		Get_Interval_Begin(1), Get_Interval_End(1));
	if (Get_Interval_Type() == 2)
	   fprintf(file, " minus %d to %d ms",
		Get_Interval_Begin(2), Get_Interval_End(2));
	if (Get_Interval_Type() == 3)
	   fprintf(file, "   (2nd interval selected, not printed.)");
        fprintf(file, "\n\t%s\n", AlignTime_String());

	fprintf(file,
"Stack class\t mean\t SD \t SE \tN \t\t spikes per trial)\n");
		
	for (i=0; i<Types; i++)		/* Find max trials per class	*/
	    if (MaxTrials < Get_Interval_Count(i,1))
		MaxTrials = Get_Interval_Count(i,1);

	for (i=0; i<Types; i++) {
	    int count = Get_Interval_Count(i,1);
	    int *data = Get_Interval_Data(i,1);
	    int j;

	    fprintf(file, " %3d\t%2d\t%5.2f\t%4.2f\t%4.2f\t%2d \t\t",
		Get_TrialType_Info(i,STACK),
		Get_TrialType_Info(i,CLASS),

		Get_Interval_Mean(i,1),
		Get_Interval_SD(i,1),
		Get_Interval_SEM(i,1),
		count);

	    /* Now print spikes in interval for each trial	*/
	    for (j=0; j<count; j++)
	       fprintf(file, " %d", *(data+j));
	    /* while (++j<MaxTrials)
	       fprintf(file, " NA");
	     */
		 
	    fprintf(file, "\n");
	    }

		/* If there's a title, append it to output file	*/
	if (Get_Title_From_CmdLine())
	   fprintf(file, "%s\n", Get_Title_From_CmdLine());

	fclose(file);
	}
/* ********************************************************************	*/

void Write_OutputFiles1() {
	int Stacks = List_Length(STACK);
	int Classes = List_Length(CLASS);
	int s,c;
	FILE *file;

	Rewind_InputFile();		/* For consistent unit,run#	*/
	file = Open_Macro_Output("StackMeans.out", -1);

	for (s=0; s<Stacks; s++)
	  for (c=0; c<Classes; c++) {
	    int stack = List_Element(s, STACK);
	    int class = List_Element(c, CLASS);
	    int type = Get_StackClass_TrialType(stack, class);
	    if (type != FAIL)
	      fprintf(file, "%d %d %d %.2f\n",
		UnitNumber_Header(),
		stack, class,
 		Get_Interval_Mean(type,1));
	    }
	fclose(file);
	}
/* ********************************************************************	*/

/* FUNCTION Print_Spikes140() */
	 /* Set -d or -D to include unit number, run number and trial number */
void Print_Spikes140() {
	FILE *file;
	int IntervalType = Get_Interval_Type();
	int ShowMore = Get_DataValue_From_CmdLine() ||
		       Get_dataValue_From_CmdLine();

	if (IntervalType == 3) {
      	   fprintf(stderr, "Cannot use '-I' with macro 140\n");
      	   Exit("", "Write_OutputFiles140()");
      	   }

	Rewind_InputFile();		/* For consistent unit,run#	*/
	file = Open_Macro_Output("Spikes.out", -1);

	/* Loop thru data: */
	while (Read_Next_Trial(WITH_DATA)) {
	   extern int Spikes_In_Interval(int interval);
	   if (ShowMore)
	      fprintf(file, "%d %d %d ",
		UnitNumber_Header(), RunNumber_Header(), TrialNumber_Header());
	   fprintf(file, "%d %d %d\n",
		   StackNumber_Header() ,
		   TableNumber_Header() ,
		   ((IntervalType==2) ? 
		      Spikes_In_Interval(1) - Spikes_In_Interval(2) :
		      Spikes_In_Interval(1))
		   ) ;
	   }
	fclose(file);
	}
/* ********************************************************************	*/

/* FUNCTION Print_SpikeTimes141 */
	 /* Write individual trial data out for Splus import	*/
	 /* macro 1411 = this with includeheader=1; 1410 is incldhdr = 0*/
void Print_SpikeTimes141(int IncludeHeader) {
	FILE *file;

	Rewind_InputFile();		/* For consistent unit,run#	*/
	file = Open_Macro_Output("Spikes", 1);

       while (Read_Next_Trial(WITH_DATA)) {
  	   extern short Spikes[];
  	   int SpikeCount = SpikeCount_Header(-1);
  	   short *spikes = Spikes;
  	   int begin = Get_Interval_Begin(1);
  	   int end   = Get_Interval_End(1);
  	   int i = 0;
	   int firstspike = 1;

	   if (IncludeHeader)
              fprintf(file, " %4d %3d %3d %2d ", 
		     	UnitNumber_Header(), RunNumber_Header(),
			StackNumber_Header(),TableNumber_Header());

           fprintf(file, " %d", TrialNumber_Header());

           while (i++ < SpikeCount) {
	       int time = (int) *spikes - Get_ZeroTime();
	       spikes++;

	       if (time < begin)
	          continue;
	       if (firstspike) {		/* Write PREVIOUS	*/
	          fprintf(file, " %d" , *(spikes-2) - Get_ZeroTime());
		  firstspike = 0;		/* Will be negative	*/
		  }
	       if (time > end) {		/* Write & leave	*/
	          fprintf(file, " %d" , time);
	          break;
		  }
	       fprintf(file, " %d" , time);
    	       }
	   fprintf(file, " 999999\n");
           }
	fprintf(file, "\n");
	fclose(file);
	}
/* ********************************************************************	*/

/* FUNCTION Print_FileInfo143() */
	 /* Set -d or -D to include unit number & run number */
void Print_Info143() {
	int ShowMore = Get_DataValue_From_CmdLine() ||
		       Get_dataValue_From_CmdLine();
	FILE *file;

	Rewind_InputFile();		/* For consistent unit,run#	*/
	file = Open_Macro_Output("FileInfo.out", -1);

	/* Loop thru data: */
	while (Read_Next_Trial(WITH_DATA)) {
	   if (ShowMore)
	      fprintf(file, "%d %d  ",
		UnitNumber_Header(), RunNumber_Header());
	   fprintf(file, "%d %d %d\n",
	   	TrialNumber_Header(),
		StackNumber_Header(),
		TableNumber_Header()
	   	);
	   }

	fclose(file);
	}
/* ********************************************************************	*/

/* FUNCTION Print_EventExtract145() */
	 /* Defaults: TIME (-d0) of 1st occurrence (-D0); APPENDS	*/
	 /* (Different from grab3 version! */
void Print_EventExtract145() {
	FILE *file;
	int Event = EventNameToNumber(Get_Title_From_CmdLine());
	int ParamNumber = Get_dataValue_From_CmdLine(); /* -d	*/
					/* 0==TIME, 1:3 == params 1:3	*/
	int Order = Get_DataValue_From_CmdLine();	/* -D	*/
					/* 99 == last, 98 2nd to last..	*/

	if (Order == 0)
	    Order = 1;

	if (Event == FAIL) {
      	   fprintf(stderr,"Cannot map %s to an event",Get_Title_From_CmdLine());
      	   Exit("", "Print_EventExtract145()");
      	   }
				
	Rewind_InputFile();		/* For consistent unit,run#	*/
	file = Open_Macro_Output("EventExtract.out", -1);

	/* Loop thru data: */
	while (Read_Next_Trial(WITH_DATA))
	   fprintf(file, "%d\n", EventExtract(Event, ParamNumber, Order));

	fclose(file);
	}
/* ********************************************************************	*/

/* FUNCTION Print_EventExtract147() */
	 /* Prints time, param1, param2, param3; APPENDS	*/
	 /* Prints -D# occurrence (default = 1st)		*/
void Print_EventExtract147() {
	FILE *file;
	int Event = EventNameToNumber(Get_Title_From_CmdLine());
	int Order = Get_DataValue_From_CmdLine();
					/* 99 == last, 98 2nd to last..	*/

	if (Order == 0)			/* Can use either -d OR -D	*/
	    Order = Get_dataValue_From_CmdLine();

	if (Order == 0)
	    Order = 1;

	Rewind_InputFile();		/* For consistent unit,run#	*/
	file = Open_Macro_Output("EventExtract.out", -1);

	/* Loop thru data: */
	while (Read_Next_Trial(WITH_DATA))
	   fprintf(file, "%hd  %d %d %d\n", 
	   	EventExtract(Event, TIME, Order),
	   	EventExtract(Event, ONE, Order),
	   	EventExtract(Event, TWO, Order),
	   	EventExtract(Event, THREE, Order)
		);

	fclose(file);
	}
/* ********************************************************************	*/

/* FUNCTION Print_EventExtract148() */
	 /* Print trial#, time, param1, param2, param3; OVERWRITES	*/
	 /* Prints ALL INSTANCES				*/
void Print_EventExtractAll148() {
	FILE *file;
	int Event = EventNameToNumber(Get_Title_From_CmdLine());
	int i;

	Rewind_InputFile();		/* For consistent unit,run#	*/
	file = Open_Macro_Output("EventExtract.out", -1);

	/* Loop thru data: */
	while (Read_Next_Trial(WITH_DATA))
	   for (i=1; i<10; i++) {	/* Always show at least one	*/
	      fprintf(file, "%3hd %3d   %2d    %5d  %d %d %d\n", 
		TrialNumber_Input(), TrialNumber_Header(),
		i,
	   	EventExtract(Event, TIME, i),
	   	EventExtract(Event, ONE, i),
	   	EventExtract(Event, TWO, i),
	   	EventExtract(Event, THREE, i)
		);
	      if (EventExtract(Event, TIME, i+1) == FAIL)
		 break;
	      }

	fclose(file);
	}
/* ********************************************************************	*/

/* FUNCTION Stats142() */
 	 /* Mean, SE and 'n' of each trial type, straight to stderr */
void Stats142() {
	int Stacks = List_Length(STACK);
	int Classes = List_Length(CLASS);
	int s,c;
	FILE *file;

	Rewind_InputFile();		/* For consistent unit,run#	*/
	file = Open_Macro_Output("Stats.out", -1);

	for (s=0; s<Stacks; s++)
	  for (c=0; c<Classes; c++) {
	    int stack = List_Element(s, STACK);
	    int class = List_Element(c, CLASS);
	    int type = Get_StackClass_TrialType(stack, class);
	    if (type != FAIL)
	      fprintf(file, "%d %d %d %.2f %.2f %d\n",
		UnitNumber_Header(),
		stack, class,
 		Get_Interval_Mean(type,1),
		(Get_Interval_Count(type,1) == 1) ?  -9999 :
						Get_Interval_SEM(type, 1),
		Get_Interval_Count(type, 1)
		);
	    }
	fclose(file);
	}
/* ********************************************************************	*/

/* FUNCTION CountSpikes152() */
 	 /* Spike count by stack/class, OR just the minimum (-oM)	*/
void CountSpikes152() {
	int Stacks = List_Length(STACK);
	int Classes = List_Length(CLASS);
	int Minimum = (Get_oValue(1,'M',-99) != -99);		/* Flag?*/
	float ConvertToCount = 1000/(Get_Interval_End(1)-Get_Interval_Begin(1));
	int s,c;
	int min = 999999;
	FILE *file;

	Rewind_InputFile();		/* For consistent unit,run#	*/
	file = Open_Macro_Output("Stats.out", -1);

	for (s=0; s<Stacks; s++)
	  for (c=0; c<Classes; c++) {
	    int stack = List_Element(s, STACK);
	    int class = List_Element(c, CLASS);
	    int type = Get_StackClass_TrialType(stack, class);
	    if (type != FAIL) {
	      int count = roundf(Get_Interval_Mean(type,1) *
			         Get_Interval_Count(type,1) *
				 ConvertToCount);
	      if (Minimum) {
		 if (count < min)
		     min = count;
	       } else
	         fprintf(file, "%d %d %d\n", stack, class, count);
	      }
	    }
	if (Minimum)
	   fprintf(file, "%d\n", min);
	fclose(file);
	}
/* ********************************************************************	*/
#define PAD_TO_MAXIMUM	36		/* Always give this many trials	*/


void Write_OutputFiles2() {
	int Types   = Count_TrialTypes();
	int i;
	FILE *file;

	Rewind_InputFile();		/* For consistent unit,run#	*/
	file = Open_Macro_Output("StackCounts.out", -1);

	for (i=0; i<Types; i++) {
	    int count = Get_Interval_Count(i,1);
	    int *data = Get_Interval_Data(i,1);
	    int j;
	    fprintf(file, "%d %d %2d %2d ",
		UnitNumber_Header(), Get_TrialType_Info(i,STACK), 
		Get_TrialType_Info(i,CLASS), count);

	    if (PAD_TO_MAXIMUM && count > PAD_TO_MAXIMUM) {
		fprintf(stderr, "Exceeded PAD_TO_MAXIMUM (%d) by %d trials\n",
			PAD_TO_MAXIMUM, count-PAD_TO_MAXIMUM);
	        count = PAD_TO_MAXIMUM;
	        }
	
	
	    /* Now print spikes in interval for each trial	*/
	    for (j=0; j<count; j++)
	       fprintf(file, " %d", *(data+j));
	    if (PAD_TO_MAXIMUM)
	       while (j++ < PAD_TO_MAXIMUM)
	          fprintf(file, " NA");
		 
	    fprintf(file, "\n");
	    }

	fclose(file);
	}

/* ********************************************************************	*/
void Write_OutputFiles3() {
	float x, y;

	Rewind_InputFile();		/* For consistent unit,run#	*/
	Get_ArmPosition(Get_Interval_Begin(1), Get_Interval_End(1), &x, &y);
	fprintf(stderr, "Arm position %d to %d ms is (%.2f,%.2f)\n",
		Get_Interval_Begin(1), Get_Interval_End(1),
		x,y);
	}
/* ********************************************************************	*/

/* FUNCTION AnalogDataDump */
void AnalogDataDump156() {
   extern int16_t Data[];
   int channels = ChannelCount_Header();
   int frames = FrameCount_Header();
   int i,j,k=0;
   int number = 1;
   FILE *file = NULL;
   char Name[80];

   Rewind_InputFile();		/* For consistent unit,run#	*/
   while (Read_Next_Trial(WITH_DATA)) {
      sprintf(Name, "out.%d", number++);
      file = Open_Macro_Output(Name, -1);

      k = 0;
      for (i=0; i<frames; i++) {
	for (j=0; j<channels; j++, k++)
            fprintf(file, "%d ", *(Data+k));
	fprintf(file, "\n");
        }
      fprintf(file, "\n");
      }
   fclose(file);
   }
/* ********************************************************************	*/
