function getTrajectory( filename, varargin )
% function getTrajectory( filename, varargin )
% This function reads in a filename of a trajectory file and outputs the
% polynomial fit movements for those trials
% Flags may be passed to override default values, flags are defined below
% and are separated from their values by a space.
%
% -t [0|1]: boolean to trim to grab computed movement start. Default = 0
% -s : value for threshold to detect movement. Default = 0.075
% -T : stack number to extract.  0 is wildcard.  If specified, must
% specify -D and -R as well. Default = 0
% -D : direction number to extract.  Must specify -T and -R as well.
% Default = null
% -R : rotation value to extract.  Must specify -T and -D as well. Default
% = null
% -o [1|2|3|4|5]: Order of poly to fit. Default = 3
% -b [0|1]: boolean to baseline correct movement trajectories, ie make
% initial point (0,0). Default = 1
% -n : number of data points per trial to write. Default = 100

warning off all

display('Setting Constants');
% SET CONSTANTS (will be overwritten if argument passed)
TRIM = 0;  %Trims the movement to the grab determined start of movement
SENS = .075; %Sensitivity for detecting movement
TYPE = 0; %0 = wildcard for stacks to run, runs on all
DIR = 0; %Direction of stacks to extract, not needed unless type ~= 0
ROT = 0; %Same as DIR
ORDER = 3; %Order of polynomial to fit
BC = 1; %Baseline correct the movement
N_T = 100; %Number of timepoints to write out


display('Checking Inputs');
% Check Inputs
narg = size(varargin,2);
if mod(narg,2) == 1
    error('Each flag must be paired with a value, include a space');
end
if narg
    for x = 1:2:(narg/2)+1
        if strcmp(varargin(x),'-t')
            if isstr(varargin(x+1)) in = str2num(varargin(x+1));else in = varargin(x+1);end;
            in = cell2mat(in);
            if in
                TRIM = 1;
            else
                TRIM = 0;
            end
        elseif strcmp(varargin(x),'-s')
            if isstr(varargin(x+1)) in = str2num(varargin(x+1));else in = varargin(x+1);end;
            SENS = cell2mat(in);
        elseif strcmp(varargin(x),'-T')
            if isstr(varargin(x+1)) in = str2num(varargin(x+1));else in = varargin(x+1);end;
            TYPE = cell2mat(in);
        elseif strcmp(varargin(x),'-D')
            if isstr(varargin(x+1)) in = str2num(varargin(x+1));else in = varargin(x+1);end;
            DIR = cell2mat(in);
        elseif strcmp(varargin(x),'-R')
            if isstr(varargin(x+1)) in = str2num(varargin(x+1));else in = varargin(x+1);end;
            ROT = cell2mat(in);
        elseif strcmp(varargin(x),'-o')
            if isstr(varargin(x+1)) in = str2num(varargin(x+1));else in = varargin(x+1);end;
            ORDER = cell2mat(in);
        elseif strcmp(varargin(x),'-b')
            if isstr(varargin(x+1)) in = str2num(varargin(x+1));else in = varargin(x+1);end;
            in = cell2mat(in);
            if in
                BC = 1;
            else
                BC = 0;
            end
        elseif strcmp(varargin(x),'-n')
            if isstr(varargin(x+1)) in = str2num(varargin(x+1));else in = varargin(x+1);end;
            N_T = cell2mat(in);
        else
            error('Flag not recognized, see help')
            
        end
        
        
    end
end

if (TYPE||DIR||ROT) && ~(TYPE&&DIR&&ROT)
    error('If -T -D or -R specified, all must be specified')
end
display('Inputs Passed');

% Read in File
display('Reading In File');
RUN = Traj_Parse(filename);


%Trim trial length
if TRIM
    display('Trimming Data');
    RUN = trimRun(RUN);
end

display('Processing Trials');

for x = 1:length(RUN.trial)
    RUN.trial{x}.bad = isBadTrial(RUN,x);
    RUN.trial{x}.movements = findMovements(RUN,x,SENS,0);
    RUN.trial{x}.bad = isBadTrial(RUN,x);
    RUN.trial{x}.first_movement = extractMovement(RUN,x,1,0,0);
    RUN.trial{x}.bad = isBadTrial(RUN,x);
end

%Removing Bad trials
display('Removing Bad Trials');
c = 0;
RUN_cut = {};
for x = 1:length(RUN.trial)
    if ~RUN.trial{x}.bad
        c = c + 1;
        RUN_cut.trial{c} = RUN.trial{x};
    end
end

RUN.header(1) = c;		%  Added by LHS 8-10-2011
RUN.header(2) = N_T;		%  Added by LHS 8-11-2011 -- number of points
RUN_cut.header = RUN.header;    %  Added by LHS 8-10-2011
RUN = RUN_cut;

display('Extracting Movements');
movements = extractTrialType(RUN,TYPE,DIR,ROT);
[single_fit,single_buffer,p] = fitSingleTrialType(movements,1,ORDER,BC,N_T);
display('Writing Out File');
writeFile(filename,RUN,single_fit);
display('Done');

end

function [ RUN ] = Traj_Parse( fileName )

% function [ RUN ] = Traj_Parse( fileName )
% Brier, MR.
% Created 6/13/11
% Last Edited 6/17/11
%
% fileName - name of file that is output as a result of grab macro
% RUN - structure of trials and time points
%
% This function reads in a filename for a Trajectory file created using
% macro 161 of grab and creates a RUN data structure.  The RUN data
% structure has in it a header that contains the information at the head of
% the trajectory file.  It has a trial cell array that contains the
% information in the individual trials.  Also, there may be an angle field
% that contains the rotation angle (not yet available).
%
% Each trial cell contains a matrix points that is nX5, column one is
% timestamp, col 2 and 4 are x pos, col 3 and 5 are y pos.

F = fopen(fileName);

n=0;

input = fgetl(F);  %grabs header line

[token remain] = strtok(input);  %tokens off first number
RUN.header(1) = str2num(token);  %changes string to number and puts in header
[token remain] = strtok(remain);
RUN.header(2) = str2num(token);
[token remain] = strtok(remain);
RUN.header(3) = str2num(token);


input = fgetl(F);  %burns blank line

t = 0; % lcv for trial count

n_c = [];

while ~feof(F)
    
    t = t + 1; %since new trial loop, incr t
    
    
    input = fgetl(F); % grabs header line of trial
    remain = input; % puts it into remain so can loop on remain not being empty
    
    n = 0; % lcv for loading trial header
    if isempty(remain)
        input = fgetl(F);
        remain = input;
    end
    while ~isempty(remain) % loop while remain has something in it to parse off header line
        n = n + 1; % since new token of trial header, incr n
        [token remain] = strtok(remain); % grab the first token
        if ~strcmp(token,'NA') % as best i can tell, NA has no meaning
            RUN.trial{t}.header(n) = str2num(token); % create the vector of header values
        end
    end
    
    n = 0; % lcv for time point number
    input = fgetl(F);
    while ~isempty(input)
        remain = input;
        
        if (input(1) ~= 'N')
            while ~isempty(remain)
                n = n + 1;                
                [token remain] = strtok(remain);
                RUN.trial{t}.point(n,1) = str2num(token);
                [token remain] = strtok(remain);
                RUN.trial{t}.point(n,2) = str2num(token);
                [token remain] = strtok(remain);
                RUN.trial{t}.point(n,3) = str2num(token);
                [token remain] = strtok(remain);
                RUN.trial{t}.point(n,4) = str2num(token);
                [token remain] = strtok(remain);
                RUN.trial{t}.point(n,5) = str2num(token);
            end
            input = fgetl(F);
        else
            input = fgetl(F);
        end

    end

    n_c(t) = n;
end

fclose(F);
end
function [ RUN ] = trimRun( RUN )
%UNTITLED Summary of this function goes here
%   Detailed explanation goes here

for x = 1 : length(RUN.trial)
    n = 1;
    while RUN.trial{x}.point(n,1) < RUN.trial{x}.header(9)
        n = n+1;
    end
    RUN.trial{x}.point = RUN.trial{x}.point(n:end,:);
end


end
function [ bad ] = isBadTrial( RUN, trial )

% function [ bad ] = isBadTrial( RUN, trial )
% Brier, MR.
%
% RUN - RUN data structure
% trial - the specific trial in question
% bad - boolean value if criteria meant
%
% This will check all possible reasons for it being a bad trial, but
% depending on when it is run it will only get so far.  Individual modules
% are commented in isBadTrial.m

bad = 0;

% Exclude positions that are too large to be reasonably within the reach
% area
try
    THRESH = 50;
    
    %Exclude large values
    
    x = RUN.trial{trial}.point(:,4); % grab x values in degrees (check this)
    y = RUN.trial{trial}.point(:,5); % grab y values in degrees (check this)
    
    xt = abs(x) > THRESH;
    yt = abs(y) > THRESH;
    
    if sum(xt) || sum(yt)
        bad = 1;
    end
    
catch
end

% Exclude first_movements that are long
try
    THRESH = 200;
    
    t = RUN.trial{trial}.first_movement(:,1); % grab x values in degrees (check this)
    t_mov = t(end) - t(1);
    if t_mov > THRESH
        bad = 2;
    end
catch
end

% exclude movments that are small
try
    
    THRESH = 5;
    x = RUN.trial{trial}.first_movement(:,4);
    y = RUN.trial{trial}.first_movement(:,5);
    
    mag = sqrt(((x(end)-x(1)).^2)+((y(end)-y(1)).^2));
    if mag < THRESH
        bad = 3;
    end
catch
end

%if movements is plain empty, thats bad
try
    RUN.trial{trial}.movements;
    if isempty(RUN.trial{trial}.movements)
        bad = 4;
    end
catch
end

end
function [ movements ] = findMovements( RUN, trial, sens, display )

% function [ movements ] = findMovements( RUN, trial, sens, display )
% Brier, MR
% 
% RUN - RUN structure
% trial - the trial number that wil be operated on
% sens - the sensitivity threshold NB. 0.075 seems to work well
% display - boolean for making figure or not.
% movements - movement data structure
%
% This function operates on a single trial.  It first reads in the RUN and
% extracts the record for a given trial.  It finds points where the
% magnitude of change between them per unit time exceeds the threshold and
% designates those times as moving.  The function then smooths those
% results.

movements = []; % initialize output

t = RUN.trial{trial}.point(:,1); % grab time points
x = RUN.trial{trial}.point(:,4); % grab x values in degrees (check this)
y = RUN.trial{trial}.point(:,5); % grab y values in degrees (check this)

dt = diff(t); % numerical differentials
dx = diff(x); % numerical differentials
dy = diff(y); % numerical differentials

dx_dt = dx./dt; % calculate derivatives
dy_dt = dy./dt; % calculate derivatives

dxy_dt = sqrt(dx_dt.^2+dy_dt.^2); % get a magnitude of the two dimensional movements

moving = dxy_dt > sens; % declare points above a threshold as moving between given time points

if display  % make display
    figure
    subplot(4,1,1:3)
    plot(t,x)
    hold on
    plot(t,y,'r')
    axis tight
    set(gcf,'color','w')
    ylabel('Movement','fontname','times','fontsize',14)
    title('Raw Movement Detection','fontname','times','fontsize',14)
    subplot(4,1,4)
    scatter(t(1:end-1),moving,'filled')
    axis ([t(1) t(end) -.1 1.1])
    ylabel('Detection','fontname','times','fontsize',14)
    xlabel('t (msec)','fontname','times','fontsize',14)
end

% loop through moving and remove spurious single time frame non-movements
for i = 1:length(moving)
    if ~moving(i)
        if i == 1
            % do nothing
        elseif i == length(moving)
            % do nothing
        else
            if moving(i-1) && moving(i+1)
                moving(i) = 1;
            end
        end
    end
end

% loop through moving and remove spurious single time frame movements
for i = 1:length(moving)
    if moving(i)
        if i == 1
            if ~moving(2)
                moving(i) = 0;
            end
        elseif i == 2
            if ~moving(1) && ~moving(3) && ~moving(4)
                moving(i) = 0;
            end
        elseif i == length(moving)
            if ~moving(length(moving)-1)
                moving(i) = 0;
            end
        elseif i == length(moving) - 1
            if ~moving(length(moving)) && ~moving(length(moving)-2) && ~moving(length(moving)-3)
                moving(i) = 0;
            end
        else
            if ~moving(i-1) && ~moving(i+1)
                moving(i) = 0;
            end
        end
    end
end
                
if display  % make display
    figure
    subplot(4,1,1:3)
    plot(t,x)
    hold on
    plot(t,y,'r')
    axis tight
    set(gcf,'color','w')
    ylabel('Movement','fontname','times','fontsize',14)
    title('Single Corrected Movement Detection','fontname','times','fontsize',14)
    subplot(4,1,4)
    scatter(t(1:end-1),moving,'filled')
    axis ([t(1) t(end) -.1 1.1])
    ylabel('Detection','fontname','times','fontsize',14)
    xlabel('t (msec)','fontname','times','fontsize',14)
end      


% extract individual movements and shove into movements
% movements wil by a 2x2xc array.  Row 1 will be in time index, row 2 will
% be in time points.  Col 1 will be movement start, col 2 will be movement
% end.  c will index movements in trial.

i = 1;
c = 0;

while i < (length(moving) + 1)
    if moving(i)
        c = c + 1;
        movements(1,1,c) = i;
        movements(2,1,c) = t(i);
        i = i + 1;
        while i <length(moving) && moving(i)
            i = i + 1;
        end
        movements(1,2,c) = i;
        movements(2,2,c) = t(i);
    else
        i = i + 1;
    end
end


end
function [ points ] = extractMovement( RUN, trial, movement, display, buffer )

% function [ points ] = extractMovement( RUN, trial, movement, display,
% buffer )
% Brier, MR
% created 6/13/11
% last edited 6/17/11
%
% RUN - The RUN data structure from which you want the movement extracted
% trial - the trial number within run for which you want the movement
% extracted
% movement - the movements will be numbered 1,...,N
% display - boolean, create figs?
% buffer - an ill concienved tool to ass time to the end and beginning of
% movements.  It time points, not msec.
% points - the nX5 matrix containing a single movement
%
% N.B. this must be run after findMovement
%
% This function uses the data from findMovement to extract a single arm
% movement from the trial.



if RUN.trial{trial}.bad
    points = [];
    return
end

% calculate time points of interest, using buffer
ti = RUN.trial{trial}.movements(1,1,movement);
tf = RUN.trial{trial}.movements(1,2,movement);

if ti-buffer > 0
    ti = ti-buffer;
end
if tf+buffer <= size(RUN.trial{trial}.point,1)
    tf = tf + buffer;
end

% extract time points of interest
points = RUN.trial{trial}.point(ti:tf,:);



%dummy variable for plots
points_u = RUN.trial{trial}.point;

if display
    figure
    plot(points_u(:,1),points_u(:,4),'b')
    hold on
    plot(points_u(:,1),points_u(:,5),'r')
    plot(points(:,1),points(:,4),'b','linewidth',2)
    plot(points(:,1),points(:,5),'r','linewidth',2) 
    axis tight
    set(gcf,'color','w')
    ylabel('Movement','fontname','times','fontsize',14)
    xlabel('t (msec)','fontname','times','fontsize',14)
    title('Extracted Movement Segment','fontname','times','fontsize',14)
end

%correct time points for offset such that the reach begins at t=0
points(:,1) = points(:,1) - points(1,1);

if display
    figure
    plot(points(:,1),points(:,4),'b')
    hold on
    plot(points(:,1),points(:,5),'r')
    axis tight
    set(gcf,'color','w')
    ylabel('Movement','fontname','times','fontsize',14)
    xlabel('t (msec)','fontname','times','fontsize',14)
    title('Reach','fontname','times','fontsize',14)
end

end
function [ movements number ] = extractTrialType( RUN, type, dir, rot )

% function [ movements ] = extractTrialType( RUN, type, dir )
% Brier, MR
%
% RUN - RUN structure with the trials to be extracted
% type - corresponds to the trial type number, 0 is wildcard
% dir - the direction of the movement
%
% This function reads the RUN structure and extracts the first movements of
% a specified type

movements = {};
number = [];
c = 0;

for x = 1:length(RUN.trial)
    
    if type ~= 0
        if RUN.trial{x}.stack == type && RUN.trial{x}.dir == dir && RUN.trial{x}.rot == rot && ~RUN.trial{x}.bad
            c = c + 1;
            movements{c} = RUN.trial{x}.first_movement;
            number(c) = x;
        end
    else
        c = c + 1;
        movements{c} =  RUN.trial{x}.first_movement;
        number(c) = x;
    end
end

end
function [ single_fit, single_buffer, p ] = fitSingleTrialType( single , type, order, baselinecorrect, n_t )

% function [ single_fit, single_buffer ] = fitSingleTrialType( single ,
% type, order )
% Brier, MR.
%
% single - nX5 structure of a single movment
% type - 1 - fit nth order poly
%      - 2 - robust fit (doesnt work)
% order - order of poly to fit
% single_fit - the estimated movement vectors
% single_buffer - the stretched version of single
% 
% this function reads in a single trial and depending on params fits it to
% a polynomial.  The principle reason for doing this is to correct for the
% differences in sampling frequency that exists between the trials given
% how the hardware collects data.  This function also stretches the data
% such that each trial takes the same amount of time. (This will be a
% switch in future implementations).  It then outputs the poly fit and the
% stretched raw.


single_fit = {};

% find largest t
tmax = 0;
for x = 1:length(single)
    if single{x}(end,1) > tmax
        tmax = single{x}(end,1);
    end
end

% %maybe adding long flat periods will be a way to fix this overshoot?
% for x = 1:length(single)
%     if single{x}(end,1) ~= tmax
%         a = [tmax single{x}(end,2) single{x}(end,3) single{x}(end,4) single{x}(end,5)];
%         single{x} = [single{x};a];
%     end
% end

% normalize the time series so that they are all the same length by
% stretching

for x = 1:length(single)
    single{x}(:,1) = (single{x}(:,1)./single{x}(end,1)).*tmax;
end

% baseline correct
if baselinecorrect
for x = 1:length(single)
    single{x}(:,4) = single{x}(:,4) - single{x}(1,4);
    single{x}(:,5) = single{x}(:,5) - single{x}(1,5);
end
end


if type == 1
    %fit to a nth order poly
    for x = 1:length(single)
        [p s] = polyfit(single{x}(:,1),single{x}(:,4),order);
        tf = 1:tmax/n_t:tmax;
        xf = polyval(p,tf);
        [p s] = polyfit(single{x}(:,1),single{x}(:,5),order);
        yf = polyval(p,tf);
        single_fit{x} = [tf' xf' yf'];
    end
elseif type == 2 % stupid idea
    for x = 1:length(single)
        [p] = robustfit(single{x}(:,1),single{x}(:,4));
        tf = 1:tmax;
        xf = polyval(p,tf);
        [p] = robustfit(single{x}(:,1),single{x}(:,5));
        yf = polyval(p,tf);
        single_fit{x} = [tf' xf' yf'];
    end    
end

single_buffer = single;

end
function [out] = writeFile(filename,RUN,single_fit)


filename = [filename '.mrb'];

%Write file header (LHS 8-9-2011)
dlmwrite(filename,RUN.header,'newline','unix','-append','roffset',1,'delimiter','\t');

for x = 1:length(RUN.trial)
    dlmwrite(filename,RUN.trial{x}.header,'newline','unix','-append','roffset',1,'delimiter','\t');
    %fprintf(fout,'\n');
    
    outmat(:,1) = single_fit{x}(:,1);
    outmat(:,2:3) = -99;
    outmat(:,4:5) = single_fit{x}(:,2:3);
    dlmwrite(filename,outmat,'newline','unix','-append','roffset',1,'delimiter','\t');
    %fprintf(fout,'\n');
end


out = 1;

end
